#include <cstdlib>
#include <iostream>
#include <fstream>
#include <vector>
#include <sstream>
#include <ctime>

#include "SimpleXML.h"

using namespace std;

/// Header of .raw files
static const char rawHeader[5] = "S2D ";

struct TSample{
    TSample( const char *_data): data( _data), annotation(""){}
    
    const char *data;
    string annotation;
};

ifstream * getRAWInfo(string & filename, unsigned & width, unsigned & height, unsigned & sampleCount);

int writeToRaw( const string filename, const vector< TSample> &data, const int width, const int height);
int writeToTxt( const string filename, const vector< TSample> &data);

string i2String( unsigned number, unsigned size);



int main(int argc, char *argv[])
{
    srand( time( NULL));
    
    string fileName;
    string outputFileName;
    int numberOfSamples = -1;
    bool normalization = true;
    string sampleClass;
    
    bool processAnnotation = false;
    bool createDatasetXML = false;
    
    int newFileCount = 1;
    
    float ratio = 0.5;
    
    for (int i = 1; i < argc; i++) {
        if (strcmp( argv[i],"-i") == 0){
            fileName = argv[ ++i];
        } else if (strcmp( argv[i],"-ratio") == 0){
            ratio = atof( argv[ ++i]);
        } else if (strcmp( argv[i],"-t") == 0){
            newFileCount = atoi( argv[ ++i]);
        } else if (strcmp( argv[i],"-a") == 0){
            processAnnotation = true;
        } else if (strcmp( argv[i],"-x") == 0){
            createDatasetXML = true;
        } else if (strcmp( argv[i],"-h") == 0){
            cout << " mixer.exe -i inputFileName [-ratio 0.5] [-t count] [-a] [-x]" << endl;
            cout << " ================================================" << endl;
            cout << endl;
            cout << " -i    Specifies the files to process (without extension)." << endl;
            cout << "       The program will add extensions as needed (.raw, .txt). " << endl;
            cout << " -a    Will process annotation." << endl;
            cout << " -x    Will create dataset XML file." << endl;
            cout << " -t    Specifies how many files to create." << endl;
            cout << " -ratio    Specifies the size ratio between training and testing sets." << endl;
            cout << endl;

            exit(1);
        } 
    }
    
    if( ratio <= 0.0 || ratio >= 1.0){
        cout << "Ratio must be from <0,1>. " << endl;
        cout << "For more information use -h" << endl;
        exit( 1);
    }
    
    if( fileName == ""){
        cout << "Input file name missing." << endl;
        cout << "Use: merge.exe -i rawInputFile [-ratio 0.5] [-a] [-x]" << endl;
        cout << "Use -h to get more help." << endl;
        exit(1);
    }
    
    if( newFileCount <= 0){
        cout << "Wrong output file count: " << newFileCount << endl;
        cout << "Use -h to get more help." << endl;
        exit(1);
    }
    
    unsigned width = 0;
    unsigned height = 0;
    unsigned totalSampleCount = 0;
    
    string inputRawFileName = fileName + ".raw";

    ifstream * rawFile = getRAWInfo( inputRawFileName, width, height, totalSampleCount);

    if ( !rawFile){
        cout << "Error: unable to open file " << inputRawFileName << "." << endl;
        exit( 1);
    }

    cout << " widht: " << width << " height: " << height << " sampleCount: " << totalSampleCount << endl;

    vector< TSample> originalImages;

    for( int i = 0; i < totalSampleCount; i++){
        char * image = new char[ width*height];
        rawFile->read( image, sizeof( char) * width*height);
        
        if( !rawFile->good()){
            cerr << "Error: Unable to read from file " << inputRawFileName << '.' << endl;
            exit( 1);
        }
        
        originalImages.push_back( TSample( image));
    }
    rawFile->close();
    delete rawFile;

    if( originalImages.empty()){
        cout << "Error: No data in raw file " << inputRawFileName << endl;
        exit( 1);
    }
    
    
    int numberOfClasses = 0;

    ifstream * annotFile = NULL;
    if( processAnnotation){

        string inputAnnotFileName = fileName + ".txt";
        annotFile = new ifstream( inputAnnotFileName.data());

        for( int i = 0; i < totalSampleCount; i++){
            string line;
            
            while( annotFile->good() && (line.empty() || line[0] == '#')) {
                getline( *annotFile, line);
            }

            if( !annotFile->good()){
                cerr << "Error: Unable to read from file " << inputAnnotFileName << '.' << endl;
                exit( 1);
            }
            originalImages[ i].annotation = line;
        }
        rawFile->close();
        delete rawFile;
        
        stringstream ss;
        
        
        ss << originalImages.begin()->annotation;
        
        while( ss.good()){
            int i;
            ss >> i;
            numberOfClasses++;
        }
        numberOfClasses--;
        
    }
    

    
    for( int i = 0; i < newFileCount; i++){
        vector< TSample> imagesFrom( originalImages);

        vector< TSample> imagesOne;
        vector< TSample> imagesTwo;
        
        int sizeTwo = int( ratio * totalSampleCount + 0.5);

        // Nahodny vyber prvni casti samplu
        for( int j = 0; j < sizeTwo; j++){
            int selected = rand() % imagesFrom.size();
            
            imagesOne.push_back( imagesFrom[ selected]);
            imagesFrom[selected] = imagesFrom.back();
            imagesFrom.pop_back();
            
        }
        
        // Nahodny vyber druhe casti samplu -> V podstate jenom promichani samplu
        while( !imagesFrom.empty()){
            int selected = rand() % imagesFrom.size();

            imagesTwo.push_back( imagesFrom[ selected]);
            imagesFrom[selected] = imagesFrom.back();
            imagesFrom.pop_back();
        }


        // Zapis RAW souboru
        string oneName = fileName + i2String( i, 2) + "-tr.raw";
        string twoName = fileName + i2String( i, 2) + "-ts.raw";
        writeToRaw( oneName, imagesOne, width, height);
        writeToRaw( twoName, imagesTwo, width, height);
        
        
        // Zapis souboru z anotacemi
        if( processAnnotation){
            string annotOneName = fileName + i2String( i, 2) + "-gr.txt";
            string annotTwoName = fileName + i2String( i, 2) + "-gs.txt";
            
            writeToTxt( annotOneName, imagesOne);
            writeToTxt( annotTwoName, imagesTwo);
        }
        
        // Zapis dataset XML
        if( createDatasetXML && processAnnotation){
            string xmlFilename = fileName + i2String( i, 2) + ".xml";
            ofstream outputFile( xmlFilename.data(), ios::trunc);
            TXMLOutput xml;
            
            xml.addElement( "dataset");
            xml.addAttribute( "name", fileName +  i2String( i, 2));
            xml.addAttribute( "directory", "../data/");
            
            xml.addElement( "params");
            
            xml.addElement( "source");
            xml.addText( " FIT VUT - Mixed training and testing sets from file " + fileName );
            xml.endElement();
            
            xml.addElement( "sampleInfo");
            xml.addAttribute( "sizeX", (int)(width));
            xml.addAttribute( "sizeY", (int)(height));
            xml.endElement();
            
            xml.endElement();
            
            xml.addElement( "subsets");
            
            // training sets
            for( int classId = 0; classId < numberOfClasses; classId++){

                xml.addElement( "subset");
                xml.addAttribute( "name", "trainingClass" + i2String( classId, 2));
                
                xml.addElement( "raw");
                xml.addAttribute( "name", fileName + i2String( i, 2) + "-tr.raw");
                xml.addAttribute( "annotation", fileName + i2String( i, 2) + "-gr.txt");
                xml.addAttribute( "set", classId);
                xml.endElement();
                
                xml.endElement(); //subset
            }
            
            
            
            // testing sets
            for( int classId = 0; classId < numberOfClasses; classId++){

                xml.addElement( "subset");
                xml.addAttribute( "name", "testingClass" + i2String( classId, 2));

                xml.addElement( "raw");
                xml.addAttribute( "name", fileName + i2String( i, 2) + "-ts.raw");
                xml.addAttribute( "annotation", fileName + i2String( i, 2) + "-gs.txt");
                xml.addAttribute( "set", classId);
                xml.endElement();

                xml.endElement(); //subset
            }

            
            
            xml.endElement(); // subsets
            
            xml.endElement(); // dataset
            
            outputFile << xml.getText( 0);
            
            
            outputFile.close();
        }

    }
    
    if( createDatasetXML && processAnnotation){
        ofstream confFile( (fileName + "ABConfiguration.txt").data(), ios::trunc);
        confFile << "    ==ALTERNATIVE 666" << endl;
        for( int i = 0; i < newFileCount; i++){
            confFile << "      ==OPTION -" << i2String( i, 2) << endl;
            confFile << "        ==ATTR path ../data/" << fileName << i2String( i, 2) << ".xml" << endl;
            confFile << "      ==ENDOPTION" << endl;
        }
        confFile << "    ==ENDALTERNATIVE" << endl;
        
    }
    return EXIT_SUCCESS;
}

int writeToTxt( const string filename, const vector< TSample> &samples){

    ofstream *outputFile = new ofstream( filename.data(), ios::trunc);

    if( outputFile == NULL ){
        cout << "ERROR: Unable to open output file " << filename << "." << endl;
        return 0;
    }
    
    if( !outputFile->good() ){
        cout << "ERROR: Unable to open output file " << filename << "." << endl;
        delete outputFile;
        return 0;
    }
    
    vector< TSample>::const_iterator sample;
    for( sample = samples.begin(); sample != samples.end(); sample++){
        *outputFile << sample->annotation << endl;
    }
    
    outputFile->close();
    delete outputFile;
    return 1;
    
}

int writeToRaw( const string filename, const vector< TSample> &data, const int width, const int height){

    ofstream *outputFile = new ofstream( filename.data(), ios::binary | ios::trunc);

    if( outputFile == NULL ){
        cout << "ERROR: Unable to open output file " << filename << "." << endl;
        return 0;
    }
    
    if( !outputFile->good() ){
        cout << "ERROR: Unable to open output file " << filename << "." << endl;
        delete outputFile;
        return 0;
    }

    outputFile->write( "S2D ", 4);
    int count = data.size();
    outputFile->write( (char *) &count, 4);
    outputFile->write( (char *) &width , 4);
    outputFile->write( (char *) &height, 4);

    vector< TSample>::const_iterator image;
    for( image = data.begin(); image != data.end(); image++){

        outputFile->write( image->data, width*height);
        
        if( !outputFile->good()){
            cout << "ERROR while writing into file " << filename << '.' << endl;
            return 0;
        }
    }
    
    outputFile->close();
    delete outputFile;
    return 1;
}


ifstream * getRAWInfo(string & filename, unsigned & width, unsigned & height, unsigned & sampleCount)
{
    ifstream * p_fr = new ifstream(filename.c_str(), ios::binary);
    
    if( !p_fr->good())
    {
        cerr << "Warning: Unable to open raw file " << filename << '.' << endl;
        p_fr->close();
        delete p_fr;
        return NULL;
    }
    
    char header[5];
    p_fr->read((char*)header, 4);

    if ( strncmp(header, rawHeader, 4) != 0)
    {
        cerr << "Warning: Wrong file or sample format." << endl;
        p_fr->close();
        delete p_fr;
        return NULL;
    }

    p_fr->read((char*)(&sampleCount), 4);
    p_fr->read((char*)(&width), 4);
    p_fr->read((char*)(&height), 4);

    if( !p_fr->good()){
        p_fr->close();
        delete p_fr;
        return NULL;
    }

    return p_fr;
} // getRawInfo()



string i2String( unsigned number, unsigned size){
    string s;
    unsigned base = 10;
    

    for( int i = 0; i < size; i++){
        char digit = number % base / ( base / 10);
        s = char('0' + digit) + s;
        number -= number % base;
        base *= 10;
    }
    return s;
}

