#include "main.h"
#include "config.h"
#include "matrix.h"
#include "image_transform.h"
#include "illumination.h"
#include "random.h"
#include "dataset.h"
#include "xml.h"
#include "Tools.h"
#include "Convolve.h"


#include <map>
#include <sstream>
#include <fstream>

#include <string>

bool loadImage( const std::string &path, int &sizeX, int &sizeY, unsigned char *&data);
void FreeData(void * Block){
    delete[] (char *) Block;  
}

double NormGen( double mean, double stddev){
    double accumulator = 0;

    for( int i = 0; i < 12; i++){ //it has to be 12; if not there has to be * sqrt( k / 12) at the end of return line
        accumulator += rand() / double( RAND_MAX);
    }

    return mean + ( accumulator - 6.0) * stddev ;
}

int main(int argc, char *argv[])
{
    const int oversizeFactor = 3;
    float *floatImg1, *floatImg2;
    ImageStruct *inputImage, *transformedImage, *outputImage, *largeOutput;
    Config config;
    ClassInfo class_info;
    Image8Value *illuminationArray;


        
    string annotationFileName;
    string configFileName;
    string rawFileName;
    map< string, int> subClasses;
    
    string checkFileName;
    
    for( int i = 1; i < argc; i++){
        if( string( argv[ i]) == "-c"){
            configFileName = argv[ ++i];   
        } else if( string( argv[ i]) == "-i"){
            annotationFileName = argv[ ++i];
        } else if( string( argv[ i]) == "-h"){
            cout << "Use: image_transform.exe -i anotation -c configuration [-raw file.raw] [-check file.txt] class1 [class2 ...] " << endl;
            cout << endl;
            cout << " Options: " << endl;
            cout << " -i anotation      Specify the annotation file." << endl;
            cout << " -c configuration  Specify the configutation file." << endl;
            cout << " -raw file.raw     Specify the output raw file name." << endl;
            cout << "                   If not specified output in separate *.tif files." << endl;
            cout << " -check file.txt   Specify the file where a copy of annotation will be stored. " << endl;
            cout << "                   Can be used to check if the annotation is loaded properly." << endl;
            cout << " class1 ...        All the object classes that should be extracted." << endl;
            exit( 1);
        } else if( string( argv[ i]) == "-raw"){
            rawFileName = argv[ ++i];
        } else if( string( argv[ i]) == "-check") {
            checkFileName = argv[ ++i];
        } else {
            subClasses[ argv[ i]] = 1;
        }
    }
    
    if( annotationFileName == ""){
        cout << "Annotation file name was not specified. Use -h to get more help." << endl;
        exit( 1);   
    }

    if( configFileName == ""){
        cout << "Configuration file name was not specified. Use -h to get more help." << endl;
        exit( 1);   
    }
        
    // Loading configuration ---------------------------------------------------
    cout << "Loading configuration from " << configFileName << endl;
    config = loadConfiguration( configFileName);
    if( !config.configOK){
        cerr << "ERROR: Unable to load configuration file " << configFileName << endl;   
        return 0;
    }
    
    cout << "------------------------------------------------------------------------------" << endl;
    cout << "transformation_nb: " << config.transformation_nb  << endl; 
    cout << "output_resolution: " << config.output_resolution.x << ' ' << config.output_resolution.y << endl; 
    cout << "------------------------------------------------------------------------------" << endl; 
    
    // image allocations
    transformedImage = NewImage8( config.output_resolution.x, config.output_resolution.y);
    outputImage = NewImage8( config.output_resolution.x, config.output_resolution.y);
    largeOutput = NewImage8( config.output_resolution.x * oversizeFactor - oversizeFactor + 1, config.output_resolution.y * oversizeFactor- oversizeFactor + 1);
    floatImg1 = new float[  config.output_resolution.x * oversizeFactor * config.output_resolution.y * oversizeFactor];
    floatImg2 = new float[  config.output_resolution.x * oversizeFactor * config.output_resolution.y * oversizeFactor];

    // get annotations and class informations     
    cout << " Read the annotations from: " << annotationFileName << endl;
    vector< TAnnotation> annotations;
    loadAnnotation( annotationFileName, annotations); 

    // Image transformations ---------------------------------------------------    

    srand(time(NULL));
    vector< TAnnotation>::iterator annotation;

    unsigned long imageCount = 0;
    long imageCountPosition;
    ofstream *rawFile = NULL;

    if( rawFileName != ""){
        rawFile =  new ofstream( rawFileName.data(), ios::binary | ios::trunc);
        if( !rawFile->good()){
            cerr << "ERROR: Unable to open file " << rawFileName << " for writing." << endl;
            exit( 1);
        }
        rawFile->write( "S2D ", 4);
        imageCountPosition = rawFile->tellp();    
        rawFile->write( (char *) &imageCount, 4);
        rawFile->write( (char *) &config.output_resolution.x , 4);
        rawFile->write( (char *) &config.output_resolution.y, 4);
    }


    if( checkFileName != ""){
        
        ofstream out( checkFileName.data(), ios::trunc);
        
        vector< TAnnotation>::iterator anot;
        for( anot = annotations.begin(); anot != annotations.end(); anot++){
            out << "<image>  " << anot->imageName << endl;
            
            vector< TObject>::iterator obj;
            for( obj = anot->objects.begin(); obj != anot->objects.end(); obj++){
                out << "   <object>" << endl;
                out << "       sub_class        = " << obj->sub_class << endl;
                out << "       roi_LeftTop      = " << obj->roi_LeftTop.x << ':' << obj->roi_LeftTop.y << endl;
                out << "       roi_LeftBottom   = " << obj->roi_LeftBottom.x << ':' << obj->roi_LeftBottom.y << endl;
                out << "       roi_RightTop     = " << obj->roi_RightTop.x << ':' << obj->roi_RightTop.y << endl;
                out << "       roi_RightBottom  = " << obj->roi_RightBottom.x << ':' << obj->roi_RightBottom.y << endl;
                out << "   </object>" << endl;
            }
            out << "</image>" << endl;
        }
        out.close();
    }


    cout << "Image generation  ...  " ;
    
    int outOfBoundCount = 0;

    for( annotation = annotations.begin(); annotation != annotations.end() ; annotation++)
    {   
        // load image
        
        bool relevant = false;
        vector< TObject>::iterator object;
        for( object = annotation->objects.begin(); object != annotation->objects.end(); object++)
        {
            if( ( subClasses[ object->sub_class] > 0)){
                relevant = true;   
            }
        }
        if( !relevant){
            continue;   
        }
        
        
        {
            int sizeX;
            int sizeY;
            unsigned char *data;
            if( loadImage( annotation->imageName, sizeX, sizeY, data)){
                cout << "Size: " << sizeX << ' ' << sizeY << endl;
                inputImage = NewImageReferenceExternal( sizeX, sizeY, 1, sizeX, Image8Linear, (void*)data, NULL, 0, (void*) data, FreeData);
 
            } else {
                inputImage = CreateImageFile( (char *) annotation->imageName.data(), 0, NULL, NULL);
            }
            if( inputImage == NULL){
                cerr << "ERROR: Unable to load image " << annotation->imageName << endl;
                continue;
            }
        }
                
        ImageStruct * grayImage;
        if( inputImage->PixelType == ImageRGB || inputImage->PixelType == ImageRGBLinear){
            grayImage = NewImage8( inputImage->XSize, inputImage->YSize);
            RGBToIntensity( inputImage, grayImage);
            DeleteImage( inputImage);
        } else if( inputImage->PixelType == Image8 || inputImage->PixelType == Image8Linear){
            grayImage = inputImage;
        } else {
            cout << "Wrong image type: " << annotation->imageName << endl;
            DeleteImage( inputImage);
            continue;
        }


        Matrix R = newTransfMat();     
        
        for( object = annotation->objects.begin(); object != annotation->objects.end(); object++)
        {   

            if( ! (subClasses[ object->sub_class] > 0)){
                continue;   
            }

           
            for( int transf_index = 0 ; transf_index < config.transformation_nb; transf_index++)
            {   
                identityMatrix(R);

                 //building the final transformation matrix      
                rotateImage( *object, R, simpleRandom( config.rotation.min,  config.rotation.max));
                
                shearImage( *object, R, 
                    simpleRandom( config.shearX.min,  config.shearX.max), 
                    simpleRandom( config.shearY.min,  config.shearY.max));
                    
                shiftImage( *object, R, 
                    simpleRandom( config.shiftX.min,  config.shiftX.max) * fabs( object->roi_RightTop.x - object->roi_LeftTop.x), 
                    simpleRandom( config.shiftY.min,  config.shiftY.max) * fabs( object->roi_RightTop.y - object->roi_RightBottom.y));
                    
                float scale = simpleRandom( config.scale.min,  config.scale.max);
                scaleImage( *object, R, scale, scale);
                
                // transforming the image
                TObject newCoordinates;
                calculateOB( (*object), newCoordinates, R);
 
                transformImage( grayImage, largeOutput, newCoordinates);

                for( int y = 0; y < outputImage->YSize; y++){
                    for( int x = 0; x < outputImage->XSize; x++){
                        Image8Pixel( outputImage, x, y) = Image8Pixel( largeOutput, x * oversizeFactor, y * oversizeFactor);
                    }
                }

                // prevod na float pole                
                for( int y = 0; y < largeOutput->YSize; y++){
                    for( int x = 0; x < largeOutput->XSize; x++){
                        floatImg1[ x + y * largeOutput->XSize] = Image8Pixel( largeOutput, x, y);
                    }
                }
                
                ConvolutionKernel gauss;
                computeKernels( oversizeFactor * 0.3, gauss);
                convolveImageHoriz( floatImg1, floatImg2, largeOutput->XSize, largeOutput->YSize, gauss);
                convolveImageVert( floatImg2, floatImg1, largeOutput->XSize, largeOutput->YSize, gauss);
                
                // resampling a prevod na 8-bit image
                for( int y = 0; y < outputImage->YSize; y++){
                    for( int x = 0; x < outputImage->XSize; x++){
                        Image8Pixel( outputImage, x, y) = 
                            max( 0, min( 255, int(  
                                NormGen( 0, config.gaussianNoiseSTDDev) 
                              + floatImg1[ x * oversizeFactor + y * oversizeFactor * largeOutput->XSize] 
                              + 0.5
                            )));
                    }
                }


                int x1 = int( newCoordinates.roi_LeftTop.x + 0.5);
                int x2 = int( newCoordinates.roi_RightBottom.x + 0.5);
                int y1 = int( newCoordinates.roi_LeftTop.y + 0.5);
                int y2 = int( newCoordinates.roi_RightBottom.y + 0.5);
               
                if( x1 < 0 || x2 < 0 || y1 < 0 || y2 < 0 
                    || x1 >= grayImage->XSize || x2 >= grayImage->XSize 
                    || y1 >= grayImage->YSize || y2 >= grayImage->YSize){
                    // cerr << "out of image: " << annotation->imageName.data()  << ' ' << x1 << ' ' << y1 << ' ' << x2 << ' ' << y2 << ' ' <<endl;     
                    outOfBoundCount++;
                    continue;
                }
                
               /*   string s = "";
                    stringstream ss;
                    ss << subClasses[ object->sub_class]++;
                    ss >> s;
                    
                    string outputName = object->sub_class + "_";
                    if( s.length() < 8){ 
                        for( int i = 0; i < 8; i++){
                            if( 8 - i - (signed)(s.length()) > 0){
                                outputName.push_back( '0');
                            } else {
                                outputName.push_back( s[ 0 - ( 8 - i - (signed)(s.length()))] );
                            }
                        }
                    } else {
                        outputName += s;
                    }
                    outputName += ".tif";
               
                for( int y = y1; y < y2; y++){
                    Image8Pixel( grayImage, x1, y) = 255; 
                    Image8Pixel( grayImage, x2, y) = 255; 
                }
                for( int x = x1; x < x2; x++){
                    Image8Pixel( grayImage, x, y1) = 255; 
                    Image8Pixel( grayImage, x, y2) = 255; 
                }
                
                SaveImageFile( (char * ) outputName.data(), 0, grayImage, NULL, NULL);*/

                subClasses[ object->sub_class]++;
                
                imageCount++;
                if( rawFile != NULL){
                    for( int y = 0; y < outputImage->YSize; y++){
                        char * line = (char *)&Image8Pixel( outputImage, 0, y);
                        rawFile->write( line, outputImage->XSize);    
                    }
                } else {
                    string s = "";
                    stringstream ss;
                    ss << subClasses[ object->sub_class];
                    ss >> s;
                    
                    string outputName = object->sub_class + "_";
                    if( s.length() < 8){ 
                        for( int i = 0; i < 8; i++){
                            if( 8 - i - (signed)(s.length()) > 0){
                                outputName.push_back( '0');
                            } else {
                                outputName.push_back( s[ 0 - ( 8 - i - (signed)(s.length()))] );
                            }
                        }
                    } else {
                        outputName += s;
                    }
                    outputName += ".tif";
                    SaveImageFile( (char * )outputName.data(), 0, outputImage, NULL, NULL);
                }
                static int counter = 0;
                cout << char( 8);
                switch (counter){
                    case 0:
                        cout << '|';
                        break;
                    case 1:
                        cout << '/';
                        break;
                    case 2:
                        cout << '-';
                        break;
                    case 3:
                        cout << '\\';
                        break;
                }
                counter = (counter + 1) % 4;


                
            }
        }
        DeleteImage( grayImage);
        freeMatrix(R);  

    }
    cout << char( 8);
    cout << " DONE" << endl;

    if( outOfBoundCount > 0){
        cout << endl;
        cout << "Warning: " << outOfBoundCount << " subimages were out of exceded image area and were not generated." << endl;
        cout << endl;
    }

    cout << "Generated " << imageCount << " images." << endl;
    
    map< string, int>::iterator it;
    for( it = subClasses.begin(); it != subClasses.end(); it++){
        cout << "Class " << it->first << ": " << max( it->second - 1, 0) << " samples" << endl;   
    }
    
    if( rawFile != NULL){
        rawFile->seekp( imageCountPosition);
        rawFile->write( (char *) &imageCount, 4);
        rawFile->close();
        delete rawFile;
    }    

 
    // Freeing memory ----------------------------------------------------------
    DeleteImage(transformedImage);
    DeleteImage(outputImage);
    
    
    cout << "ALL DONE" << endl;    

    // end of program ----------------------------------------------------------
    return EXIT_SUCCESS;
}



bool loadImage( const std::string &path, int &sizeX, int &sizeY, unsigned char *&data){
    // only PGM image file format is supported at the moment so there is not much to do
    {
        ifstream inputFile( path.c_str(), ios::binary);
        if( inputFile.bad())
        {
            cerr << "Error: Unable to open file \"" << path << "\"." << endl;
            return false;
        }
        char buff[256];
        unsigned levels;
    
        inputFile.getline( buff, 256);
        if( strncmp( buff, "P5", 2) != 0)
        {
            cerr << "Error: Wrong image file format \"" << path << "\"." << endl;
            return false;
        }

        do // skip comments
        {
            inputFile.getline(buff, 256);
        } while (buff[0] == '#');

        if( sscanf( buff, "%d %d", &sizeX, &sizeY) != 2){
            cerr << "Error: Corupted image file \"" << path << "\"." << endl;
            return false;            
        }
        
        inputFile.getline(buff, 256);
        sscanf(buff, "%d", &levels);

        data = new unsigned char[ sizeX * sizeY];
        
        if( data == NULL)
        {
            cerr << "Error: Unable to allocate memory in loadImage()." << endl;
            return false;
        }

        inputFile.read( (char *) data, sizeX * sizeY * sizeof( unsigned char));
        
        if( !inputFile.good()){
            cerr << "Error: Corupted image file \"" << path << "\"." << endl;
            delete[] data;
            return false;
        }
        
        return true;
    }    
}
