#include "main.h"
#include "config.h"
#include "matrix.h"
#include "image_transform.h"
#include "ImageInt.h"
#include "random.h"

// rotateImage -----------------------------------------------------------------
//
// description : function that builds a transformation matrix to rotate an image
//               around a center. Therefore, it combines 3 transformation matrix :
//               a translation to the center, a rotation and the inverse translation.
//               
// arguments :   Annotation annotation           -      the image annotations
//               int objectID                    -      objectID of the image
//               Matrix &R                       -      the transformation matrix
//               float theta                     -      the rotation angle
// -----------------------------------------------------------------------------

void rotateImage( const TObject &object, Matrix &R, const float theta)
{
    Matrix TM, TM_inv;               // translation matrix 
    Matrix RM;                       // rotation matrix
    
    // conversion from degrees to radians
    float thetaRad =  (theta / 180.f * PI);
    
    // initialisations of the transformation matrix
    RM = newRotationMatrix( thetaRad);
    TM = newTranslationMatrix( ImageXCenter( object), ImageYCenter( object), 1);
    TM_inv = newTranslationMatrix( -ImageXCenter( object), -ImageYCenter( object), 1);  
    
    // building the final transformation matrix
    updateTransfMat( R, TM);
    updateTransfMat( R, RM);
    updateTransfMat( R, TM_inv);
    
    // freeing memory
    freeMatrix(RM);
    freeMatrix(TM);
    freeMatrix(TM_inv);
}

// shearImage ------------------------------------------------------------------
//
// description : function that builds a transformation matrix to shear an image
//               around a center. Therefore, it combines 3 transformation matrix :
//               a translation to the center, a shear and the inverse translation.
//               
// arguments :   Annotation annotation           -      the image annotations
//               int objectID                    -      objectID of the image
//               Matrix &R                       -     the transformation matrix
//               float sx                        -     the x axis shear value
//               float sy                        -     the y axis shear value
// -----------------------------------------------------------------------------

void shearImage( const TObject &object, Matrix &R, float sx, float sy)
{
    Matrix TM, TM_inv;               // translation matrix 
    Matrix ShM;                      // shear matrix
    
    // initialisations of the transformation matrix
    ShM = newShearMatrix( sx, sy); 
    TM = newTranslationMatrix( ImageXCenter( object), ImageYCenter( object), 1);
    TM_inv = newTranslationMatrix( -ImageXCenter( object), -ImageYCenter( object), 1);  
    
    // building the final transformation matrix
    updateTransfMat(R, TM);
    updateTransfMat(R, ShM);
    updateTransfMat(R, TM_inv);
    
    // freeing memory
    freeMatrix(ShM);
    freeMatrix(TM);
    freeMatrix(TM_inv);
}

// shearImage ------------------------------------------------------------------
//
// description : function that build a shift translation matrix. It calls a
//               simple translation matrix. 
//               
// arguments :   Annotation annotation           -      the image annotations
//               int objectID                    -      objectID of the image
//               Matrix &R                       -     the transformation matrix
//               float dx                        -     the x axis shift value
//               float dy                        -     the y axis shift value
// -----------------------------------------------------------------------------

void shiftImage( const TObject &object, Matrix &R, float dx, float dy)
{
    Matrix TM;                       // translation matrix    
    
    // initialisations of the transformation matrix
    TM = newTranslationMatrix( -dx, -dy, 1); 
    
    // building the final transformation matrix
    updateTransfMat(R, TM);

    // freeing memory
    freeMatrix(TM);
}

// scaleImage ------------------------------------------------------------------
//
// description : function that builds a transformation matrix to scale an image
//               around a center. Therefore, it combines 3 transformation matrix :
//               a translation to the center, a scale and the inverse translation.
//               
// arguments :   Annotation annotation           -      the image annotations
//               int objectID                    -      objectID of the image
//               Matrix &R                       -     the transformation matrix
//               float sx                        -     the x axis scale value
//               float sy                        -     the y axis scale value
// -----------------------------------------------------------------------------

void scaleImage( const TObject &object, Matrix &R, float sx, float sy)
{
    Matrix TM, TM_inv;               // translation matrix 
    Matrix ScM;                      // scale matrix 

    // initialisations of the transformation matrix
    ScM = newScaleMatrix(1/sx,1/sy);
    TM = newTranslationMatrix( ImageXCenter(object), ImageYCenter( object), 1);
    TM_inv = newTranslationMatrix( -ImageXCenter( object), -ImageYCenter( object), 1);  
    
    // building the final transformation matrix
    updateTransfMat( R, TM);
    updateTransfMat( R, ScM);
    updateTransfMat( R, TM_inv);

    // freeing memory
    freeMatrix( ScM);
    freeMatrix( TM);
    freeMatrix( TM_inv);
}

// transformImage --------------------------------------------------------------
//
// description : it is only a function calling the Interpolate image function. Its
//               only purpose is to convert the point structures used by the program
//               in point structures usable by this particular function.
//               
// arguments :   const ImageStruct * inputImage      -     the input image
//               ImageStruct * transformedImage      -     the output image
//               Annotation annotation           -      the image annotations
// -----------------------------------------------------------------------------

void transformImage( const ImageStruct * inputImage, ImageStruct * transformedImage, const TObject & object)
{
    PointStruct LeftTop;
    PointStruct RightTop;
    PointStruct LeftBottom;
    PointStruct RightBottom;
    
    // storing the corners coordinates in the relevant data structure for the InterpolateImage function
    LeftTop.X      = object.roi_LeftTop.x;
    LeftTop.Y      = object.roi_LeftTop.y;
    RightTop.X     = object.roi_RightTop.x;
    RightTop.Y     = object.roi_RightTop.y;
    LeftBottom.X   = object.roi_LeftBottom.x;
    LeftBottom.Y   = object.roi_LeftBottom.y;
    RightBottom.X  = object.roi_RightBottom.x;
    RightBottom.Y  = object.roi_RightBottom.y;
    
    // funtion call
    InterpolateImage( transformedImage, inputImage, &LeftTop, &RightTop, &LeftBottom, &RightBottom);
}

// calculateOB --------------------------------------------------------------
//
// description : this function calculates the output bounds (OB) that is to
//               say the coordinates of the corners of the input image after
//               the transformation by the matrix R.
//               
// arguments :   Annotation annotation           -      the image annotations
//               int objectID                    -      objectID of the image
//               Matrix R                        -     the transformation matrix        
// -----------------------------------------------------------------------------

void calculateOB( const TObject &object, TObject &newCoordinates, Matrix R)
{
    Matrix old_LeftTop, old_LeftBottom, old_RightTop, old_RightBottom;          // coordinates of the original corners
    Matrix new_LeftTop, new_LeftBottom, new_RightTop, new_RightBottom;          // coordinates of the corners after transformation
    
    // initialisation    
    old_LeftTop     = newPoint( object.roi_LeftTop.x    , object.roi_LeftTop.y , 1);
    old_LeftBottom  = newPoint( object.roi_LeftBottom.x , object.roi_LeftBottom.y , 1);
    old_RightTop    = newPoint( object.roi_RightTop.x   , object.roi_RightTop.y , 1);
    old_RightBottom = newPoint( object.roi_RightBottom.x, object.roi_RightBottom.y , 1);
    
    // computes new coordinates
    new_LeftTop     = multMatrix( R, old_LeftTop);
    new_LeftBottom  = multMatrix( R, old_LeftBottom);
    new_RightTop    = multMatrix( R, old_RightTop);
    new_RightBottom = multMatrix( R, old_RightBottom);   

    // stores the results in the configuration structure
    newCoordinates.roi_LeftBottom.x  = pointXCoord( new_LeftBottom);
    newCoordinates.roi_LeftBottom.y  = pointYCoord( new_LeftBottom);
    newCoordinates.roi_LeftTop.x     = pointXCoord( new_LeftTop);
    newCoordinates.roi_LeftTop.y     = pointYCoord( new_LeftTop);
    newCoordinates.roi_RightBottom.x = pointXCoord( new_RightBottom);
    newCoordinates.roi_RightBottom.y = pointYCoord( new_RightBottom);
    newCoordinates.roi_RightTop.x    = pointXCoord( new_RightTop);
    newCoordinates.roi_RightTop.y    = pointYCoord( new_RightTop);
    
    // frees the memory
    freeMatrix(old_LeftTop);
    freeMatrix(old_RightTop);
    freeMatrix(old_LeftBottom);
    freeMatrix(old_RightBottom);
    freeMatrix(new_LeftTop);
    freeMatrix(new_RightTop);
    freeMatrix(new_LeftBottom);
    freeMatrix(new_RightBottom);  
}

// ImageXCenter --------------------------------------------------------------
//
// description : this function returns the x coordinate of the center of the
//               region of interest.
//               
// arguments :   Annotation annotation           -      the image annotations
//               int objectID                    -      objectID of the image
// -----------------------------------------------------------------------------

int ImageXCenter( const TObject & object)
{
    return (int) (object.roi_LeftTop.x + (object.roi_RightBottom.x - object.roi_LeftTop.x)/2 );
}

// ImageYCenter --------------------------------------------------------------
//
// description : this function returns the y coordinate of the center of the
//               region of interest.
//               
// arguments :   Annotation annotation           -      the image annotations
//               int objectID                    -      objectID of the image 
// -----------------------------------------------------------------------------

int ImageYCenter( const TObject & object)
{
    return (int) (object.roi_LeftTop.y + ( object.roi_RightBottom.y - object.roi_LeftTop.y)/2 );
}
