/* Standard includes */
#include <assert.h>
#include <math.h>
#include <stdlib.h>   /* malloc(), realloc() */

#include "Convolve.h"

/*********************************************************************
 * _computeKernels
 */

void computeKernels(
  float sigma,
  ConvolutionKernel &gauss)
{
    const float factor = 0.005f;   // for truncating tail 
    
    assert(MAX_KERNEL_WIDTH % 2 == 1);
    assert(sigma >= 0.0);
    assert(factor < 1.0f);
  
    {  
        const int hw = MAX_KERNEL_WIDTH / 2;
        
        // Compute gauss
        float sum = 0;
        for( int i = -hw ; i <= hw ; i++)  {
            gauss.data[ i + hw] = (float) exp( - i*i / (2*sigma*sigma));
        }
        
        // Compute width
        gauss.width = MAX_KERNEL_WIDTH;
        for( int i = -hw ; gauss.data[ i + hw] / gauss.data[ hw] < factor; 
             i++, gauss.width -= 2)
        ;
    
        // Shift the kernel
        for( int i = 0 ; i < gauss.width ; i++)
        {
            gauss.data[i] = gauss.data[ i + ( MAX_KERNEL_WIDTH - gauss.width) / 2];
        }
    }
     
    {   
        // Normalize the gauss 
        const int hw = gauss.width / 2;
        float sum = 0.0;
        for( int i = 0 ; i < gauss.width ; i++)
        {
            sum += gauss.data[i];
        }
        for( int i = 0 ; i < gauss.width ; i++)
        {
            gauss.data[i] /= sum;
        }
    }
}

void convolveImageHoriz(
    float* imgIn,
    float* imgOut,
    int sizeX, int sizeY,
    ConvolutionKernel kernel)
{

    int hw = kernel.width / 2;
    
    for( int y = 0; y < sizeY; y++){
        for( int x = 0; x < sizeX; x++){
            int from = -hw;
            int to = hw;
            
            if( x + from < 0){
                from = -x;
            }
            if( x + to > sizeX){
                to = sizeX - x;
            }
                
            float sum = 0;
            float coefSum = 0;
            for( int i = from; i < to; i++){
                sum += kernel.data[ hw + i] * imgIn[ x + i + y * sizeX];
                coefSum += kernel.data[ hw + i];
            }
            imgOut[ x + y * sizeX] = sum / coefSum;
        }
    }
}

void convolveImageVert(
    float* imgIn,
    float* imgOut,
    int sizeX, int sizeY,
    ConvolutionKernel kernel)
{

    int hw = kernel.width / 2;
    
    for( int y = 0; y < sizeY; y++){
        for( int x = 0; x < sizeX; x++){
            int from = -hw;
            int to = hw;
            
            if( y + from < 0){
                from = -y;
            }
            if( y + to > sizeY){
                to = sizeY - y;
            }
                
            float sum = 0;
            float coefSum = 0;
            for( int i = from; i < to; i++){
                sum += kernel.data[ hw + i] * imgIn[ x + (y + i) * sizeX];
                coefSum += kernel.data[ hw + i];
            }
            imgOut[ x + y * sizeX] = sum / coefSum;
        }
    }
}



