/****************************************************************************/
/*                                                                          */
/*                                                                          */
/*                 GraList.h - Graphics List (c) 1998 PEZIK                 */
/*                                                                          */
/*                                                                          */
/****************************************************************************/

#ifndef _GRALIST
#define _GRALIST

#include "ColorDef.h"

#ifdef USE_LIBENTRY
# include <LibEntry.h>
#endif

#ifndef LIBENTRY
# define LIBENTRY
#endif

#define _GRALIMACROS

#define MaxPointLabelSize 8

typedef struct PointStruct {
  ColorType Color:24;
  FontType Font:8;
  PointShape Shape;
  float X;
  float Y;
  char Label[MaxPointLabelSize];
} PointStruct;

typedef struct LineStruct {
  ColorType Color;
  float X1;
  float Y1;
  float X2;
  float Y2;
} LineStruct;

typedef struct RectangleStruct {
  ColorType Color;
  float X1;
  float Y1;
  float X2;
  float Y2;
} RectangleStruct;

typedef struct ArcStruct {
  ColorType Color;
  float X;
  float Y;
  float R;
  float B;
  float E;
} ArcStruct;


typedef int GraphicsType;
#define GraphicsNothing -1
#define GraphicsPoint 0
#define GraphicsLine 1
#define GraphicsRectangle 2
#define GraphicsArc 3

typedef struct GraphicsItemStruct {
  GraphicsType Type;
  union {
    PointStruct Point;
    LineStruct Line;
    RectangleStruct Rectangle;
    ArcStruct Arc;
  } Graphics;
  void * ExtraInfo;
} GraphicsItemStruct;

typedef struct GraphicsListStruct {
  GraphicsItemStruct * Base;
  int Count;
  int MaxCount;
  void (*Callback)(struct GraphicsListStruct * List,void * ID);
  void * ID;
  int DestroyBase;
  int Dense;
  void * ExtraInfo;
} GraphicsListStruct;

#ifdef _GRALIMACROS

#ifdef __BORLANDC__
#pragma warn -eff
#endif

/* Point macros */
#define SetPoint(Point,X,Y) SetPointColor((Point),(X),(Y),GrayColor)
#define SetPointColor(Point,X,Y,Color) SetPointType((Point),(X),(Y),(Color),CrossPoint)
#define SetPointType(Point,X,Y,Color,Shape) SetPointLabel((Point),(X),(Y),(Color),(Shape),"")
#define SetPointLabel(Point,X,Y,Color,Shape,Label) SetPointLabelFont((Point),(X),(Y),(Color),(Shape),(Label),NormalFont)
#define SetPointLabelFont(Point,iX,iY,iColor,iShape,iLabel,iFont) ((Point)->Color = (iColor), (Point)->Font = (iFont), (Point)->Shape = (iShape), (Point)->X = (iX), (Point)->Y = (iY), strncpy((Point)->Label,(iLabel),MaxPointLabelSize), Point)

#define MovePoint(Point,iX,iY) ((Point)->X += (iX), (Point)->Y += (iY), Point)
#define ScalePoint(Point,XScale,YScale) ((Point)->X *= (XScale), (Point)->Y *= (YScale), Point)

/* Line macros */
#define SetLine(Line,X1,Y1,X2,Y2) SetLineColor((Line),(X1),(Y1),(X2),(Y2),GrayColor)
#define SetLineColor(Line,iX1,iY1,iX2,iY2,iColor) ((Line)->Color = (iColor), (Line)->X1 = (iX1), (Line)->Y1 = (iY1), (Line)->X2 = (iX2), (Line)->Y2 = (iY2), Line)
#define SetLineNext(Line,Previous,iX,iY) SetLineColor((Line),(Previous)->X2,(Previous)->Y2,(iX),(iY))
#define SetLineNextColor(Line,Previous,iX,iY,iColor) SetLineColor((Line),(Previous)->X2,(Previous)->Y2,(iX),(iY),(iColor))

#define MoveLine(Line,X,Y) ((Line)->X1 += (X), (Line)->Y1 += (Y), (Line)->X2 += (X), (Line)->Y2 += (Y), Line)
#define ScaleLine(Line,XScale,YScale) ((Line)->X1 *= (XScale), (Line)->Y1 *= (YScale), (Line)->X2 *= (XScale), (Line)->Y2 *= (YScale), Line)

/* Rectangle macros */
#define SetRectangle(Rectangle,X1,Y1,X2,Y2) SetRectangleColor((Rectangle),(X1),(Y1),(X2),(Y2),GrayColor)
#define SetRectangleColor(Rectangle,iX1,iY1,iX2,iY2,iColor) ((Rectangle)->Color = (iColor), (Rectangle)->X1 = (iX1), (Rectangle)->Y1 = (iY1), (Rectangle)->X2 = (iX2), (Rectangle)->Y2 = (iY2), Rectangle)
#define SetRectangleSize(Rectangle,X,Y,Width,Height) SetRectangleSizeColor((Rectangle),(X),(Y),(Width),(Height),GrayColor)
#define SetRectangleSizeColor(Rectangle,X,Y,Width,Height,Color) SetRectangleColor((Rectangle),(X),(Y),(X)+(Width),(Y)+(Height),(Color))

#define MoveRectangle(Rectangle,X,Y) ((Rectangle)->X1 += (X), (Rectangle)->Y1 += (Y), (Rectangle)->X2 += (X), (Rectangle)->Y2 += (Y), Rectangle)
#define ScaleRectangle(Rectangle,XScale,YScale) ((Rectangle)->X1 *= (XScale), (Rectangle)->Y1 *= (YScale), (Rectangle)->X2 *= (XScale), (Rectangle)->Y2 *= (YScale), Rectangle)

/* Arc macros */
#define SetArc(Arc,X,Y,R) SetArcBeginEnd((Arc),(X),(Y),(R),0,360)
#define SetArcBeginEnd(Arc,X,Y,R,B,E) SetArcBeginEndColor((Arc),(X),(Y),(R),(B),(E),GrayColor)
#define SetArcColor(Arc,X,Y,R,Color) SetArcBeginEndColor((Arc),(X),(Y),(R),0,360,(Color))
#define SetArcBeginEndColor(Arc,iX,iY,iR,iB,iE,iColor) ((Arc)->Color = (iColor), (Arc)->X = (iX), (Arc)->Y = (iY), (Arc)->R = (iR), (Arc)->B = (iB), (Arc)->E = (iE), Arc)

#define MoveArc(Arc,iX,iY) ((Arc)->X += (iX), (Arc)->Y += (iY), Arc)
#define ScaleArc(Arc,XScale,YScale) ((Arc)->X *= (XScale), (Arc)->Y *= (YScale), (Arc)->R *= ((XScale)+(YScale))/2, Arc)

#endif

#ifdef __cplusplus
extern "C" {
#endif

#ifndef _GRALIMACROS
/* Point functions */
PointStruct * LIBENTRY SetPoint(PointStruct * Point,float X,float Y);
PointStruct * LIBENTRY SetPointColor(PointStruct * Point,float X,float Y,ColorType Color);
PointStruct * LIBENTRY SetPointType(PointStruct * Point,float X,float Y,ColorType Color,PointShape Shape);
PointStruct * LIBENTRY SetPointLabel(PointStruct * Point,float X,float Y,ColorType Color,PointShape Shape,char * Label);
PointStruct * LIBENTRY SetPointLabelFont(PointStruct * Point,float X,float Y,ColorType Color,PointShape Shape,char * Label,FontType Font);

PointStruct * LIBENTRY MovePoint(PointStruct * Point,float X,float Y);
PointStruct * LIBENTRY ScalePoint(PointStruct * Point,float XScale,float YScale);

/* Line functions */
LineStruct * LIBENTRY SetLine(LineStruct * Line,float X1,float Y1,float X2,float Y2);
LineStruct * LIBENTRY SetLineColor(LineStruct * Line,float X1,float Y1,float X2,float Y2,ColorType Color);
LineStruct * LIBENTRY SetLineNext(LineStruct * Line,LineStruct * Previous,float X,float Y);
LineStruct * LIBENTRY SetLineNextColor(LineStruct * Line,LineStruct * Previous,float X,float Y,ColorType Color);

LineStruct * LIBENTRY MoveLine(LineStruct * Line,float X,float Y);
LineStruct * LIBENTRY ScaleLine(LineStruct * Line,float XScale,float YScale);

/* Rect functions */
RectangleStruct * LIBENTRY SetRectangle(RectangleStruct * Rectangle,float X1,float Y1,float X2,float Y2);
RectangleStruct * LIBENTRY SetRectangleColor(RectangleStruct * Rectangle,float X1,float Y1,float X2,float Y2,ColorType Color);
RectangleStruct * LIBENTRY SetRectangleSize(RectangleStruct * Rectangle,float X,float Y,float Width,float Height);;
RectangleStruct * LIBENTRY SetRectangleSizeColor(RectangleStruct * Rectangle,float X,float Y,float Width,float Height,ColorType Color);

RectangleStruct * LIBENTRY MoveRectangle(RectangleStruct * Rectangle,float X,float Y);
RectangleStruct * LIBENTRY ScaleRectangle(RectangleStruct * Rectangle,float XScale,float YScale);

/* Arc functions */
ArcStruct * LIBENTRY SetArc(ArcStruct * Arc,float X,float Y,float R);
ArcStruct * LIBENTRY SetArcBeginEnd(ArcStruct * Arc,float X,float Y,float R,float B,float E);
ArcStruct * LIBENTRY SetArcColor(ArcStruct * Arc,float X,float Y,float R,ColorType Color);
ArcStruct * LIBENTRY SetArcBeginEndColor(ArcStruct * Arc,float X,float Y,float R,float B,float E,ColorType Color);

ArcStruct * LIBENTRY MoveArc(ArcStruct * Arc,float X,float Y);
ArcStruct * LIBENTRY ScaleArc(ArcStruct * Arc,float XScale,float YScale);

#endif

/* List macros */

#define ListExtraInfo(List) ((List)->ExtraInfo)
#define ListItemExtraInfo(Item) ((Item)->ExtraInfo)

/* Point add macros */
#define AddToListPoint(List,X,Y) AddToListPointColor((List),(X),(Y),GrayColor)
#define AddToListPointColor(List,X,Y,Color) AddToListPointType((List),(X),(Y),(Color),CrossPoint)
#define AddToListPointType(List,X,Y,Color,Shape) AddToListPointLabel((List),(X),(Y),(Color),(Shape),"")
#define AddToListPointLabel(List,X,Y,Color,Shape,Label) { PointStruct T; SetPointLabel(&T,(X),(Y),(Color),(Shape),(Label)); AddPointToList((List),&T); }

/* Line add macros */
#define AddToListLine(List,X1,Y1,X2,Y2) AddToListLineColor((List),(X1),(Y1),(X2),(Y2))
#define AddToListLineColor(List,X1,Y1,X2,Y2,Color) { LineStruct T; SetLineColor(&T,(X1),(Y1),(X2),(Y2),(Color)); AddLineToList((List),&T); }

/* Rectangle add macros */
#define AddToListRectangle(List,X1,Y1,X2,Y2) AddToListRectangleColor((List),(X1),(Y1),(X2),(Y2),GrayColor)
#define AddToListRectangleColor(List,X1,Y1,X2,Y2,Color) { RectangleStruct T; SetRectangleColor(&T,(X1),(Y1),(X2),(Y2),(Color)); AddRectangleToList((List),&T); }
#define AddToListRectangleSize(List,X,Y,Width,Height) AddToListRectangleSizeColor((List),(X),(Y),(Width),(Height),GrayColor)
#define AddToListRectangleSizeColor(List,X,Y,Width,Height,Color) AddToListRectangleColor((List),(X),(Y),(X)+(Width),(Y)+(Height),(Color))

/* Arc add macros */
#define AddToListArc(List,X,Y,R) AddToListArcBeginEnd((List),(X),(Y),(R),0,360)
#define AddToListArcBeginEnd(List,X,Y,R,B,E) AddToListArcBeginEndColor((List),(X),(Y),(R),(B),(E),GrayColor)
#define AddToListArcColor(Arc,X,Y,R,Color) AddToListArcBeginEndColor((List),(X),(Y),(R),0,360,(Color))
#define AddToListArcBeginEndColor(List,X,Y,R,B,E,Color) { ArcStruct T; SetArcBeginEndColor(&T,(X),(Y),(R),(B),(E),(Color)); AddArcToList((List),&T); }

/* List functions */

int LIBENTRY InitListSize(GraphicsListStruct * List,int MaxCount);
void LIBENTRY InitListBase(GraphicsListStruct * List,GraphicsItemStruct * Base,int MaxCount);
int LIBENTRY InitListCopy(GraphicsListStruct * List,const GraphicsListStruct * Original);
int LIBENTRY CopyList(GraphicsListStruct * List,const GraphicsListStruct * Original);
void LIBENTRY SetListCallback(GraphicsListStruct * List,void (*Callback)(GraphicsListStruct * List,void * ID),void * ID);
void LIBENTRY DestroyList(GraphicsListStruct * List);

int LIBENTRY AddToList(GraphicsListStruct * List,const GraphicsItemStruct * Item);
int LIBENTRY InsertToList(GraphicsListStruct * List,const GraphicsItemStruct * Item,int Index);
int LIBENTRY SetToList(GraphicsListStruct * List,const GraphicsItemStruct * Item,int Index);
GraphicsItemStruct * LIBENTRY GetFromList(const GraphicsListStruct * List,int Index);
int LIBENTRY RemoveFromList(GraphicsListStruct * List,int Index);
int LIBENTRY ResetInList(GraphicsListStruct * List,int Index);
int LIBENTRY GetListCount(const GraphicsListStruct * List);
void LIBENTRY ClearList(GraphicsListStruct * List);

void LIBENTRY MoveListItem(GraphicsItemStruct * Item,float X,float Y);
void LIBENTRY ScaleListItem(GraphicsItemStruct * Item,float XScale,float YScale);

GraphicsListStruct * LIBENTRY MoveList(GraphicsListStruct * List,float X,float Y);
GraphicsListStruct * LIBENTRY ScaleList(GraphicsListStruct * List,float XScale,float YScale);

int LIBENTRY AddPointToList(GraphicsListStruct * List,const PointStruct * Point);
int LIBENTRY AddLineToList(GraphicsListStruct * List,const LineStruct * Line);
int LIBENTRY AddRectangleToList(GraphicsListStruct * List,const RectangleStruct * Rectangle);
int LIBENTRY AddArcToList(GraphicsListStruct * List,const ArcStruct * Arc);

int LIBENTRY SetPointToList(GraphicsListStruct * List,const PointStruct * Point,int Index);
int LIBENTRY SetLineToList(GraphicsListStruct * List,const LineStruct * Line,int Index);
int LIBENTRY SetRectangleToList(GraphicsListStruct * List,const RectangleStruct * Rectangle,int Index);
int LIBENTRY SetArcToList(GraphicsListStruct * List,const ArcStruct * Arc,int Index);

PointStruct * LIBENTRY GetPointFromList(const GraphicsListStruct * List,int Index);
LineStruct * LIBENTRY GetLineFromList(const GraphicsListStruct * List,int Index);
RectangleStruct * LIBENTRY GetRectangleFromList(const GraphicsListStruct * List,int Index);
ArcStruct * LIBENTRY GetArcFromList(const GraphicsListStruct * List,int Index);

#ifdef __cplusplus
}
#endif

#endif

