/* Standard includes */
#include <assert.h>
#include <cmath>
#include <cstdlib>   /* malloc(), realloc() */
#include <cstring>

#include "Convolve.h"

/*********************************************************************
 * _computeKernels
 */

void computeKernels(float sigma, ConvolutionKernel &gauss)
{
    const float factor = 0.04f;   // for truncating tail 
    
    assert(MAX_KERNEL_WIDTH % 2 == 1);
    assert(sigma >= 0.0);
    assert(factor < 1.0f);
	float tmpData[MAX_KERNEL_WIDTH];
	int width = MAX_KERNEL_WIDTH;
  
    {  
        const int hw = MAX_KERNEL_WIDTH / 2;
        
        // Compute gauss
        float sum = 0;
        for( int i = -hw ; i <= hw ; i++)  {
            tmpData[ i + hw] = (float) exp( - i*i / (2*sigma*sigma));
        }
        
        // Compute width
        width = MAX_KERNEL_WIDTH;
        for( int i = -hw ; tmpData[ i + hw] / tmpData[ hw] < factor; 
             i++, width -= 2)
        ;
    
        // Shift the kernel
        for( int i = 0 ; i < width ; i++)
        {
            tmpData[i] = tmpData[ i + ( MAX_KERNEL_WIDTH - width) / 2];
        }
		gauss.width = width;
    }
     
    {   
        // Normalize the gauss 
        const int hw = gauss.width / 2;
        float sum = 0.0;
		memset(gauss.data, 0, MAX_KERNEL_WIDTH * sizeof(float));
        for( int i = 0 ; i < gauss.width ; i++)
        {
            sum += tmpData[i];
        }
        for( int i = 0 ; i < gauss.width ; i++)
        {
            gauss.data[i] = (tmpData[i] / sum);
        }
    }
}


void convolveImageHoriz(
    unsigned char * imgIn,
    unsigned char * imgOut,
    int sizeX, int sizeY,
    ConvolutionKernel & kernel)
{

    int hw = kernel.width / 2;
    
    for( int y = 0; y < sizeY; y++){
        for( int x = 0; x < sizeX; x++){
            int from = -hw;
            int to = hw;
            
            if( x + from < 0){
                from = -x;
            }
            if( x + to > sizeX){
                to = sizeX - x;
            }
                
            float sum = 0;
            float coefSum = 0;
            for( int i = from; i < to; i++){
                sum += (kernel.data[ hw + i] * imgIn[ x + i + y * sizeX]);
                coefSum += kernel.data[ hw + i];
            }
            imgOut[ x + y * sizeX] = int(sum / coefSum);
        }
    }
}

void convolveImageVert(
    unsigned char * imgIn,
    unsigned char * imgOut,
    int sizeX, int sizeY,
    ConvolutionKernel & kernel)
{

    int hw = kernel.width / 2;
    
    for( int y = 0; y < sizeY; y++){
        for( int x = 0; x < sizeX; x++){
            int from = -hw;
            int to = hw;
            
            if( y + from < 0){
                from = -y;
            }
            if( y + to > sizeY){
                to = sizeY - y;
            }
                
            float sum = 0;
            float coefSum = 0;
            for( int i = from; i < to; i++){
                sum += (kernel.data[ hw + i] * imgIn[ x + (y + i) * sizeX]);
                coefSum += kernel.data[ hw + i];
            }
            imgOut[ x + y * sizeX] = int(sum / coefSum);
        }
    }
}