/* 
   CVDLBridge.h
   Author: Jiri Venera
   Email: venera@fit.vutbr.cz
*/
#ifndef _CVDLBRIDGE_H_
#define _CVDLBRIDGE_H_

#include <stdio.h>
#include "ImageFil.h"
#include "cv.h"
#include "cxcore.h"
#include "cxmisc.h"

/*
   If you want export bellow defined function, you must define this fuction as __declspec(dllexport) 
*/
#ifndef DLL_EXPORT
 #define DLL_EXPORT __declspec(dllexport)
#endif

/*
   Memory managenemt function for OpenCVDigILibBridge
*/


/*
   Pointer definition to function for memory buffer allocation. 
   Function substitutes simple alloc or cv default alloc function (icvDefaultAlloc).
*/
typedef void* (__cdecl *CVDLAllocPtr)(size_t size);
/*
   Pointer definition to function for memory buffer deallocation.
   Function substitutes simple free or cv default free function (icvDefaultFree). 
*/
typedef void (__cdecl *CVDLFreePtr)(void* ptr);
/*
   Allocates memory buffer for cvdl. 
*/   
extern "C" DLL_EXPORT void* __cdecl cvdlAlloc( size_t size );
/*
   Deallocates memory buffer for cvdl. 
*/
extern "C" DLL_EXPORT void __cdecl cvdlFree( void* pptr );

/*
   cvdl allocation function. 
   If you want use OpenCVDigILib functions, you must override OpenCV default function by its.
   OpenCV allocation function is automaticly overriden at the attaching of DLL.
   This function extends information about strored images.
*/
extern "C" DLL_EXPORT void* __cdecl cvdlAllocFunc(size_t size, void* userdata);
/*
   cvdl dealocation function. 
   If you want use OpenCVDigILib functions, you must override OpenCV default function by its.
   OpenCV deallocation function is automaticly overriden at the attaching of DLL.
   This function uses extended informations stored by cvdlAllocFunc for memory freeing.
*/
extern "C" DLL_EXPORT int __cdecl cvdlFreeFunc(void* pptr, void* userdata);
/*
   OpenCVDiILib memory managment initialization function. 
   This function must be called before using any function declared in this module. 
   Function is called automaticly at the DLL begining. If yuo override OpenCV memory 
   management function or function for Ipl images allocation you maust call this 
   function again in your code.
*/
extern "C" DLL_EXPORT int __cdecl cvdlInit();

/*
   Memory managenemt function for OpenCVDigILibBridge
*/

/*
   Function for Ipl image cloning. Function substitutes default OpenCV function called by
   cvCloneImage(). For image cloning use cvCloneImage() function.
*/
extern "C" DLL_EXPORT IplImage* __stdcall cvdlCloneIplImage( const IplImage* src );
/*
   Function for Ipl image creating. Function substitutes default OpenCV function called by
   cvCreateImageHeader(). For creating of Ipl image header use cvCreateImageHeader() function.
*/
extern "C" DLL_EXPORT IplImage* __stdcall cvdlCreateIplImageHeader
( 
   int channels , 
   int, 
   int depth,
   char* colorModel,
   char* channelSeq,
   int dataOrderPixel,
   int origin,
   int align,
   int width,
   int height,
   IplROI* iplRoi ,
   IplImage* iplImage ,
   void*,
   IplTileInfo* iplTitleInfo
);
/*
   Function for Ipl image data releasing. Function substitutes default OpenCV function called by
   cvReleaseImageData(). For releasing of image data use cvReleaseImageData() function.
*/
extern "C" DLL_EXPORT void __stdcall cvdlReleaseIplImageData( IplImage* iplImage , int section_deallocation );
/*
   Function for Ipl image data creation. Function substitutes default OpenCV function called by
   cvCreateImageData(). For creating of image data use cvCreateImageData() function.
*/
extern "C" DLL_EXPORT void __stdcall cvdlCreateIplImageData( IplImage* iplImage , int , int );
/*
   Function for Ipl image RIO creation. Function substitutes default OpenCV function called by
   cvCreateImageROI(). For creating of image ROI use cvCreateImageROI() function.
*/
extern "C" DLL_EXPORT IplROI* __stdcall cvdlCreateIplImageROI( int coi, int xOffset, int yOffset, int width, int height );
/*
   Function converts ImageStruct stucture to IplImage format.
   This function supports conversion only for ImageStruct formats:
   - ImageRGBLinear
   - ImageRGB
   - Image8Linear
   - Image16Linear
   - Image32Linear
   - ImageFloatLinear
   If the function successed, returns pointer to created IplImage and double pointer to ImageStruct
   passed as function argument is set to NULL.
   If the function failed, calls CVDL_SHOW_ERRORS macro (if it's defined) or return pointer to NULL.
   
   Function does not work with COI or ROI for now.
*/
extern "C" DLL_EXPORT IplImage* ImageStruct2IplImage(ImageStruct** img );
/*
   Function converts IplImage format to ImageStruct structure.
   This function supports conversion only for IplImage formats:
   - RGBA image with 4 channels and depth IPL_DEPTH_8U (unsigned char)
   - RGB image with 3 channels and depth IPL_DEPTH_8U (unsigned char)
   - GRAY image with 1 channel
            . depth IPL_DEPTH_16S
            . IPL_DEPTH_8U
            . IPL_DEPTH_32S
            
   If the function successed, returns pointer to created ImageStruct structure and double pointer to IplImage
   passed as function argument is set to NULL.
   If the function failed, calls CVDL_SHOW_ERRORS macro (if it's defined) or return pointer to NULL.
   
   Function does not work with COI or ROI for now.
            
*/
extern "C" DLL_EXPORT ImageStruct* IplImage2ImageStruct(IplImage** img );

/*
   Definitiom of C++ class used for cvdl library initialization.
   C-tordirectly calls ::cvdlInit() function for setting of memory management functions.
*/
#ifdef __cplusplus
   class DLL_EXPORT CVDLInit
   {
   public:
      int retCode;
      CVDLInit()
         :retCode(CV_OK) 
      {
         this->retCode = ::cvdlInit();
      };
   };
#endif
/*
   Initialization macro for cvdl.
*/
#ifndef _CVDL_BEGIN_
   #ifdef __cplusplus
      #define _CVDL_BEGIN_ CVDLInit cvdl_init;
   #else
      #define _CVDL_BEGIN_ int result = ::cvdlInit();
   #endif
#endif

/*
   Define macro for runtime errors. If you want disable error showing, undef this macro
*/
#define CVDL_SHOW_ERRORS

#ifdef CVDL_SHOW_ERRORS
   #define CVDL_ERROR( Code, Msg ) CV_ERROR( Code, Msg )
#else
   #define CVDL_ERROR( Code, Msg ) goto exit
#endif



#endif