/* 
   CVDLBridge.cpp
   Author: Jiri Venera
   Email: venera@fit.vutbr.cz
*/
#include "CVDLBridge.h"
#include <stdlib.h>

/*
   cvdl Initialization.
*/
_CVDL_BEGIN_

typedef enum _CVDL_ALLOC_TYPE_
{
   CV_ALLOC_DATA  = 0x0,
   DL_ALLOC_DATA  = 0x1,
} 
_CVDL_ALLOC_TYPE_;

typedef struct _CVDL_ALLOC_DATA_
{
   _CVDL_ALLOC_TYPE_ cvdl_alloc_type;
   union 
   {
      void* void_ptr;
      ImageStruct* image_stuct;
   }
   data_ptr;
}
_CVDL_ALLOC_DATA_;

static char* cvdlDescription = "CVDL Bridge";


extern "C" int __cdecl  cvdlInit()
{
   CV_FUNCNAME( "cvdlInit()" );

   __BEGIN__;
   
   cvSetMemoryManager( 
      cvdlAllocFunc, 
      cvdlFreeFunc , 
      (void*)cvdlDescription );
   cvSetIPLAllocators( 
      cvdlCreateIplImageHeader,
      cvdlCreateIplImageData,
      cvdlReleaseIplImageData,
      cvdlCreateIplImageROI,
      cvdlCloneIplImage );
      
   __END__;
   return CV_OK;
}


static CVDLAllocPtr cvdlAllocPtr = cvdlAlloc;
static CVDLFreePtr cvdlFreePtr = cvdlFree;


extern "C" void* __cdecl cvdlAlloc( size_t size )
{
   char *ptr, *ptr0 = (char*)malloc(
        (size_t)(size + CV_MALLOC_ALIGN*((size >= 4096) + 1) + sizeof(char*)));

   if( !ptr0 )
     return 0;

   ptr = (char*)cvAlignPtr(ptr0 + sizeof(char*) + 1, CV_MALLOC_ALIGN);
   *(char**)(ptr - sizeof(char*)) = ptr0;

   return ptr;
}

extern "C" void __cdecl cvdlFree( void* pptr )
{
   assert( ((size_t)pptr & (CV_MALLOC_ALIGN-1)) == 0 );
   free( *((char**)pptr - 1) );
}



extern "C" void* __cdecl cvdlAllocFunc(size_t size, void* userdata)
{
   void* ptr = 0;

   CV_FUNCNAME( "cvdlAllocFunc" );
   (userdata);

   __BEGIN__;

   if( (unsigned)size > CV_MAX_ALLOC_SIZE )
   {
      CVDL_ERROR( CV_StsOutOfRange, "Negative or too large argument of cvdlAllocPtr function" );
   }

   ptr = cvdlAllocPtr( size);
   if( !ptr )
     CVDL_ERROR( CV_StsNoMem, "Out of memory" );

   __END__;

   return ptr;
}
extern "C" int __cdecl cvdlFreeFunc(void* pptr, void* userdata)
{
   CV_FUNCNAME( "cvdlFreeFunc" );
   (userdata);

   __BEGIN__;

   if( pptr )
   {
      cvdlFreePtr(pptr);
   }

   __END__;

   return CV_OK;
}





extern "C" IplImage* Image8Linear2IplImage(ImageStruct** img)
{
   IplImage* iplPtr = NULL;
   _CVDL_ALLOC_DATA_* cvdlAllocType = NULL;
   
   CV_FUNCNAME("Image8Linear2IplImage");
   
   __BEGIN__;
   
   if ( img == NULL || *img == NULL)
   {
      CVDL_ERROR(CV_StsNullPtr,"Null double pointer to ImageStruct passed as a function argument.");
   }
   if ( (*img)->PixelType != Image8Linear || (*img)->XOffset != sizeof(Image8Value) )
   {
      CVDL_ERROR(CV_StsError, "ImageStruct must be PixelType equal to Image8Linear and XOffset equal 1.");
   } 
   
   iplPtr = cvCreateImageHeader( cvSize((*img)->XSize , (*img)->YSize) , IPL_DEPTH_8U , 1 );
	cvdlAllocType = (_CVDL_ALLOC_DATA_*)(iplPtr+1);
	cvdlAllocType->cvdl_alloc_type = DL_ALLOC_DATA;
	cvdlAllocType->data_ptr.image_stuct = *img;
	
	if ( (*img)->YOffset < 0 )
	{
	   iplPtr->imageData = iplPtr->imageDataOrigin = (char*)((*img)->Raster+((*img)->YSize-1)*(*img)->YOffset);
	   iplPtr->origin = 1;
	   iplPtr->widthStep = -(*img)->YOffset;
	}
	else
	{
	   iplPtr->imageData = iplPtr->imageDataOrigin = (char*)(*img)->Raster;
	   iplPtr->origin = 0;
	   iplPtr->widthStep = (*img)->YOffset;
	}
   *img = NULL;
   
   __END__;
   return iplPtr;
   
}

extern "C" IplImage* ImageRGBLinear2IplImage(ImageStruct** img)
{
   IplImage* iplPtr = NULL;
   _CVDL_ALLOC_DATA_* cvdlAllocType = NULL;
   
   CV_FUNCNAME("ImageRGBLinear2IplImage");
   
   __BEGIN__;
   
   if ( img == NULL || *img == NULL)
   {
      CVDL_ERROR(CV_StsNullPtr,"Null double pointer to ImageStruct passed as a function argument.");
   } 
   if ( (*img)->PixelType != ImageRGBLinear && (*img)->XOffset != sizeof(ImageRGBValueRGB))
   {
      CVDL_ERROR(CV_StsError, "ImageStruct must be PixelType equal to ImageRGBLinear and XOffset equal 4.");
   }
   
   iplPtr = cvCreateImageHeader( cvSize((*img)->XSize , (*img)->YSize) , IPL_DEPTH_8U , 4 );
	cvdlAllocType = (_CVDL_ALLOC_DATA_*)(iplPtr+1);
	cvdlAllocType->cvdl_alloc_type = DL_ALLOC_DATA;
	cvdlAllocType->data_ptr.image_stuct = *img;
			
	if ( (*img)->YOffset < 0 )
	{
	   iplPtr->imageData = iplPtr->imageDataOrigin = (char*)((*img)->Raster+((*img)->YSize-1)*(*img)->YOffset);
	   iplPtr->origin = 1;
	   iplPtr->widthStep = -(*img)->YOffset;
	}
	else
	{
	   iplPtr->imageData = iplPtr->imageDataOrigin = (char*)(*img)->Raster;
	   iplPtr->origin = 0;
	   iplPtr->widthStep = (*img)->YOffset;
	}
   *img = NULL;
   
   __END__;
   return iplPtr;
}

extern "C" IplImage* ImageRGB2IplImage(ImageStruct** img)
{
   IplImage* iplPtr = NULL;
   _CVDL_ALLOC_DATA_* cvdlAllocType = NULL;
   
   CV_FUNCNAME("ImageRGB2IplImage");
   
   __BEGIN__;
   
   if ( img == NULL || *img == NULL)
   {
      CVDL_ERROR(CV_StsNullPtr,"Null double pointer to ImageStruct passed as a function argument.");
   } 
   if ( (*img)->PixelType != ImageRGB && (*img)->XOffset != 3 )
   {
      CVDL_ERROR(CV_StsError, "ImageStruct must be PixelType equals to ImageRGB and XOffset equal 3.");
   }
   
   iplPtr = cvCreateImageHeader( cvSize((*img)->XSize , (*img)->YSize) , IPL_DEPTH_8U , 3 );
	cvdlAllocType = (_CVDL_ALLOC_DATA_*)(iplPtr+1);
	cvdlAllocType->cvdl_alloc_type = DL_ALLOC_DATA;
	cvdlAllocType->data_ptr.image_stuct = *img;
			
	if ( (*img)->YOffset < 0 )
	{
	   iplPtr->imageData = iplPtr->imageDataOrigin = (char*)((*img)->Raster+((*img)->YSize-1)*(*img)->YOffset);
	   iplPtr->origin = 1;
	   iplPtr->widthStep = -(*img)->YOffset;
	}
	else
	{
	   iplPtr->imageData = iplPtr->imageDataOrigin = (char*)(*img)->Raster;
	   iplPtr->origin = 0;
	   iplPtr->widthStep = (*img)->YOffset;
	}
   *img = NULL;
   
   __END__;
   return iplPtr;
}

extern "C" IplImage* ImageFloatLinear2IplImage(ImageStruct** img)
{
   IplImage* iplPtr = NULL;
   _CVDL_ALLOC_DATA_* cvdlAllocType = NULL;
   
   CV_FUNCNAME("Image32Linear2IplImage");
   
   __BEGIN__;
   
   if ( img == NULL || *img == NULL)
   {
      CVDL_ERROR(CV_StsNullPtr,"Null double pointer to ImageStruct passed as a function argument.");
   } 
   if ( (*img)->PixelType != ImageFloatLinear && (*img)->XOffset != sizeof(ImageFloatValue) )
   {
      CVDL_ERROR(CV_StsError, "ImageStruct must be PixelType equal to Image32Linear and XOffset equal 4.");
   }
   
   iplPtr = cvCreateImageHeader( cvSize((*img)->XSize , (*img)->YSize) , IPL_DEPTH_32F , 1 );
	cvdlAllocType = (_CVDL_ALLOC_DATA_*)(iplPtr+1);
	cvdlAllocType->cvdl_alloc_type = DL_ALLOC_DATA;
	cvdlAllocType->data_ptr.image_stuct = *img;
			
	if ( (*img)->YOffset < 0 )
	{
	   iplPtr->imageData = iplPtr->imageDataOrigin = (char*)((*img)->Raster+((*img)->YSize-1)*(*img)->YOffset);
	   iplPtr->origin = 1;
	   iplPtr->widthStep = -(*img)->YOffset;
	}
	else
	{
	   iplPtr->imageData = iplPtr->imageDataOrigin = (char*)(*img)->Raster;
	   iplPtr->origin = 0;
	   iplPtr->widthStep = (*img)->YOffset;
	}
   *img = NULL;
   
   __END__;
   return iplPtr;
}

extern "C" IplImage*  Image32Linear2IplImage(ImageStruct** img )
{
   IplImage* iplPtr = NULL;
   _CVDL_ALLOC_DATA_* cvdlAllocType = NULL;
   
   CV_FUNCNAME("Image32Linear2IplImage");
   
   __BEGIN__;
   
   if ( img == NULL || *img == NULL)
   {
      CVDL_ERROR(CV_StsNullPtr,"Null double pointer to ImageStruct passed as a function argument.");
   } 
   if ( (*img)->PixelType != Image32Linear && (*img)->XOffset != sizeof(Image32Value) )
   {
      CVDL_ERROR(CV_StsError, "ImageStruct must be PixelType equal to Image32Linear and XOffset equal 4.");
   }
   
   iplPtr = cvCreateImageHeader( cvSize((*img)->XSize , (*img)->YSize) , IPL_DEPTH_32S , 1 );
	cvdlAllocType = (_CVDL_ALLOC_DATA_*)(iplPtr+1);
	cvdlAllocType->cvdl_alloc_type = DL_ALLOC_DATA;
	cvdlAllocType->data_ptr.image_stuct = *img;
			
	if ( (*img)->YOffset < 0 )
	{
	   iplPtr->imageData = iplPtr->imageDataOrigin = (char*)((*img)->Raster+((*img)->YSize-1)*(*img)->YOffset);
	   iplPtr->origin = 1;
	   iplPtr->widthStep = -(*img)->YOffset;
	}
	else
	{
	   iplPtr->imageData = iplPtr->imageDataOrigin = (char*)(*img)->Raster;
	   iplPtr->origin = 0;
	   iplPtr->widthStep = (*img)->YOffset;
	}
   *img = NULL;
   
   __END__;
   return iplPtr;
}

extern "C" IplImage* Image16Linear2IplImage(ImageStruct** img)
{
   IplImage* iplPtr = NULL;
   _CVDL_ALLOC_DATA_* cvdlAllocType = NULL;
   
   CV_FUNCNAME("Image16Linear2IplImage");
   
   __BEGIN__;
   
   if ( img == NULL || *img == NULL)
   {
      CVDL_ERROR(CV_StsNullPtr,"Null double pointer to ImageStruct passed as a function argument.");
   } 
   if ( (*img)->PixelType != Image16Linear && (*img)->XOffset != sizeof(Image16Value) )
   {
      CVDL_ERROR(CV_StsError, "ImageStruct must be PixelType equal to Image16Linear and XOffset equal 2.");
   }
   
   iplPtr = cvCreateImageHeader( cvSize((*img)->XSize , (*img)->YSize) , IPL_DEPTH_16S , 1 );
	cvdlAllocType = (_CVDL_ALLOC_DATA_*)(iplPtr+1);
	cvdlAllocType->cvdl_alloc_type = DL_ALLOC_DATA;
	cvdlAllocType->data_ptr.image_stuct = *img;
			
	if ( (*img)->YOffset < 0 )
	{
	   iplPtr->imageData = iplPtr->imageDataOrigin = (char*)((*img)->Raster+((*img)->YSize-1)*(*img)->YOffset);
	   iplPtr->origin = 1;
	   iplPtr->widthStep = -(*img)->YOffset;
	}
	else
	{
	   iplPtr->imageData = iplPtr->imageDataOrigin = (char*)(*img)->Raster;
	   iplPtr->origin = 0;
	   iplPtr->widthStep = (*img)->YOffset;
	}
   *img = NULL;
   
   __END__;
   return iplPtr;
}




extern "C" IplImage* ImageStruct2IplImage(ImageStruct** img )
{
   

   CV_FUNCNAME("ImageStruct2IplImage");

   __BEGIN__;
   if ( img == NULL || *img == NULL)
   {
      CVDL_ERROR(CV_StsNullPtr,"Null double pointer to ImageStruct passed as a function argument.");
   } 
   switch ( (*img)->PixelType )
   {
      case ImageRGBLinear :
      {
         return ImageRGBLinear2IplImage(img);
      }
      break;
      case ImageRGB :
      {
         return ImageRGB2IplImage(img);
      }
      break;
      case Image8Linear :
      {
         return Image8Linear2IplImage(img);
      }
      break;
      case Image16Linear :
      {
         return Image16Linear2IplImage(img);
      }
      break;
      case Image32Linear :
      {
         return Image32Linear2IplImage(img);
      }
      break;
      case ImageFloatLinear :
      {
         return ImageFloatLinear2IplImage(img);
      }
      break;
      default : 
         CVDL_ERROR(CV_StsError, "Unsupported format of ImageStruct. Image can't be converted." );
      break;
   }

   __END__;

   return NULL;
}

extern "C" ImageStruct* IplImage2ImageRGBLinear(IplImage** img )
{
   ImageStruct* imageSruct = NULL;
   IplImage* iplImage = NULL;
   
   CV_FUNCNAME("IplImage2ImageRGBLinear");
   
   __BEGIN__;
   
   
   if ( img == NULL || *img == NULL)
   {
      CVDL_ERROR(CV_StsNullPtr,"Null double pointer to IplImage passed as a function argument.");
   } 
   if ( (*img)->nChannels != 4 )
   {
      CVDL_ERROR(CV_StsError, "IplImage isn't in RGBA format! Function accepts only images with IPL_DEPTH_8U and nChannels equal to 4.");
   }
   if ( (*img)->depth != IPL_DEPTH_8U )
   {
      CVDL_ERROR(CV_BadDepth, "Image don't have IPL_DEPTH_8U depth, can't convert this format to DigILib! Function accepts only images with IPL_DEPTH_8U and nChannels equal to 4." );
   }


   imageSruct = (ImageStruct*)cvdlAllocPtr(sizeof(ImageStruct));

   memset((void*)imageSruct,0,sizeof(ImageStruct));
   imageSruct->PixelType = ImageRGBLinear;
   imageSruct->References = 1;
   imageSruct->ImageType = RegularImage;
   imageSruct->XSize = (*img)->width;
   imageSruct->YSize = (*img)->height;
   imageSruct->XOffset = (*img)->nChannels;

   imageSruct->Optional.Regular.Free = cvdlFreePtr;
               
   if ( (*img)->origin == 1 ) /* Obrazek je vzhuru nohama:-), imlicitni pro DigILib */
   {
      imageSruct->YOffset = -(*img)->widthStep;
      imageSruct->Raster = (unsigned char*)(((char*)(*img)->imageData) + ((*img)->height-1)*(*img)->widthStep);
   }
   else
   {
      imageSruct->YOffset = (*img)->widthStep;
      imageSruct->Raster = (unsigned char*)((*img)->imageData);
   }
   iplImage = *img;
   *img = NULL;

   cvReleaseImageHeader(&iplImage);
   
   __END__;
   
   return imageSruct;
}

extern "C" ImageStruct* IplImage2ImageRGB(IplImage** img )
{
   ImageStruct* newImg = NULL;
   
   CV_FUNCNAME("IplImage2ImageRGB");
   
   __BEGIN__;
   
   
   if ( img == NULL || *img == NULL)
   {
      CVDL_ERROR(CV_StsNullPtr,"Null double pointer to IplImage passed as a function argument.");
   } 
   if ( (*img)->nChannels != 3 )
   {
      CVDL_ERROR(CV_StsError, "IplImage isn't in RGB format! Function accepts only images with IPL_DEPTH_8U and nChannels equal to 3.");
   }
   if ( (*img)->depth != IPL_DEPTH_8U )
   {
      CVDL_ERROR(CV_BadDepth, "Image don't have IPL_DEPTH_8U depth, can't convert this format to DigILib! Function accepts only images with IPL_DEPTH_8U and nChannels equal to 3." );
   }

   newImg = (ImageStruct*)cvdlAllocPtr(sizeof(ImageStruct));
   memset((void*)newImg,0,sizeof(ImageStruct));

   newImg->PixelType = ImageRGB;
   newImg->References = 1;
   newImg->ImageType = RegularImage;
   newImg->XSize = (*img)->width;
   newImg->YSize = (*img)->height;
   newImg->XOffset = (*img)->nChannels;

   newImg->Optional.Regular.Free = cvdlFreePtr;
               
   if ( (*img)->origin ) /* Obrazek je vzhuru nohama:-), imlicitni pro DigILib */
   {
      newImg->YOffset = -(*img)->widthStep;
      newImg->Raster = (unsigned char*)(((char*)(*img)->imageData) + ((*img)->height-1)*(*img)->widthStep);
   }
   else
   {
      newImg->YOffset = (*img)->widthStep;
      newImg->Raster = (unsigned char*)((*img)->imageData);
   }

   cvReleaseImageHeader(img);
   *img = NULL;
   
   __END__;

   return newImg;
}

ImageStruct* IplImage2Image8(IplImage** img )
{
   ImageStruct* newImg = NULL;
   (img);
   
   /* CV_FUNCNAME("IplImage2Image8");
   
   __BEGIN__;
   
   
   if ( img == NULL || *img == NULL)
   {
      CVDL_ERROR(CV_StsNullPtr,"Null double pointer to IplImage passed as a function argument.");
   } 
   if ( (*img)->nChannels != 1 )
   {
      CVDL_ERROR(CV_StsError, "IplImage isn't in GRAY format! Function accepts only images with IPL_DEPTH_8U and nChannels equal to 1.");
   }
   if ( (*img)->depth != IPL_DEPTH_8U )
   {
      CVDL_ERROR(CV_BadDepth, "Image don't have IPL_DEPTH_8U depth, can't convert this format to DigILib! Function accepts only images with IPL_DEPTH_8U and nChannels equal to 3." );
   }
  
   newImg = (ImageStruct*)cvdlAllocPtr(sizeof(ImageStruct));
   memset((void*)newImg,0,sizeof(ImageStruct));

   newImg->PixelType = Image8;
   newImg->References = 1;
   newImg->ImageType = RegularImage;
   newImg->XSize = (*img)->width;
   newImg->YSize = (*img)->height;
   newImg->XOffset = (*img)->nChannels;

   newImg->Optional.Regular.Free = cvdlFreePtr;
               
   if ( (*img)->origin ) //Obrazek je vzhuru nohama:-), imlicitni pro DigILib
   {
      newImg->YOffset = -(*img)->widthStep;
      newImg->Raster = (unsigned char*)(((char*)(*img)->imageData) + ((*img)->height-1)*(*img)->widthStep);
   }
   else
   {
      newImg->YOffset = (*img)->widthStep;
      newImg->Raster = (unsigned char*)((*img)->imageData);
   }

   cvReleaseImageHeader(img);
   img = NULL;
   
   __END__; */

   return newImg;
}

extern "C" ImageStruct* IplImage2Image16Linear(IplImage** img)
{
   ImageStruct* newImg = NULL;
   
   CV_FUNCNAME("IplImage2Image16Linear");
   
   __BEGIN__;
   
   
   if ( img == NULL || *img == NULL)
   {
      CVDL_ERROR(CV_StsNullPtr,"Null double pointer to IplImage passed as a function argument.");
   } 
   if ( (*img)->nChannels != 1 )
   {
      CVDL_ERROR(CV_StsError, "IplImage does not have 1 channel ! Function accepts only images with IPL_DEPTH_16S and nChannels equal to 1.");
   }
   if ( (*img)->depth != IPL_DEPTH_16S )
   {
      CVDL_ERROR(CV_BadDepth, "Image don't have IPL_DEPTH_16U depth, can't convert this format to DigILib! Function accepts only images with IPL_DEPTH_16S and nChannels equal to 1." );
   }

   newImg = (ImageStruct*)cvdlAllocPtr(sizeof(ImageStruct));
   memset((void*)newImg,0,sizeof(ImageStruct));

   newImg->PixelType = Image16Linear;
   newImg->References = 1;
   newImg->ImageType = RegularImage;
   newImg->XSize = (*img)->width;
   newImg->YSize = (*img)->height;
   newImg->XOffset = (*img)->nChannels * sizeof(Image16Value);

   newImg->Optional.Regular.Free = cvdlFreePtr;
   //            
   if ( (*img)->origin ) /* Obrazek je vzhuru nohama:-), imlicitni pro DigILib */
   {
      newImg->YOffset = -(*img)->widthStep;
      newImg->Raster = (unsigned char*)(((char*)(*img)->imageData) + ((*img)->height-1)*(*img)->widthStep);
   }
   else
   {
      newImg->YOffset = (*img)->widthStep;
      newImg->Raster = (unsigned char*)((*img)->imageData);
   }

   cvReleaseImageHeader(img);
   *img = NULL;
   
   __END__;

   return newImg;
}

extern "C" ImageStruct* IplImage2Image32Linear(IplImage** img )
{
   ImageStruct* newImg = NULL;
   
   CV_FUNCNAME("IplImage2Image32Linear");
   
   __BEGIN__;
   
   
   if ( img == NULL || *img == NULL)
   {
      CVDL_ERROR(CV_StsNullPtr,"Null double pointer to IplImage passed as a function argument.");
   } 
   if ( (*img)->nChannels != 1 )
   {
      CVDL_ERROR(CV_StsError, "IplImage isn't in GRAY format! Function accepts only images with IPL_DEPTH_32S and nChannels equal to 1.");
   }
   if ( (*img)->depth != IPL_DEPTH_32S )
   {
      CVDL_ERROR(CV_BadDepth, "Image don't have IPL_DEPTH_32S depth, can't convert this format to DigILib! Function accepts only images with IPL_DEPTH_32S and nChannels equal to 1." );
   }

   newImg = (ImageStruct*)cvdlAllocPtr(sizeof(ImageStruct));
   memset((void*)newImg,0,sizeof(ImageStruct));

   newImg->PixelType = Image32Linear;
   newImg->References = 1;
   newImg->ImageType = RegularImage;
   newImg->XSize = (*img)->width;
   newImg->YSize = (*img)->height;
   newImg->XOffset = (*img)->nChannels*sizeof(Image32Value);

   newImg->Optional.Regular.Free = cvdlFreePtr;
   //            
   if ( (*img)->origin ) /* Obrazek je vzhuru nohama:-), imlicitni pro DigILib */
   {
      newImg->YOffset = -(*img)->widthStep;
      newImg->Raster = (unsigned char*)(((char*)(*img)->imageData) + ((*img)->height-1)*(*img)->widthStep);
   }
   else
   {
      newImg->YOffset = (*img)->widthStep;
      newImg->Raster = (unsigned char*)((*img)->imageData);
   }

   cvReleaseImageHeader(img);
   *img = NULL;
   
   __END__;

   return newImg;
}

extern "C" ImageStruct* IplImage2Image8Linear(IplImage** img )
{
   ImageStruct* newImg = NULL;
   
   CV_FUNCNAME("IplImage2Image8Linear");
   
   __BEGIN__;
   
   
   if ( img == NULL || *img == NULL)
   {
      CVDL_ERROR(CV_StsNullPtr,"Null double pointer to IplImage passed as a function argument.");
   } 
   if ( (*img)->nChannels != 1 )
   {
      CVDL_ERROR(CV_StsError, "IplImage isn't in GRAY format! Function accepts only images with IPL_DEPTH_8U and nChannels equal to 1.");
   }
   if ( (*img)->depth != IPL_DEPTH_8U )
   {
      CVDL_ERROR(CV_BadDepth, "Image don't have IPL_DEPTH_8U depth, can't convert this format to DigILib! Function accepts only images with IPL_DEPTH_8U and nChannels equal to 1." );
   }

   newImg = (ImageStruct*)cvdlAllocPtr(sizeof(ImageStruct));
   memset((void*)newImg,0,sizeof(ImageStruct));

   newImg->PixelType = Image8Linear;
   newImg->References = 1;
   newImg->ImageType = RegularImage;
   newImg->XSize = (*img)->width;
   newImg->YSize = (*img)->height;
   newImg->XOffset = (*img)->nChannels;

   newImg->Optional.Regular.Free = cvdlFreePtr;
   //            
   if ( (*img)->origin ) /* Obrazek je vzhuru nohama:-), imlicitni pro DigILib */
   {
      newImg->YOffset = -(*img)->widthStep;
      newImg->Raster = (unsigned char*)(((char*)(*img)->imageData) + ((*img)->height-1)*(*img)->widthStep);
   }
   else
   {
      newImg->YOffset = (*img)->widthStep;
      newImg->Raster = (unsigned char*)((*img)->imageData);
   }

   cvReleaseImageHeader(img);
   *img = NULL;
   
   __END__;

   return newImg;
}

extern "C" ImageStruct* cvdlOriginalImageStruct(IplImage** img)
{
   ImageStruct* isImg = NULL;
   _CVDL_ALLOC_DATA_* cvdlAllocData = (_CVDL_ALLOC_DATA_*)((*img) + 1 );
   CV_FUNCNAME("IplImage2ImageStruct");
   

   __BEGIN__;
   if ( img == NULL || *img == NULL)
   {
      CVDL_ERROR(CV_StsNullPtr,"Null double pointer to IplImage passed as a function argument.");
   }
   
   if ( cvdlAllocData->cvdl_alloc_type == DL_ALLOC_DATA )
   {
      isImg = cvdlAllocData->data_ptr.image_stuct;
      cvdlFreePtr(*img);
      *img = NULL;
   }
   __END__;

   return isImg;
}


CV_IMPL ImageStruct* IplImage2ImageStruct(IplImage** img )
{   
   ImageStruct* isPtr = NULL;
   CV_FUNCNAME("IplImage2ImageStruct");

   __BEGIN__;
   if ( img == NULL || *img == NULL)
   {
      CVDL_ERROR(CV_StsNullPtr,"Null double pointer to IplImage passed as a function argument.");
   }
   
   if ((isPtr=cvdlOriginalImageStruct(img))!=NULL)
   {
      return isPtr;
   }

   switch ( (*img)->nChannels )
   {
      case 4 : /* RGBA - alpha channel isn't used ImageRGBLinear */
      {    
         if ( (*img)->depth == IPL_DEPTH_8U )
         {
            return IplImage2ImageRGBLinear(img); 
         }
         else
         {
            CVDL_ERROR(CV_StsError, "Unsupported format of IplImage. Image can't be converted.");            
         }
      }
      break;
      case 3 : /* RGB */
      {
         if ( (*img)->depth == IPL_DEPTH_8U )
         {
            return IplImage2ImageRGB(img );
         }
         else
         {
            CVDL_ERROR(CV_StsError, "Unsupported format of IplImage. Image can't be converted.");            
         }
      }
      break;
      case 1 : /* Gray linear */
      {
         if ( (*img)->depth == IPL_DEPTH_16S )
         {
            return IplImage2Image16Linear(img);
         }
         else if ( (*img)->depth == IPL_DEPTH_8U )
         {
            return IplImage2Image8Linear(img);
         }
         else if ( (*img)->depth == IPL_DEPTH_32S )
         {
            return IplImage2Image32Linear(img);
         }
      }      
      default:
         CVDL_ERROR(CV_StsError, "Unsupported format of IplImage. Image can't be converted.");
      break;
   }
   
  

   __END__;
   return NULL;
}

extern "C" IplImage* __stdcall cvdlCloneIplImage( const IplImage* src )
{
   size_t      allocDataSize  = 0;
   IplImage*   iplImage       = NULL;

   CV_FUNCNAME( "cvdlCloneIplImage" );

   __BEGIN__;

   if( !CV_IS_IMAGE_HDR( src ))
       CVDL_ERROR( CV_StsBadArg, "Bad image header" );
   
   allocDataSize = sizeof(IplImage) + sizeof(_CVDL_ALLOC_DATA_);
   iplImage = (IplImage*)cvdlAllocPtr(allocDataSize);

   
   if ( allocDataSize > sizeof(IplImage) )
   {
      memset((void*)(iplImage+1),0,(size_t)((int)allocDataSize-(int)sizeof(IplImage)));
   } 
   memcpy( iplImage, src, sizeof(IplImage));
   iplImage->imageData = iplImage->imageDataOrigin = 0;
   iplImage->roi = NULL;

   if( src->roi )
   {
      iplImage->roi = cvdlCreateIplImageROI( src->roi->coi, src->roi->xOffset,
                 src->roi->yOffset, src->roi->width, src->roi->height );
   }

   if( src->imageData )
   {
      int size = src->imageSize;
      cvCreateData( iplImage );
      memcpy( iplImage->imageData, src->imageData, size );
   }
   __END__;


   return iplImage;
}

extern "C" IplImage* __stdcall cvdlCreateIplImageHeader( 
   int channels , 
   int , 
   int depth,
   char*,
   char*,
   int,
   int origin,
   int align,
   int width,
   int height,
   IplROI*,
   IplImage*,
   void*,
   IplTileInfo*)
{
   size_t      allocDataSize  = 0;
   IplImage*   iplImage       = NULL;
   IplImage*   retImage       = NULL;
   
   CV_FUNCNAME( "cvdlCreateIplImageHeader" );

   __BEGIN__;

   allocDataSize = sizeof(IplImage) + sizeof(_CVDL_ALLOC_DATA_);
   iplImage = (IplImage*)cvdlAllocPtr(allocDataSize);

   
   if ( allocDataSize > sizeof(IplImage) )
   {
      memset((void*)(iplImage+1),0,(size_t)((int)allocDataSize-(int)sizeof(IplImage)));
   }  
   retImage = cvInitImageHeader(iplImage, cvSize(width,height), depth, channels, origin, align );

   if (!retImage)
   {
      cvdlFreePtr((void*)iplImage);
      CVDL_ERROR(CV_HeaderIsNull,"Can't create IplImage Header.");
   }
   __END__;
   return retImage;
}

extern "C" void __stdcall cvdlCreateIplImageData( IplImage* iplImage , int , int )
{
   CV_FUNCNAME( "cvdlCreateIplImageData" );

   __BEGIN__;

   CV_CALL( iplImage->imageData = iplImage->imageDataOrigin = 
                        (char*)cvdlAllocPtr( (size_t)iplImage->imageSize ));

   __END__;
}

extern "C" IplROI* __stdcall cvdlCreateIplImageROI( int coi, int xOffset, int yOffset, int width, int height )
{
   IplROI* roi = NULL;

   CV_FUNCNAME( "cvdlCreateIplImageROI" );

   __BEGIN__;

   CV_CALL( roi = (IplROI*)cvdlAllocPtr( sizeof(IplROI)));

   roi->coi = coi;
   roi->xOffset = xOffset;
   roi->yOffset = yOffset;
   roi->width = width;
   roi->height = height;
     

   __END__;

   return roi;
}

extern "C" void __stdcall cvdlReleaseIplImageData( IplImage* iplImage , int section_deallocation )
{
   _CVDL_ALLOC_DATA_* cvdlAllocData = NULL;

    CV_FUNCNAME( "cvdlReleaseIplImageData" );

   __BEGIN__;
   
   if (!iplImage)
   {
      CVDL_ERROR(CV_StsNullPtr, "Null pointer to iplImage passed as a function argument.");
   }

   cvdlAllocData = (_CVDL_ALLOC_DATA_*)(iplImage+1);

   if (section_deallocation & IPL_IMAGE_DATA)
   {
      if ( (*cvdlAllocData).cvdl_alloc_type == DL_ALLOC_DATA )
      {
         iplImage->imageData = iplImage->imageDataOrigin = NULL;
         if ((*cvdlAllocData).data_ptr.image_stuct)
         {
            DeleteImage((*cvdlAllocData).data_ptr.image_stuct);
         }
         (*cvdlAllocData).cvdl_alloc_type = CV_ALLOC_DATA;
         (*cvdlAllocData).data_ptr.image_stuct = NULL;
      }
      else
      {
         char* data = iplImage->imageData;
      
         iplImage->imageData = iplImage->imageDataOrigin = NULL;
         if (data)
         {
            cvdlFreePtr((void*)data);
         }
      }
   }
   if (section_deallocation & IPL_IMAGE_ROI)
   {
      if ( iplImage->roi )
      {
         cvdlFreePtr( (void*)(iplImage->roi) );
         iplImage->roi = NULL;
      }
   }
   if (section_deallocation & IPL_IMAGE_HEADER)
   {
      cvdlFreePtr((void*)iplImage);
   }
   __END__;
}

