//-----------------------------------------------------------------------------
// imagetools.h
// Image representation and low level image manipulation
// Roman Juranek, DCGM FIT BUT, Brno
//-----------------------------------------------------------------------------


#ifndef _ImageTools_H_
#define _ImageTools_H_


#include <cmath>
#include <algorithm>


/// Size reprezentation.
struct TSize
{
	int w, h;
	TSize(){}
	TSize(int _w, int _h):w(_w), h(_h){}
	int area() const
	{
		return w * h;
	}
	bool operator==(const TSize & s) const
	{
		return (w == s.w) && (h == s.h);
	}
}; // TSize


/// Position representation.
struct TPoint
{
	int x, y;
	TPoint(){}
	TPoint(int _x, int _y):x(_x), y(_y){}
	TPoint(const TPoint & p):x(p.x), y(p.y){}
}; // TPoint


/// Rectangular area representation.
struct TRect
{
	int x; ///< Horizontal position
	int y; ///< Vertical position
	int w; ///< Width
	int h; ///< Height
	TRect(){}
	TRect(int _x, int _y, int _w, int _h):x(_x), y(_y), w(_w), h(_h){}
	TSize size() const
	{
		return TSize(w, h);
	}
	TPoint position() const
	{
		return TPoint(x, y);
	}
	TPoint center() const
	{
		return TPoint(x + w/2, y + h/2);
	}
}; // TRect


/// General image class.
/// \todo Upside down image representation for faster pixel access
template <class _T>
class TImage
{
public: // Types
	typedef _T      TImageData;    ///< Type for image data
	typedef _T    * TImageDataPtr; ///< Pointer to image data
	typedef char    TData;         ///< Raw data
	typedef TData * TDataPtr;      ///< General ptr

private: // Data members
	TDataPtr data; ///< Image data
	TSize sz;      ///< Image dimmensions
	int yofs;      ///< Offset to subsequent line (in Bytes!)
	bool dealloc;  ///< Delete data in destructor flag (false in reference images)

protected:
	/// \returns true if image is reference
	bool isReference() { return !dealloc; }
	
	/// \returns void ptr to data
	TDataPtr getData() { return data; }

public:
	/// Default constructor - does nothing.
	TImage()
		:data(0), sz(0,0), yofs(0), dealloc(false) {}

	/// Initialize new image of specified dimmensions.
	/// Does not clear the image!
	/// @param size Image width
	TImage(TSize size)
		:data(0), sz(size), yofs(sz.w * sizeof(TImageData))
	{
		if (dealloc && data)
			delete[] data;
		dealloc = true;
		data = new TData[sz.area() * sizeof(TImageData)];
	}

	/// Initialize reference image.
	/// @param d Allocated image data
	/// @param size Width of image
	/// @param yoffset Offset to pixel in next row
	TImage(TImageDataPtr d, TSize size, unsigned yoffset)
		:data(TDataPtr(d)), sz(size), yofs(yoffset), dealloc(false) {}

	/// Destructor.
	/// Deallocates data if dealloc==true
	~TImage()
	{
		if (dealloc && data)
			delete[] data;
		data = 0;
	}

	/// Creates new image instance as reference to original image.
	/// @param rect Area to make reference image
	/// \returns New TImage instance
	TImage<TImageData> subimage(TRect rect)
	{
		return TImage(TImageDataPtr(data + rect.y * yofs) + rect.x, rect.size(), yofs);
	}

	/// Creates new instance as reference of original image.
	/// \returns New reference image.
	TImage<TImageData> reference()
	{
		return TImage(TImageDataPtr(data), sz, yofs);
	}

	/// Linear image addressing.
	/// @param i Offset
	const TImageData & operator[] (unsigned i) const
	{
		return TImageDataPtr(data)[i];
	}

	/// Linear image addressing
	/// @param i Offset
	TImageData & operator[] (unsigned i)
	{
		return TImageDataPtr(data)[i];
	}

	/// Row adressing.
	/// @param r Row number
	/// \returns Pointer to row begining
	TImageDataPtr const row(unsigned r) const
	{
		return TImageDataPtr(data + r * yofs);
	}

	/// Row adressing.
	/// @param r Row number
	/// \returns Pointer to row begining
	TImageDataPtr row(unsigned r)
	{
		return TImageDataPtr(data + r * yofs);
	}

	/// Size of image.
	/// \returns Size of the image in pixels
	TSize size() const
	{
		return sz;
	}

	/// Y offset.
	/// \returns Address offset of subsequent rows.
	unsigned yOffset() const
	{
		return yofs;
	}
}; // TImage


// Various image types
typedef TImage<float> TImageFloat; ///< 32 bit FP image
typedef TImage<unsigned int> TImageUInt; ///< 32 bit integer image
typedef TImage<unsigned char> TImageUChar; ///< 8 bit integer image


/// Calculate scaling table.
/// Table could be used with remap image to perform resize operation.
/// @param scale Scale factor
/// @param size size of an array
/// e.g.: scale==1/1.5, size==8 returns {0, 1, 3, 4, 6, 7, 9, 10}, scale==1/2, size==8 returns {0, 2, 4, 6, 8, 10, 12, 14}
int * calcScaleTable(float scale, int size);


/// Map image rows and columns using tables.
/// This function could be useful for scale or flip image.
/// @param src Source image
/// @param dst Destination image
/// @param xtable Horizontal remap table.
/// @param ytable Vertical remap table.
/// User should provide correctly calculated tables. Size of horizontal/vertical remap table
/// should at least correspond to respective size of dst image.
/// Table examples:
/// 2x resize: xtable and ytable = {0,0,1,1,2,2,3,3, ...}
/// 1/2x resize: xtable and ytable = {0,2,4,6,8, ...}
/// hflip or vflip: xtable/ytable = {w-1, w-2, w-3, ..., 0}
/// Note: This function actually copies data so it could be slow when used in certain cases.
void remapImage(TImageUChar & src, TImageUChar & dst, int * xtable, int * ytable);


/// Integrate image.
/// Calculates two different integral images - sum of original values
/// and sum of its squares to accelerate calculation of subwindow standard
/// deviation.
/// @param image Input image
/// @param sum Integral image
/// @param sum2 Integral image
void integrate(const TImageUChar & image, TImageUInt & sum1, TImageUInt & sum2);


/// Separable convolution filter with integer kernel.
/// Conovolves image with kernel and store result in dstImage. If dstImage==0
/// result is stored in srcImage. This implementation DOES NOT handle borders!
/// @param srcImage Image to filter.
/// @param dstImage Result image.
/// @param kernel Horizontal convolution kernel. e.g.: k={1,2,1} for 3x3 gaussian.
/// @param size Size of kernel
/// @param scale Scale factor of horizontal kernel (4 for {1,2,1} kernel).
void convolve(TImageUChar * srcImage, TImageUChar * dstImage, int * kernel, int size, int scale);


class TFrame
{
	TImageUChar intensity;
	TImageUInt  integral1;
	TImageUInt  integral2;

public:
	// Constructor.
	// Allocates intensity and integral images.
	// @param size Size of frame.
	TFrame(TSize sz)
		:intensity(sz), integral1(sz), integral2(sz) {}

	~TFrame(){}
	
	/// insert new frame.
	/// Input image is resized to fit intensity.
	/// Integral images are calculated.
	void insert(TImageUChar & image, int * rescale)
	{
		remapImage(image, intensity, rescale, rescale);
		integrate(intensity, integral1, integral2);
	}

	TImageUChar & intensityImage()
	{
		return intensity;
	}
	TImageUInt & integralImage1()
	{
		return integral1;
	}
	TImageUInt & integralImage2()
	{
		return integral2;
	}
	TSize size()
	{
		return intensity.size();
	}
};


/// Sample image for feature response calculation.
/// Contains intensity image (reference to video frame)
/// and integral image calculated during constructor from intensity.
/// This class can be used for blob classification. It is not efficient for
/// general object detection though (because of calculation of integral image for
/// every sample separately)
class TSampleImage
{
	TImageUChar intensity; ///< Intensity image
	TImageUInt  integral; ///< Integral image (normal)
	float stddev; ///< Standard deviation of sample

	/// Integrate intensity image.
	/// intensiy -> integral
	/// As side effect calculates sum of pixel values and
	/// sum of squares of pixel values.
	/// @param sum Reference to store $sum x$
	/// @param sum2 Reference to store $sum x^{2}$
	void integrate(unsigned & sum, unsigned & sum2);

public:
	/// Initialize sample image from image area.
	/// Resize the area to specified size and integrate.
    TSampleImage(TImageUChar & image, TRect & area, TSize sampleSize)
		:intensity(sampleSize), integral(sampleSize)
	{
		unsigned sum1, sum2;
		sum1=sum2=0;
		TImageUChar tmpimage = image.subimage(area);

		int * scaleX = calcScaleTable(float(tmpimage.size().w) / intensity.size().w, intensity.size().w);
		int * scaleY = calcScaleTable(float(tmpimage.size().h) / intensity.size().h, intensity.size().h);

		remapImage(tmpimage, intensity, scaleX, scaleY);

		delete [] scaleX;
		delete [] scaleY;

		integrate(sum1, sum2);
		stddev = std::sqrt(float(sum2 - (sum1 * sum1)));
	}

	/// Initialize sample from TFrame area.
	/// Integral images are already calculated so just get standard deviation.
	TSampleImage(TFrame & frame, TRect & area)
	{
		intensity = frame.intensityImage().subimage(area);
		integral = frame.integralImage1().subimage(area);
		TImageUInt integral2 = frame.integralImage2().subimage(area);
		// calc stddev from integral1 and integral2
		float sum1 =
			float(integral.row(0)[0] +
			integral.row(integral.size().h-1)[integral.size().w-1] -
			integral.row(0)[integral.size().w-1] -
			integral.row(integral.size().h-1)[0])/integral.size().area();
		float sum2 =
			float(integral2.row(0)[0] +
			integral2.row(integral.size().h-1)[integral.size().w-1] -
			integral2.row(0)[integral.size().w-1] -
			integral2.row(integral.size().h-1)[0])/integral.size().area();
		if (sum2 < 1.0f)
		{
			stddev = 1.0f;
			return;
		}
		stddev = std::sqrt(sum2 - (sum1*sum1));
	}

	/// Get intensity image.
	/// \returns Reference to intensity image
	const TImageUInt & integralImage() const
	{
		return integral;
	}

	/// Get integral image.
	/// \returns Reference to integral image
	TImageUChar & intensityImage()
	{
		return intensity;
	}
	
	/// Standard deviation of intensity image.
	float stdDev() const
	{
		return stddev;
	}
}; // TSampleImage

#endif
