//---------------------------------------------------------------------------
// Feature interface and implementation
// Roman Juranek, UPGM FIT BUT, Brno
//
// - Continuous feature interface
// - Haar features
//---------------------------------------------------------------------------


#ifndef featuresH
#define featuresH


#include <cassert>
#include "imagetools.h"
#include "simplexml.h"


/*
/// Feature interface.
class TFeature
{
public:
    virtual ~TFeature() {}

    /// Load feature from XML structure.
    /// @param node Feature root node.
    virtual int loadFromXML(xmlNodePtr node) = 0;
}; // TFeature
*/

// Continuous feature interface
class TContFeature
{
public:
	virtual ~TContFeature() {};
    /// Evaluate feature on a sample.
    /// @param sample Sample to evaluate
    virtual float eval(const TSampleImage&) const = 0;
};


/// Haar Horizontal Double Feature.
//  A-----C-----E
//  |  -  |  +  |
//  B-----D-----F
class THaarDoubleHorizontal:public TContFeature
{
    int x, y, bw, bh;
    float blockSize;

public:
    THaarDoubleHorizontal(int posX, int posY, int width, int height)
		:TContFeature(), x(posX), y(posY), bw(width), bh(height), blockSize(float(width * height))
		{   }

    THaarDoubleHorizontal()
		:TContFeature(), x(0), y(0), bw(0), bh(0), blockSize(0)
		{   }

	~THaarDoubleHorizontal() {}

    float eval(const TSampleImage & sample) const
    {
		const TImageUInt & image = sample.integralImage();
		const char * base = (const char*)(image.row(y) + x);

		int yofs = bh * image.yOffset();
		int xofs = bw * sizeof(TImageUInt::TImageData);
		
		//// f = -A + B + 2C - 2D - E + F
		int f = 0;
		f += *(unsigned*)base + *(unsigned*)(base + yofs);
		base += xofs;
		f += 2 * (*(unsigned*)base - *(unsigned*)(base + yofs));
		base += xofs;
		f += *(unsigned*)(base + yofs) - *(unsigned*)base;

		return float(f) / (blockSize * sample.stdDev());
    }

    int loadFromXML(xmlNodePtr);   
}; // THaarDoubleHorizontal


/// Haar Vertical Double Feature.
//  A-----B
//  |  -  |
//  C-----D
//  |  +  |
//  E-----F
class THaarDoubleVertical : public TContFeature
{
    int x, y, bw, bh;
    float blockSize;

public:

    THaarDoubleVertical(int posX, int posY, int width=1, int height=1)
		:TContFeature(), x(posX), y(posY), bw(width), bh(height), blockSize(float(width * height))
		{    }

    THaarDoubleVertical()
		:TContFeature(), x(0), y(0), bw(0), bh(0), blockSize(0)
		{   }

    float eval(const TSampleImage & sample) const
    {
		const TImageUInt & image = sample.integralImage();
        const char * base = (const char*)(image.row(y)+x);

		int xofs = bw * sizeof(TImageUInt::TImageData);
		int yofs = bh * image.yOffset();

        // f = -A + B - 2C + 2D - E + F
        int f = 0;
        f += *(unsigned*)(base + xofs) - *(unsigned*)base;
        base += yofs;
        f += 2 * (*(unsigned*)base - *(unsigned*)(base + xofs));
        base += yofs;
        f += *(unsigned*)(base + xofs) - *(unsigned*)base;

        return f / (blockSize * sample.stdDev());
    }

    int loadFromXML(xmlNodePtr);
}; // THaarDoubleVertical

/// Haar Horizontal Triple Feature.
//  A-----C-----E-----G
//  |     |     |     |
//  B-----D-----F-----H
class THaarTripleHorizontal:public TContFeature
{
    int x, y, bw, bh;
    float blockSize;

public:
    THaarTripleHorizontal(int posX, int posY, int width=1, int height=1)
		:TContFeature(), x(posX), y(posY), bw(width), bh(height), blockSize(float(width * height))
		{    }

    THaarTripleHorizontal()
		:TContFeature(), x(0), y(0), bw(0), bh(0), blockSize(0)
		{   }

    float eval(const TSampleImage & sample) const
    {
        const TImageUInt & image = sample.integralImage();
        const char * base = (const char *)(image.row(y)+x);

		int xofs = bw * sizeof(TImageUInt::TImageData);
        int yofs = bh * image.yOffset();

        // f = A - B - 3C + 3D + 3E - 3F - G + H
        int f = 0;
        f += *(unsigned*)base - *(base + yofs);
        base += xofs;
        f += 3 * (*(unsigned*)(base + yofs) - *(unsigned*)base);
        base += xofs;
        f += 3 * (*(unsigned*)base - *(unsigned*)(base + yofs));
        base += xofs;
        f += *(unsigned*)(base + yofs) - *(unsigned*)base;

        return f / (blockSize * sample.stdDev());
    }

    int loadFromXML(xmlNodePtr);
}; // THaarTripleHorizontal

/// Haar Vertical Triple Feature.
//  A-----B
//  |     |
//  C-----D
//  |     |
//  E-----F
//  |     |
//  G-----H
class THaarTripleVertical:public TContFeature
{
    int x, y, bw, bh;
    float blockSize;

public:

    THaarTripleVertical(int posX, int posY, int width=1, int height=1)
		:TContFeature(), x(posX), y(posY), bw(width), bh(height), blockSize(float(width * height))
		{    }

    THaarTripleVertical()
		:TContFeature(), x(0), y(0), bw(0), bh(0), blockSize(0)
		{   }

    float eval(const TSampleImage & sample) const
    {
        const TImageUInt & image = sample.integralImage();
        const char * base = (const char*)(image.row(y)+x);

		int xofs = bw * sizeof(TImageUInt::TImageData);
        int yofs = bh * image.yOffset();

        // f = A - B - 3C + 3D + 3E - 3F - G + H
        int f = 0;
        f += *(unsigned*)base - *(unsigned*)(base + xofs);
        base += yofs;
        f += 3 * (*(unsigned*)(base + xofs) - *(unsigned*)base);
        base += yofs;
        f += 3 * (*(unsigned*)base - *(unsigned*)(base + xofs));
        base += yofs;
        f += *(unsigned*)(base + xofs) - *(unsigned*)base;

        return f / (blockSize * sample.stdDev());
    }

    int loadFromXML(xmlNodePtr);
}; // THaarTripleVertical

TContFeature * createContFeature(const xmlNodePtr);

#endif
