//---------------------------------------------------------------------------
// classifier.h
// Weak and strong classifiers interface
//---------------------------------------------------------------------------

#ifndef classifierH
#define classifierH


#include <vector>
#include <list>
#include <utility>
#include <libxml/tree.h>
#include "imagetools.h"
#include "pyramid.h"
#include "features.h"
#include "simplexml.h"


/// Decision stump continuous weak classifier
/// One threshold decision
class TDecisionStump
{
    TContFeature * feature;
    float alpha;     ///< Weight of classifier
    float threshold; ///< AdaBoost threshold
    float parity;    ///< Classifier parity
    
public:
    /// New weak classifier
    TDecisionStump(TContFeature * f, float a=1.0f, float t=0.0f, float p=1)
		:feature(f), alpha(a), threshold(t), parity(p)
		{   }

    TDecisionStump()
		:feature(0), alpha(0), threshold(0), parity(0)
		{   }

    ~TDecisionStump()
    {
        if (feature) delete feature;
    }

    /// Evaluate weak classifier on sample.
    float eval(const TSampleImage & sample) const
    {
        if (feature->eval(sample) >= threshold)
            return parity * alpha;
        else
            return -parity * alpha;
    }

    /// Load weak classifier from XML structure.
    /// @param wClassifier Root XML node of a classifier
    int loadFromXML(const xmlNodePtr);
}; // THaarWeakClassifier


///////////////////////////////////////////////////////////////////////////////
/// Strong classifeir.
class TClassifier
{
    /// Vector of weak hypotheses
	typedef std::vector<TDecisionStump*> StageList;
	/// Vector of positive and negative waldboost thresholds
    typedef std::vector< std::pair<float, float> > ThrList;
    
    StageList wbStage;   ///< Weak classifeir list
    ThrList wbThreshold; ///< WaldBoost thresholds for each stage
    float T;             ///< Final threshold (usually 0.0f)
	TSize size;          ///< Classifier window size
	int shiftX, shiftY;  ///< Shift of window when scanning an image
    bool status;         ///< Flag that the instance is initialized correctly

public:
    TClassifier(const xmlNodePtr root = 0)
		:T(0.0), size(TSize(0,0)), shiftX(2), shiftY(2), status(false) // default params
        {
            loadFromXML(root);
        }

    ~TClassifier()
    {	}

    /// Evaluate the classifier on a sample
    int evalSample(const TSampleImage & sample);

	/// Scan frame with sliding window.
	int scanFrame(TFrame & frame, std::vector<TRect> & detections, unsigned first, float scale);

    /// Load classifier from XML structure.
    /// @param root Root node of classifier structure
    int loadFromXML(const xmlNodePtr root);

    // Get/Set functions

    float & threshold() { return T; }

	int & shiftx() { return shiftX; } 
	int & shifty() { return shiftY; }

	TSize windowSize() { return size; }

	std::size_t length() { return wbStage.size(); }
   
    bool ok()
    {
        return status;
    }
}; // TClassifier


/// Load classifier from XML and create new instance.
TClassifier * loadStrongClassifier(const std::string filename);


/// Scans all images in pyramid with given classifier.
/// @param pyramid Image pyramid to scan.
/// @param classifier Classifier that will perform scan
/// @param detections Vector of areas with positive classifier response
/// \returns Number of detections
int scanImagePyramid(TImagePyramid * pyramid, TClassifier * classifier, std::vector<TRect> & detections);


/// Get average number of features evaluated during last video frame scan.
/// Valid for last call of scanImageFrame;
float getAverageFeaturesPerSample();

unsigned getSamplesCount();

#endif
