/*
 *   lrd_engine.cpp
 *   Local Rank Difference evaluation
 *   $Id: lrd_engine.h 9 2008-05-02 08:14:34Z andyman $
 *   ----
 *   Department of Computer Graphics and Multimedia
 *   Faculty of Information Technology, Brno University of Technology
 *   ----
 *   Naive LRD evaluation engine.
 */


#ifndef __LRD_ENGINE_H__
#define __LRD_ENGINE_H__

#include <cxcore.h>

/// A detection structure.
/// Holds informations about a detection - position, size and response value.
struct TDetection
{
    float response; ///< Response of classifier
    CvRect area; ///< Position and size
};


/// List (actually a vector) of deteections.
/// Only first 'count' of detections is valid.
/// \todo consider use of CvSeq
struct TDetectionList
{
    unsigned count; ///< Actual count of detections
    unsigned maxCount; ///< Maximum count
    TDetection * detections; ///< Allocated detections
    
	/// Constructor - allocates max number of detections.
    TDetectionList(unsigned max)
		:count(0), maxCount(max)
	{
		detections = new TDetection[max];
	}

	/// Deletes the structure.
    ~TDetectionList()
	{
		delete [] detections;
	}

	/// Access to i-th detections.
    TDetection & operator[](unsigned i)
	{
		return detections[i];
	}
};


/// A single weak hypothesis.
/// One LRD feature and threshold
//   <-w->
// ^ +---+---+---+
// h | 0 | 1 | 2 |
// V +---+---+---+
//   | 3 | 4 | 5 |
//   +---+---+---+
//   | 6 | 7 | 8 |
//   +---+---+---+
struct TStage
{
	// Feature and hypothesis parameters
    unsigned x, y; // Position in sample [px]
	unsigned w, h; // Size of one feature block [px]
	unsigned A, B; // Indexes of Rank blocks
	float theta_b; // wald_boost threshold

	// Following values are calculated at runtime
	int conv; // Not used!
	int code; // Not used!
	float * alpha; // Ptr to table with alphas - assigned at runtime
    unsigned offset; // Feature offset relative to a sample
};


/// Classifier structure.
/// Holds informations about a classifier - parameters,
/// weak hypotheses and alphas.
struct TClassifier
{
    unsigned stageCount; ///< Number of stages
    unsigned alphaCount; ///< Alphas per stage
    float threshold; ///< Final classification threshold
    unsigned width; ///< Width of scan window
	unsigned height; ///< Height of scan window
    TStage * stage; ///< List of stages
    float * alpha; ///< List of alphas
};


/// Initialize a classifier.
/// Must be called before the classifier is used!!
/// @param classifier The classifier to process
void initClassifier(TClassifier * classifier);

/// Recalculates feature offset for specified image size.
/// The function must be called whenever size of input image is changed!
/// This allows passing feature offsets directly to evaluation instead of
/// coordinates - performance gain.
/// @param classifier The classifier to process
/// @param widthStep Address offset between subsequent rows (IplImgae::widthStep)
void setClassifierImageSize(TClassifier * classifier, unsigned widthStep);

/// Evaluates a classifier of specified position in image.
/// Evaluates classifier on image area defined by x,y and classifier size.
/// @param image Input intensity image (IPL_DEPTH_8U, single chanel)
/// @param classifier Classifier to evaluate
/// @param x Horizontal position of top-left corner of the area
/// @param y Vertical position of top-left corner of the area
/// @param response Response of the classifier
/// @returns 1 for positive, 0 for negative classification
int evalLRDClassifier(IplImage * image, unsigned smpOffset, TClassifier * classifier, float * response);

/// Scan entire image with a classifier.
/// @param image Input intensity image (IPL_DEPTH_8U, single chanel)
/// @param classifier Classifier to evaluate
/// @param results Preallocated vector for results
/// @param first Position from which starts storage in the results
/// @returns Number of detections
unsigned scanImage(IplImage * image, TClassifier * classifier, TDetectionList * results, unsigned first);


#endif
