/*
  Ruzicka's formal model core
  Copyright (C) 2007 Jaroslav Skarvada <skarvada@fit.vutbr.cz>  
  
  This program is free software: you can redistribute it and/or modify  
  it under the terms of the GNU General Public License as published by  
  the Free Software Foundation, either version 3 of the License, or  
  (at your option) any later version.  
  
  This program is distributed in the hope that it will be useful,  
  but WITHOUT ANY WARRANTY; without even the implied warranty of  
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the  
  GNU General Public License for more details.  
 
  You should have received a copy of the GNU General Public License  
  along with this program.  If not, see <http://www.gnu.org/licenses/>.  
*/

#ifndef icstdio
#include <cstdio>
#define icstdio
#endif

#ifndef imap
#include <map>
#define imap
#endif

#ifndef iset
#include <set>
#define iset
#endif

#ifndef ivector
#include <vector>
#define ivector
#endif

#ifndef iiostream
#include <iostream>
#define iiostream
#endif

#ifndef ifstream
#include <fstream>
#define ifstream
#endif

#ifndef istring
#include <string>
#define istring
#endif

#ifndef ierror
#include "error.h"
#define ierror
#endif

#ifndef iruzcore
#include "ruzcore.h"
#define iruzcore
#endif

using namespace std;

void
tuua::clear (void)
{
  name.erase ();
  E.clear ();
  INP.clear ();
  OUTP.clear ();
  CI.clear ();
  PI.clear ();
  PO.clear ();
  P.clear ();
  R.clear ();
  INV.clear ();
  SCAN.clear ();
  C.clear ();
  M.clear ();
  I.clear ();
  EG.clear ();
  PG.clear ();
  CG.clear ();
  RO.clear ();
  MI.clear ();
  DELTA.clear ();
  V.clear ();
  TDR.clear ();
  TRV.clear ();
  TIR.clear ();
  TOR.clear ();
  PSI.clear ();
  ELN.clear ();
  ELT.clear ();
  GN.clear ();
  WIDTH.clear ();
}

bool
tuua::ipt (tid from, tid to)
{
  return (I.find (tpair (from, to)) != I.end ());
}

// makes some simple validity tests 
// so far only non empty sets test and IDX positive test
terr
tuua::check_validity (void)
{
  telset::iterator it;

  if (INP.empty ())
    return e_noin;
  if (OUTP.empty ())
    return e_noout;
  if (C.empty ())
    return e_nocon;
  if (R.empty ())
    return e_noreg;
  for (it = E.begin (); it != E.end (); it++)
    if (*it <= 0)
      return e_invelidx;
  for (it = P.begin (); it != P.end (); it++)
    if (*it <= 0)
      return e_invgateidx;

  return e_ok;
}

// check for unbound inputs from INPUT set, can be self.CI or self.INP
tid
tuua::check_ipunbound (telset & INPUT)
{
  bool found;
  tpair pair;
  telset Q;

  Q.clear ();
  Q.insert (OUTP.begin (), OUTP.end ());
  Q.insert (CI.begin (), CI.end ());
  Q.insert (PI.begin (), PI.end ());
  for (telset::iterator it_in = INPUT.begin (); it_in != INPUT.end ();
       it_in++)
    {
      found = false;
      for (telset::iterator it_out = Q.begin ();
	   (it_out != Q.end ()) && !found; it_out++)
	{
	  pair.f = *it_out;
	  pair.s = *it_in;
	  if ((C.find (pair) != C.end ()) && (*it_in != *it_out))
	    found = true;
	}
      if (!found)
	return *it_in;
    }
  return 0;
}

// checks for unbound outputs
tid
tuua::check_opunbound (void)
{
  bool found;
  tpair pair;
  telset Q;

  Q.clear ();
  Q.insert (INP.begin (), INP.end ());
  Q.insert (CI.begin (), CI.end ());
  Q.insert (PO.begin (), PO.end ());
  for (telset::iterator it_out = OUTP.begin (); it_out != OUTP.end ();
       it_out++)
    {
      found = false;
      for (telset::iterator it_in = Q.begin (); (it_in != Q.end ()) && !found;
	   it_in++)
	{
	  pair.f = *it_out;
	  pair.s = *it_in;
	  if ((C.find (pair) != C.end ()) && (*it_in != *it_out))
	    found = true;
	}
      if (!found)
	return *it_out;
    }
  return 0;
}

// extracts one char <char> from stream <f>
char
tuua::get_char (istream & f)
{
  bool char_read;
  char c;
  ios::fmtflags flags;

  char_read = false;
  flags = f.flags ();
  while (f && !char_read)
    {
      f.flags (flags | ios::skipws);
      f >> c;
      if (c == c_comment)
	{
	  f.flags (flags & ~ios::skipws);
	  while (f && (c != '\n'))
	    f >> c;
	}
      else
	char_read = true;
    }

  if (!f)
    c = c_eof;

  return c;
}

// tries to skip char <chr> in stream <f>, returns true on success
bool
tuua::skip_char (istream & f, char chr)
{
  char c;
  if ((c = get_char (f)) == chr)
    return true;
  else
    {
      f.putback (c);
      return false;
    }
}

// gets token <token> from stream <f>
bool
tuua::get_token (istream & f, string & token)
{
  char c;
  ios::fmtflags flags;

  token.erase ();
  flags = f.flags ();
  c = get_char (f);
  if (!f || (c == c_eof) || (string (seps).find (c) != string::npos))
    {
      f.putback (c);
      return false;
    }
  f.flags (flags & ~ios::skipws);
  while (f && (string (seps).find (c) == string::npos))
    {
      token += c;
      f >> c;
    }
  if (f)
    f.putback (c);
  f.flags (flags | ios::skipws);
  return f;
}

// gets string <str> from stream <f> and removes quoting 
bool
tuua::get_string (istream & f, string & str)
{
  char c;
  ios::fmtflags flags;

  str.erase ();
  flags = f.flags ();
  c = get_char (f);
  if (!f || (c == c_eof) || (string (seps).find (c) != string::npos)
      || (c != c_string))
    return false;
  f.flags (flags & ~ios::skipws);
  f >> c;
  while (f)
    {
      if (c == c_string)
	{
	  f >> c;
	  if (!f || (c != c_string))
	    {
	      f.putback (c);
	      f.flags (flags | ios::skipws);
	      return true;
	    }
	  else
	    str += c;
	}
      else
	str += c;
      f >> c;
    }
  f.flags (flags | ios::skipws);
  return f;
}

// gets elset <elset> from stream <f> 
bool
tuua::get_elset (istream & f, telset & elset)
{
  tid id;
  string str;

  elset.clear ();
  str.erase ();
  while (f && get_token (f, str) && (sscanf (str.c_str (), "%d", &id) == 1)
	 && (id > 0) && skip_char (f, ','))
    elset.insert (id);
  return (get_char (f) == '.');
}

// gets sequence <sequence> from stream <f> 
bool
tuua::get_sequence (istream & f, tsequence & sequence)
{
  tid id;
  string str;

  sequence.clear ();
  str.erase ();
  while (f && get_token (f, str) && (sscanf (str.c_str (), "%d", &id) == 1)
	 && (id > 0) && skip_char (f, ','))
    sequence.push_back (id);
  return (get_char (f) == '.');
}

// gets gnames <gnames> from istream <f>
bool
tuua::get_gnames (istream & f, tgnames & gnames)
{
  tid id;
  string str, str2;

  gnames.clear ();
  str.erase ();
  str2.erase ();
  while (f && get_token (f, str) && (sscanf (str.c_str (), "%d", &id) == 1)
	 && (id > 0) && skip_char (f, '-') && get_string (f, str2)
	 && skip_char (f, ';'))
    gnames.insert (tgnames::value_type (id, str2));
  return (get_char (f) == '.');
}

// gets pairstrmap <pm> from istream <f>
bool
tuua::get_pairstrmap (istream & f, tpairstrmap & pm)
{
  tid id1, id2;
  string str, str2, str3;

  pm.clear ();
  str.erase ();
  str2.erase ();
  str3.erase ();

  while (f && get_token (f, str) && (sscanf (str.c_str (), "%d", &id1) == 1)
	 && (id1 > 0) && skip_char (f, ',') && get_token (f, str2)
	 && (sscanf (str2.c_str (), "%d", &id2) == 1) && (id2 > 0)
	 && skip_char (f, '-') && get_string (f, str3) && skip_char (f, ';'))
    pm.insert (tpairstrmap::value_type (tpair (id1, id2), str3));
  return (get_char (f) == '.');
}

// gets idsetmap <idsetmap> from istream <f>
bool
tuua::get_idsetmap (istream & f, tidsetmap & idsetmap)
{
  tid id;
  string str;
  telset elset;

  idsetmap.clear ();
  str.erase ();
  while (f && get_token (f, str) && (sscanf (str.c_str (), "%d", &id) == 1)
	 && (id > 0) && skip_char (f, '-') && get_elset (f, elset)
	 && skip_char (f, ';'))
    idsetmap.insert (tidsetmap::value_type (id, elset));
  return (get_char (f) == '.');
}

// gets pairset <pairset> from istream <f>
bool
tuua::get_pairset (istream & f, tpairset & pairset)
{
  tid id1, id2;
  string str, str2;

  pairset.clear ();
  str.erase ();
  str2.erase ();
  while (f && get_token (f, str) && (sscanf (str.c_str (), "%d", &id1) == 1)
	 && (id1 > 0) && skip_char (f, ',') && get_token (f, str2)
	 && (sscanf (str2.c_str (), "%d", &id2) == 1) && (id2 > 0)
	 && skip_char (f, ';'))
    pairset.insert (tpair (id1, id2));
  return (get_char (f) == '.');
}

// gets ro <ro> from istream <f>
bool
tuua::get_ro (istream & f, tro & ro)
{
  tid id1, id2;
  string str, str2;
  tsequence sequence;

  ro.clear ();
  str.erase ();
  str2.erase ();
  while (f && get_token (f, str) && (sscanf (str.c_str (), "%d", &id1) == 1)
	 && (id1 > 0) && skip_char (f, ',') && get_token (f, str2)
	 && (sscanf (str2.c_str (), "%d", &id2) == 1) && (id2 > 0)
	 && skip_char (f, '-') && get_sequence (f, sequence)
	 && skip_char (f, ';'))
    ro.insert (tro::value_type (tpair (id1, id2), sequence));
  return (get_char (f) == '.');
}

// gets delta <delta> from istream <f>
bool
tuua::get_delta (istream & f, tdelta & delta)
{
  bool err;
  int x;
  tid id1, id2;
  string str, str2, str3;
  tsequence sequence;

  err = false;
  delta.clear ();
  str.erase ();
  str2.erase ();
  str3.erase ();
  while (f && !err && get_token (f, str)
	 && (sscanf (str.c_str (), "%d", &id1) == 1) && (id1 > 0)
	 && skip_char (f, ',') && get_token (f, str2)
	 && (sscanf (str2.c_str (), "%d", &id2) == 1) && (id2 > 0)
	 && skip_char (f, '-') && get_token (f, str3) && skip_char (f, ';'))
    {
      if (str3 == "i")
	x = INT_MAX;
      else if ((sscanf (str3.c_str (), "%d", &x) != 1) || (x < 0))
	err = true;
      if (!err)
	delta.insert (tdelta::value_type (tpair (id1, id2), x));
    }

  return (!err) && (get_char (f) == '.');
}

// gets miel <miel> from istream <f>
bool
tuua::get_miel (istream & f, tmiel & miel)
{
  tid id;
  string str, str2;

  miel.clear ();
  str.erase ();
  str2.erase ();
  while (f && get_token (f, str) && (sscanf (str.c_str (), "%d", &id) == 1)
	 && (id > 0) && skip_char (f, '=') && get_string (f, str2)
	 && skip_char (f, ','))
    miel.insert (tmiel::value_type (id, str2));
  return (get_char (f) == '.');
}

// gets mi <mi> from istream <f>
bool
tuua::get_mi (istream & f, tmi & mi)
{
  tid id1, id2;
  string str, str2;
  tmiel miel;

  mi.clear ();
  str.erase ();
  str2.erase ();
  while (f && get_token (f, str) && (sscanf (str.c_str (), "%d", &id1) == 1)
	 && (id1 > 0) && skip_char (f, ',') && get_token (f, str2)
	 && (sscanf (str2.c_str (), "%d", &id2) == 1) && (id2 > 0)
	 && skip_char (f, '-') && get_miel (f, miel) && skip_char (f, ';'))
    mi.insert (tmi::value_type (tpair (id1, id2), miel));
  return (get_char (f) == '.');
}

// skips unknown set in istream <f>
void
tuua::skip_set (fstream & f)
{
  bool quit;
  char c;

  quit = false;
  while (!quit && (f >> c))
    {
      if (c == '$')
	{
	  f >> c;
	  if (!f)
	    quit = true;
	  else if (c != '$')
	    {
	      f.putback (c);
	      f.putback ('$');
	      quit = true;
	    }
	}
    }
}

// gets delta <delta> from istream <f>
bool
tuua::get_width (istream & f, tidintmap & width)
{
  bool err;
  tid id, w;
  string str, str2;

  err = false;
  width.clear ();
  str.erase ();
  str2.erase ();
  while (f && !err && get_token (f, str)
	 && (sscanf (str.c_str (), "%d", &id) == 1) && (id > 0)
	 && skip_char (f, '-') && get_token (f, str2)
	 && (sscanf (str2.c_str (), "%d", &w) == 1) && (w > 0)
	 && skip_char (f, ';'))
    width.insert (tidintmap::value_type (id, w));

  return (!err) && (get_char (f) == '.');
}

// formats string <str> for output as string
string
tuua::format_str (string str)
{
  unsigned int x;
  string str2;

  str2 = "\"";
  for (x = 0; x < str.length (); x++)
    {
      if (str[x] == '"')
	str2 += "\"";
      str2 += str[x];
    }
  str2 += "\"";
  return str2;
}

// prints elset <elset> to ostream <f>
void
tuua::print_elset (ostream & f, telset & elset)
{
  telset::iterator it;

  for (it = elset.begin (); it != elset.end (); it++)
    f << (int) *it << ",";
  f << ".";
}

// prints gnames <gnames> to ostream <f>
void
tuua::print_gnames (ostream & f, tgnames & gnames)
{
  tgnames::iterator it;

  for (it = gnames.begin (); it != gnames.end (); it++)
    f << (int) (*it).first << "-" << format_str ((*it).second) << ";";
  f << ".";
}

// prints idsetmap <idsetmap> to ostream <f>
void
tuua::print_idsetmap (ostream & f, tidsetmap & idsetmap)
{
  tidsetmap::iterator it;

  for (it = idsetmap.begin (); it != idsetmap.end (); it++)
    {
      f << (int) (*it).first << "-";
      print_elset (f, (*it).second);
      f << ";";
    }
  f << ".";
}

// prints pairset <pairset> to ostream <f>
void
tuua::print_pairset (ostream & f, tpairset & pairset)
{
  tpairset::iterator it;

  for (it = pairset.begin (); it != pairset.end (); it++)
    f << (int) (*it).f << "," << (int) (*it).s << ";";
  f << ".";
}

// prints pairset <pairset> to ostream <f>
void
tuua::print_pairstrmap (ostream & f, tpairstrmap & pm)
{
  tpairstrmap::iterator it;

  for (it = pm.begin (); it != pm.end (); it++)
    f << (int) (*it).first.f << "," << (int) (*it).first.s << "-" <<
      format_str ((*it).second) << ";";
  f << ".";
}

// prints ro <ro> to ostream <f>
void
tuua::print_ro (ostream & f, tro & ro)
{
  tro::iterator it_ro;
  tsequence::iterator it_seq;

  for (it_ro = ro.begin (); it_ro != ro.end (); it_ro++)
    {
      f << (int) (*it_ro).first.f << "," << (int) (*it_ro).first.s << "-";
      for (it_seq = (*it_ro).second.begin ();
	   it_seq != (*it_ro).second.end (); it_seq++)
	f << (int) *it_seq << ",";
      f << ".;";
    }
  f << ".";
}

// prints delta <delta> to ostream <f>
void
tuua::print_delta (ostream & f, tdelta & delta)
{
  tdelta::iterator it;

  for (it = delta.begin (); it != delta.end (); it++)
    {
      f << (int) (*it).first.f << "," << (int) (*it).first.s << "-";
      if ((*it).second == INT_MAX)
	f << "i";
      else
	f << (int) (*it).second;
      f << ";";
    }
  f << ".";
}

// prints mi <mi> to ostream <f>
void
tuua::print_mi (ostream & f, tmi & mi)
{
  tmi::iterator it_mi;
  tmiel::iterator it_miel;

  for (it_mi = mi.begin (); it_mi != mi.end (); it_mi++)
    {
      f << (int) (*it_mi).first.f << "," << (int) (*it_mi).first.s << "-";
      for (it_miel = (*it_mi).second.begin ();
	   it_miel != (*it_mi).second.end (); it_miel++)
	f << (int) (*it_miel).f << "=" << format_str ((*it_miel).s) << ",";
      f << ".;";
    }
  f << ".";
}

// prints width <width> to ostream <f>
void
tuua::print_width (ostream & f, tidintmap & width)
{
  tidintmap::iterator it;

  for (it = width.begin (); it != width.end (); it++)
    f << (int) (*it).first << "-" << (int) (*it).second << ";";
  f << ".";
}

// prints UUA formal model to ostream <f>
void
tuua::print_uua (ostream & f)
{
  f << "# Ruzicka's RTL formal model description file" << endl;
  f << "# Version 1.0" << endl << endl;
  f << "$NAME: " << format_str (name) << endl << endl;

  f << "$E: ";
  print_elset (f, E);
  f << endl << endl;

  f << "$R: ";
  print_elset (f, R);
  f << endl << endl;

  f << "$INV: ";
  print_elset (f, INV);
  f << endl << endl;

  f << "$ELN: ";
  print_gnames (f, ELN);
  f << endl << endl;

  f << "$ELT: ";
  print_gnames (f, ELT);
  f << endl << endl;

  f << "$PI: ";
  print_elset (f, PI);
  f << endl << endl;

  f << "$PO: ";
  print_elset (f, PO);
  f << endl << endl;

  f << "$INP: ";
  print_elset (f, INP);
  f << endl << endl;

  f << "$OUTP: ";
  print_elset (f, OUTP);
  f << endl << endl;

  f << "$GN: ";
  print_gnames (f, GN);
  f << endl << endl;

  f << "$WIDTH: ";
  print_width (f, WIDTH);
  f << endl << endl;

  f << "$PSI: ";
  print_idsetmap (f, PSI);
  f << endl << endl;

  f << "$CI: ";
  print_elset (f, CI);
  f << endl << endl;

  f << "$P: ";
  print_elset (f, P);
  f << endl << endl;

  f << "$V: ";
  print_idsetmap (f, V);
  f << endl << endl;

  f << "$MI: ";
  print_mi (f, MI);
  f << endl << endl;

  f << "$C: ";
  print_pairset (f, C);
  f << endl << endl;

  f << "$M: ";
  print_pairset (f, M);
  f << endl << endl;

  f << "$I: ";
  print_pairset (f, I);
  f << endl << endl;

  f << "$II: ";
  print_pairset (f, II);
  f << endl << endl;

  f << "$EG: ";
  print_gnames (f, EG);
  f << endl << endl;

  f << "$PG: ";
  print_gnames (f, PG);
  f << endl << endl;

  f << "$CG: ";
  print_pairstrmap (f, CG);
  f << endl << endl;

  f << "$RO: ";
  print_ro (f, RO);
  f << endl << endl;

  f << "$DELTA: ";
  print_delta (f, DELTA);
  f << endl << endl;

  f << "$TDR: ";
  print_idsetmap (f, TDR);
  f << endl << endl;

  f << "$TRV: ";
  print_idsetmap (f, TRV);
  f << endl << endl;

  f << "$TIR: ";
  print_idsetmap (f, TIR);
  f << endl << endl;

  f << "$TOR: ";
  print_idsetmap (f, TOR);
  f << endl << endl;

  f << "$SCAN: ";
  print_elset (f, SCAN);
  f << endl << endl;
}

// write formal model to a file
terr
tuua::WriteFormalUUA (string ruzname)
{
  int err;
  fstream f;

  err = e_ok;
  f.open (ruzname.c_str (), ios::out);
  if (!f)
    return e_fcreate;
  print_uua (f);
  if (!f)
    err = e_fwrite;
  f.close ();

  return err;
}

// UUA file parser
terr
tuua::LoadFormalUUA (string fname)
{
  bool eread, namef, e, r, eln, elt, pi, po, inp, outp, gn, width, psi, ci, p,
    v, mi, c, m, i;
  bool eg, pg, cg, ro, delta, tdr, trv, tir, tor, scan, inv, ii;
  char chr;
  string str;
  fstream f;

  eread = namef = e = r = eln = elt = pi = po = inp = outp = gn = width =
    psi = ci = p = v = mi = c = m = i = false;
  eg = pg = cg = ro = delta = tdr = trv = tir = tor = scan = inv = ii = false;
  f.open (fname.c_str (), ios::in);
  if (!f)
    return e_fopen;
  while (!eread && f)
    {
      chr = get_char (f);
      if (f)
	{
	  if (chr != '$')
	    eread = true;
	  else
	    {
	      if (!get_token (f, str) || !skip_char (f, ':'))
		eread = true;
	      else
		{
		  cout << "parsing token: " << str << endl;
		  if (str == "NAME")
		    eread = !(namef = get_string (f, name));
		  else if (str == "E")
		    eread = !(e = get_elset (f, E));
		  else if (str == "R")
		    eread = !(r = get_elset (f, R));
		  else if (str == "INV")
		    eread = !(inv = get_elset (f, INV));
		  else if (str == "ELN")
		    eread = !(eln = get_gnames (f, ELN));
		  else if (str == "ELT")
		    eread = !(elt = get_gnames (f, ELT));
		  else if (str == "PI")
		    eread = !(pi = get_elset (f, PI));
		  else if (str == "PO")
		    eread = !(po = get_elset (f, PO));
		  else if (str == "INP")
		    eread = !(inp = get_elset (f, INP));
		  else if (str == "OUTP")
		    eread = !(outp = get_elset (f, OUTP));
		  else if (str == "GN")
		    eread = !(gn = get_gnames (f, GN));
		  else if (str == "WIDTH")
		    eread = !(width = get_width (f, WIDTH));
		  else if (str == "PSI")
		    eread = !(psi = get_idsetmap (f, PSI));
		  else if (str == "CI")
		    eread = !(ci = get_elset (f, CI));
		  else if (str == "P")
		    eread = !(p = get_elset (f, P));
		  else if (str == "V")
		    eread = !(v = get_idsetmap (f, V));
		  else if (str == "MI")
		    eread = !(mi = get_mi (f, MI));
		  else if (str == "C")
		    eread = !(c = get_pairset (f, C));
		  else if (str == "M")
		    eread = !(m = get_pairset (f, M));
		  else if (str == "I")
		    eread = !(i = get_pairset (f, I));
		  else if (str == "II")
		    eread = !(ii = get_pairset (f, II));
		  else if (str == "EG")
		    eread = !(eg = get_gnames (f, EG));
		  else if (str == "PG")
		    eread = !(pg = get_gnames (f, PG));
		  else if (str == "CG")
		    eread = !(cg = get_pairstrmap (f, CG));
		  else if (str == "RO")
		    eread = !(ro = get_ro (f, RO));
		  else if (str == "DELTA")
		    eread = !(delta = get_delta (f, DELTA));
		  else if (str == "TDR")
		    eread = !(tdr = get_idsetmap (f, TDR));
		  else if (str == "TRV")
		    eread = !(trv = get_idsetmap (f, TRV));
		  else if (str == "TIR")
		    eread = !(tir = get_idsetmap (f, TIR));
		  else if (str == "TOR")
		    eread = !(tor = get_idsetmap (f, TOR));
		  else if (str == "SCAN")
		    eread = !(scan = get_elset (f, SCAN));
		  else
		    skip_set (f);
		}
	    }
	}
    }
  f.close ();

  if (eread)
    return e_fformat;
  if (!namef || !e || !r || !eln || !elt || !pi || !po || !inp || !outp || !gn
      || !width || !psi || !ci || !p || !v || !mi || !c || !m || !i || !eg
      || !pg || !cg || !ro || !delta || !tdr || !trv || !tir || !tor || !scan
      || !inv || !ii)
    return e_incomplete;
  return e_ok;
}
