/*
 * File:     squares.c, compiled by gcc
 * Date:     2009-02-02 21:03:45 
 * Author:   Bronislav Pribyl, xpriby12@stud.fit.vutbr.cz
 * Project:  Separable image resampling
 * Encoding: utf-8
 * Version:  1.5
 * Desc:     Module to handle coordinates for interpolation algorithm
 */

#include "squares.h"
#include "bit-handling.h"
#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>


/**
 * Initializes structure of coefficients of one square.
 * @param coords Pointer to structure of coefficients of one square
 * @param fileName Name of text file with square coefficients
 */
void coordsInitialize(TCoords *coords, char *fileName)
{
	FILE *f;
	int c;
	
	//try to open file
	if((f = fopen(fileName, "r")) == NULL){
		fprintf(stderr, "Nelze otevrit soubor %s.\n", fileName);
		exit(EXIT_FAILURE);
	}
	
	//load square size
	if(fscanf(f, "square-size:%d", &(coords->sqSize)) != 1){
		fprintf(stderr, "Soubor %s neobsahuje korektni specifikaci velikosti ctvercu.\n", fileName);
		exit(EXIT_FAILURE);
	}
	
	//skip white space
	while(isspace(c = fgetc(f)))
		;
	ungetc(c, f);

	//load number of squares horizontally and vertically
	if(fscanf(f, "squares:%d;%d", &(coords->xSquares), &(coords->ySquares)) != 2){
		fprintf(stderr, "Soubor %s neobsahuje korektni specifikaci poctu ctvercu.\n", fileName);
		exit(EXIT_FAILURE);
	}
		
	//allocate memory for square coefficients
	if((coords->squares = malloc(2 * coords->xSquares * coords->ySquares * sizeof(TCoordsSquare))) == NULL){
		fprintf(stderr, "Alokace pameti se nezdarila.\n");
		exit(EXIT_FAILURE);
	}

	//skip white space
	while(isspace(c = fgetc(f)))
		;
	ungetc(c, f);
	
	for(int i = 0; i < (coords->xSquares * coords->ySquares); i++){

		float S0x, S0y, DC0x, DC0y, DRx, DRy, DDCx, DDCy;
		if(fscanf(f, "%f:%f;%f:%f;%f:%f;%f:%f\n", &S0x, &S0y, &DC0x, &DC0y, &DRx, &DRy, &DDCx, &DDCy) != 8){
			fprintf(stderr, "Soubor %s, x-ovy ctverec %d neni korektne zapsan.\n", fileName, i);
			exit(EXIT_FAILURE);
		}

		TPoint2D S0, DC0, DR, DDC;
		bool overflow;
		//convert floats to bit representations
		S0.x  = float2pos(S0x, &overflow);
		if (overflow) fprintf(stderr, "Overflow: square %d, coefficient S0.x: |%f| > %f\n", i, S0x, POS_MAX);
		S0.y  = float2pos(S0y, &overflow);
		if (overflow) fprintf(stderr, "Overflow: square %d, coefficient S0.y: |%f| > %f\n", i, S0y, POS_MAX);
		DC0.x = float2vec(DC0x, &overflow);
		if (overflow) fprintf(stderr, "Overflow: square %d, coefficient DC0.x: |%f| > %.10f\n", i, DC0x, VEC_MAX);
		DC0.y = float2vec(DC0y, &overflow);
		if (overflow) fprintf(stderr, "Overflow: square %d, coefficient DC0.y: |%f| > %.10f\n", i, DC0y, VEC_MAX);
		DR.x  = float2vec(DRx, &overflow);
		if (overflow) fprintf(stderr, "Overflow: square %d, coefficient DR.x: |%f| > %.10f\n", i, DRx, VEC_MAX);
		DR.y  = float2vec(DRy, &overflow);
		if (overflow) fprintf(stderr, "Overflow: square %d, coefficient DR.y: |%f| > %.10f\n", i, DRy, VEC_MAX);
		DDC.x = float2dif(DDCx, &overflow);
		if (overflow) fprintf(stderr, "Overflow: square %d, coefficient DDC.x: |%f| > %f\n", i, DDCx, DIF_MAX);
		DDC.y = float2dif(DDCy, &overflow);
		if (overflow) fprintf(stderr, "Overflow: square %d, coefficient DDC.y: |%f| > %f\n", i, DDCy, DIF_MAX);

		coordsInitializeSquare(&(coords->squares[i]), S0, DC0, DR, DDC);

	}
	
	//close file
	fclose(f);
	return;
}


/**
 * Prints coefficients of all squares.
 * @param coords Pointer to structure with coefficients of all squares
 */
void coordsPrint(TCoords *coords)
{
	printf("X-squares:\n");
	for(int i = 0; i < (coords->xSquares * coords->ySquares); i++){
		coordsPrintSquare(&(coords->squares[i]));
	}
	
	printf("Y-squares:\n");
	for(int i = 0; i < (coords->xSquares * coords->ySquares); i++){
		coordsPrintSquare(&(coords->squares[coords->xSquares * coords->ySquares + i]));
	}

	return;
}

/**
 * Initializes structure of coefficients of one square.
 * @param coords Pointer to structure with coefficients of one square
 * @param S0 Value of S0 coefficient
 * @param DC0 Value of DC0 coefficient
 * @param DR Value of DR coefficient
 * @param DDC Value of DDC coefficient
 */
void coordsInitializeSquare(TCoordsSquare *coords, TPoint2D S0, TPoint2D DC0, TPoint2D DR, TPoint2D DDC)
{
	coords->S0.x   = S0.x;
	coords->S0.y   = S0.y;
	coords->DC0.x = DC0.x;
	coords->DC0.y = DC0.y;
	coords->DR.x = DR.x;
	coords->DR.y = DR.y;
	coords->DDC.x = DDC.x;
	coords->DDC.y = DDC.y;

	return;
}


/**
 * Prints coefficients of one square.
 * @param coords Pointer to structure with coefficients of one square
 */
void coordsPrintSquare(TCoordsSquare *coords)
{
	printf(  "S0.x: %.4f [%lx]",   pos2float(coords->S0.x), coords->S0.x);
	printf("  S0.y: %.4f [%lx]",   pos2float(coords->S0.y), coords->S0.y);
	printf("  DC0.x: %.10f [%lx]", vec2float(coords->DC0.x), coords->DC0.x);
	printf("  DC0.y: %.10f [%lx]", vec2float(coords->DC0.y), coords->DC0.y);
	printf("  DR.x: %.10f [%lx]", vec2float(coords->DR.x), coords->DR.x);
	printf("  DR.y: %.10f [%lx]", vec2float(coords->DR.y), coords->DR.y);
	printf("  DDC.x: %.16f [%lx]", dif2float(coords->DDC.x), coords->DDC.x);
	printf("  DDC.y: %.16f [%lx]", dif2float(coords->DDC.y), coords->DDC.y);
	printf("\n");

	return;
}
 
/* end of file squares.c */

