/*
 * File:     bit-handling.h, compiled by gcc
 * Date:     2009-02-02 21:02:21 
 * Author:   Bronislav Pribyl, xpriby12@stud.fit.vutbr.cz
 * Project:  Separable image resampling
 * Encoding: utf-8
 * Version:  1.5
 * Desc:     Emulates exact bit representations of variables and coefficients of interpolation algorithm
 */
 
 
#include <stdbool.h>
#include <math.h>
#include <stdlib.h>


#define INTEGER 16 //bit with weight 2^0 == 1

#define POS_L 12 //LSb
#define POS_H 21 //MSb - sign bit
#define POS_MAX  ((~(~0 << (POS_H))  &  (~0 << (POS_L))) / (pow(2, (INTEGER))))

#define VEC_L 6  //LSb
#define VEC_H 14 //MSb - sign bit
#define VEC_MAX  ((~(~0 << (VEC_H))  &  (~0 << (VEC_L))) / (pow(2, (INTEGER))))

#define DIF_L 0  //LSb
#define DIF_H 12 //MSb - sign bit
#define DIF_MAX  ((~(~0 << (DIF_H))  &  (~0 << (DIF_L))) / (pow(2, (INTEGER))))

#define VAR_L 0  //LSb
#define VAR_H 24 //MSb - sign bit


typedef enum varType {
	POSITION = 1,   //S0
	VECTOR,         //DC0, DR
	DIFFERENCE,     //DDC
	VARIABLE        //SoR
} TVarType;


/**
 * Prints bits of the number in groups of 8.
 * @param num Integer to be printed
 */
void printIntBits (int num);


/**
 * Cutts of garbage in bit representation of variable
 * @param var Bit representation of variable
 */  
#define CLEAN_VAR(var)\
	((var) &= (~0 << (VAR_L)));



/**
 * Converts bit representation of position coefficient (S0) into float.
 * @param pos Bit representation of position coefficient
 * @return Float position coefficient
 */
float pos2float (long int pos);


/**
 * Converts float into bit representation of position ocefficient (S0). If overflow
 * occurs, parameter ovf is set to true, otherwise to false.
 * @param f Float number
 * @param ovf Reference to boolean variable where information about overflow is stored
 * @return Bit representation of position coefficient
 */
long int float2pos (float f, bool *ovf);


/**
 * Converts bit representation of translation vector coefficient (DC0, DR) into float.
 * @param vec Bit representation of translation vector coefficient
 * @return Float translation vector coefficient
 */
float vec2float (long int vec);


/**
 * Converts float into bit representation of translation vector ocefficient (DC0, DR).
 * If overflow occurs, parameter ovf is set to true, otherwise to false.
 * @param f Float number
 * @param ovf Reference to boolean variable where information about overflow is stored
 * @return Bit representation of translation vector coefficient
 */
long int float2vec (float f, bool *ovf);


/**
 * Converts bit representation of difference vector coefficient (DDC) into float.
 * @param dif Bit representation of difference vector coefficient
 * @return Float difference vector coefficient
 */
float dif2float (long int dif);


/**
 * Converts float into bit representation of difference vector ocefficient (DDC).
 * If overflow occurs, parameter ovf is set to true, otherwise to false.
 * @param f Float number
 * @param ovf Reference to boolean variable where information about overflow is stored
 * @return Bit representation of difference vector coefficient
 */
long int float2dif (float f, bool *ovf);


/**
 * Converts bit representation of variable (SoR, DC, S) into float.
 * @param var Bit representation of variable
 * @return Float variable
 */
float var2float (long int var);


/**
 * Converts float into bit representation of variable (SoR, DC, S).
 * If overflow occurs, parameter ovf is set to true, otherwise to false.
 * @param f Float number
 * @param ovf Reference to boolean variable where information about overflow is stored
 * @return Bit representation of variable
 */
long int float2var (float f, bool *ovf);


/**
 * Determines FIR variant from fractional part of offset.
 * If offset is negative, fractional part must be increased by 1.0.
 * @param offset Offset in the form of a variable
 * @return FIR variant (integer between 0 and 15)
 */
int firVariant (long int offset);


/**
 * Checks whether index is in range according to variable type.
 * If check does not succeed, error message is printed and program exits with failure.
 * @param type Variable type
 * @param index Bit number (0 = LSB)
 */
void checkIndex (TVarType type, int index);


/**
 * Sets bit Nr. index in variable var of type type to value.
 * @param var Pointer to variable
 * @param type Type of variable
 * @param index Bit number (0 = LSB)
 * @param value 0 or 1, false or true
 */
void setBit (long int *var, TVarType type, int index, bool value);


/**
 * Gets bit Nr. index from variable var of type type.
 * @param var Variable
 * @param type Type of variable
 * @param index Bit number (0 = LSB)
 * @return Bit value
 */
bool getBit(long int var, TVarType type, int index);

/* end of file bit-handling.h */

