/*
 * File:     bit-handling.c, compiled by gcc
 * Date:     2009-02-02 21:02:03 
 * Author:   Bronislav Pribyl, xpriby12@stud.fit.vutbr.cz
 * Project:  Separable image resampling
 * Encoding: utf-8
 * Version:  1.5
 * Desc:     Emulates exact bit representations of variables and coefficients of interpolation algorithm
 */


#include "bit-handling.h"
#include "FIR.h"
#include <math.h>
#include <stdio.h>
#include <stdbool.h>


/**
 * Prints bits of the number in groups of 8.
 * @param num Integer to be printed
 */
void printIntBits (int num)
{
	for(int i = 31; i >= 0; i--)
	{
		printf("%d", (num >> i & 0x00000001));
		if(i % 8 == 0)
			printf(" ");
	}
	printf("\n");

	return;
}


/**
 * Converts bit representation of position coefficient (S0) into float.
 * @param pos Bit representation of position coefficient
 * @return Float position coefficient
 */
float pos2float (long int pos)
{
	//determine sign
	bool sign = pos & (1 << POS_H);
	
	//mask garbage and sign
	pos &= (~(~0 << POS_H)  &  (~0 << POS_L));
	
	//shift away all unused lower bits
	pos >>= POS_L;
	
	//insert sign
	if(sign)
		pos |= ~0 << (POS_H - POS_L);
	
	//convert
	float tmp = (float)pos;
	//adjust decimal point
	tmp /= pow(2, INTEGER - POS_L);

	return tmp;
}


/**
 * Converts float into bit representation of position ocefficient (S0). If overflow
 * occurs, parameter ovf is set to true, otherwise to false.
 * @param f Float number
 * @param ovf Reference to boolean variable where information about overflow is stored
 * @return Bit representation of position coefficient
 */
long int float2pos (float f, bool *ovf)
{
	//adjust decimal point
	f *= pow(2, INTEGER - POS_L);
	//convert to integer
	long int tmp = (long int)f;
	//convert to positive integer
	long int tmpPos = (long int)fabs(f);
		
	//align decimal point
	tmp <<= POS_L; tmpPos <<= POS_L;
	
	//check overflow
	*ovf = (tmpPos & (~0 << POS_H));
	
	//cut off garbage
	tmp &= ~0 << POS_L;
		
	return tmp;
}


/**
 * Converts bit representation of translation vector coefficient (DC0, DR) into float.
 * @param vec Bit representation of translation vector coefficient
 * @return Float translation vector coefficient
 */
float vec2float (long int vec)
{
	//determine sign
	bool sign = vec & (1 << VEC_H);
	
	//mask garbage and sign
	vec &= (~(~0 << VEC_H)  &  (~0 << VEC_L));
	
	//shift away all unused lower bits
	vec >>= VEC_L;
	
	//insert sign
	if(sign)
		vec |= ~0 << (VEC_H - VEC_L);
	
	//convert
	float tmp = (float)vec;
	//adjust decimal point
	tmp /= pow(2, INTEGER - VEC_L);

	return tmp;
}


/**
 * Converts float into bit representation of translation vector ocefficient (DC0, DR).
 * If overflow occurs, parameter ovf is set to true, otherwise to false.
 * @param f Float number
 * @param ovf Reference to boolean variable where information about overflow is stored
 * @return Bit representation of translation vector coefficient
 */
long int float2vec (float f, bool *ovf)
{
	//adjust decimal point
	f *= pow(2, INTEGER - VEC_L);
	//convert to integer
	long int tmp = (long int)f;
	//convert to positive integer
	long int tmpPos = (long int)fabs(f);
		
	//align decimal point
	tmp <<= VEC_L; tmpPos <<= VEC_L;
	
	//check overflow
	*ovf = (tmpPos & (~0 << VEC_H));
	
	//cut off garbage
	tmp &= ~0 << VEC_L;
		
	return tmp;
}


/**
 * Converts bit representation of difference vector coefficient (DDC) into float.
 * @param dif Bit representation of difference vector coefficient
 * @return Float difference vector coefficient
 */
float dif2float (long int dif)
{
	//determine sign
	bool sign = dif & (1 << DIF_H);
	
	//mask garbage and sign
	dif &= (~(~0 << DIF_H)  &  (~0 << DIF_L));
	
	//shift away all unused lower bits
	dif >>= DIF_L;
	
	//insert sign
	if(sign)
		dif |= ~0 << (DIF_H - DIF_L);
	
	//convert
	float tmp = (float)dif;
	//adjust decimal point
	tmp /= pow(2, INTEGER - DIF_L);

	return tmp;
}


/**
 * Converts float into bit representation of difference vector ocefficient (DDC).
 * If overflow occurs, parameter ovf is set to true, otherwise to false.
 * @param f Float number
 * @param ovf Reference to boolean variable where information about overflow is stored
 * @return Bit representation of difference vector coefficient
 */
long int float2dif (float f, bool *ovf)
{
	//adjust decimal point
	f *= pow(2, INTEGER - DIF_L);
	//convert to integer
	long int tmp = (long int)f;
	//convert to positive integer
	long int tmpPos = (long int)fabs(f);
		
	//align decimal point
	tmp <<= DIF_L; tmpPos <<= DIF_L;
	
	//check overflow
	*ovf = (tmpPos & (~0 << DIF_H));
	
	//cut off garbage
	tmp &= ~0 << DIF_L;
		
	return tmp;
}


/**
 * Converts bit representation of variable (SoR, DC, S) into float.
 * @param var Bit representation of variable
 * @return Float variable
 */
float var2float (long int var)
{
	//determine sign
	bool sign = var & (1 << VAR_H);
	
	//mask garbage and sign
	var &= (~(~0 << VAR_H)  &  (~0 << VAR_L));
	
	//shift away all unused lower bits
	var >>= VAR_L;
	
	//insert sign
	if(sign)
		var |= ~0 << (VAR_H - VAR_L);
	
	//convert
	float tmp = (float)var;
	//adjust decimal point
	tmp /= pow(2, INTEGER - VAR_L);

	return tmp;
}


/**
 * Converts float into bit representation of variable (SoR, DC, S).
 * If overflow occurs, parameter ovf is set to true, otherwise to false.
 * @param f Float number
 * @param ovf Reference to boolean variable where information about overflow is stored
 * @return Bit representation of variable
 */
long int float2var (float f, bool *ovf)
{
	//adjust decimal point
	f *= pow(2, INTEGER - VAR_L);
	//convert to integer
	long int tmp = (long int)f;
	//convert to positive integer
	long int tmpPos = (long int)fabs(f);
		
	//align decimal point
	tmp <<= VAR_L; tmpPos <<= VAR_L;
	
	//check overflow
	*ovf = (tmpPos & (~0 << VAR_H));
	
	//cut off garbage
	tmp &= ~0 << VAR_L;
		
	return tmp;
}


/**
 * Determines FIR variant from fractional part of offset.
 * If offset is negative, fractional part must be increased by 1.0. // <-- je toto zohlednene?
 * @param offset Offset in the form of a variable
 * @return FIR variant (integer between 0 and 15)
 */
int firVariant (long int offset)
{
	int variant;
	
	//mask integer part
	offset &= (long int)(pow(2, INTEGER) - 1);
	//shift right to maintain bits determining FIR variant (16 variants = 4 bits)
	variant = offset >> (INTEGER - (int)log2(FIR_VARIANTS));
	//variant = variant + (FIR_VARIANTS / 2); //<-- nove
	//(variant + FIR_VARIANTS/2) % FIR_VARIANTS
	//variant = FIR_VARIANTS - variant;

	if(variant > 15) return variant - 16;// <- toto upravene tak, aby to fungovalo jako modulo
	if(variant < 0)  return variant + 16;
	return variant;
}


/**
 * Checks whether index is in range according to variable type.
 * If check does not succeed, error message is printed and program exits with failure.
 * @param type Variable type
 * @param index Bit number (0 = LSB)
 */
void checkIndex (TVarType type, int index)
{
	switch (type) {
		case POSITION:
			if (index < POS_L || index > POS_H) {
				fprintf(stderr, "Index %d is out of range <%d, %d> of type %d.\n", index, POS_H, POS_L, type);
				exit(EXIT_FAILURE);
			}
			break;
			
		case VECTOR:
			if (index < VEC_L || index > VEC_H) {
				fprintf(stderr, "Index %d is out of range <%d, %d> of type %d.\n", index, VEC_H, VEC_L, type);
				exit(EXIT_FAILURE);
			}
			break;
			
		case DIFFERENCE:
			if (index < DIF_L || index > DIF_H) {
				fprintf(stderr, "Index %d is out of range <%d, %d> of type %d.\n", index, DIF_H, DIF_L, type);
				exit(EXIT_FAILURE);
			}
			break;
		
		case VARIABLE:
			if (index < VAR_L || index > VAR_H) {
				fprintf(stderr, "Index %d is out of range <%d, %d> of type %d.\n", index, VAR_H, VAR_L, type);
				exit(EXIT_FAILURE);
			}
			break;
			
		default:
			fprintf(stderr, "Variable type %d is not supported in varType.\n", type);
			exit(EXIT_FAILURE);
			break;
	}
	
	return;
}


/**
 * Sets bit Nr. index in var to value.
 * @param var Pointer to variable
 * @param type Type of variable
 * @param index Bit number (0 = LSB)
 * @param value 0 or 1, false or true
 */
void setBit (long int *var, TVarType type, int index, bool value)
{
	checkIndex(type, index);
	
	long int pattern = 1 << (index);
	
	if(value) {
		// set to 1
    (*var) |= pattern;
  } else {
  	// set to 0
    (*var) &= ~pattern;
  }
  
  return;
}


/**
 * Gets bit Nr. index from variable var of type type.
 * @param var Variable
 * @param type Type of variable
 * @param index Bit number (0 = LSB)
 * @return Bit value
 */
bool getBit(long int var, TVarType type, int index)
{
	checkIndex(type, index);
	return var & (1 << index);
}

/* end of file bit-handling.c */

