//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)    \n
 * Copyright (c) 2003-2006 by Michal Spanel     \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz  \n
 * File:    mdsMutexTEST.cpp                    \n
 * Section: libSystemTEST                       \n
 * Date:    2006/01/18                          \n
 *
 * $Id: mdsLockTEST.cpp 64 2006-08-11 08:45:24Z spanel $
 *
 * Description:
 * - Testing of the mds::CLock class.
 */

#include <MDSTk/Base/mdsSetup.h>
#include <MDSTk/Base/mdsLock.h>
#include <MDSTk/System/mdsMutex.h>
#include <MDSTk/System/mdsCriticalSection.h>

#include <cstdlib>
#include <ctime>

// STL
#include <iostream>


//==============================================================================
/*
 * Global constants and variables.
 */

//! The number of testing loops.
const int COUNT     = 1000000;


//==============================================================================
/*
 * Global variables.
 */

//! Clock counter
clock_t ClockCounter;


//==============================================================================
/*
 * Global functions.
 */

//! Waiting for a key
void keypress()
{
    while( std::cin.get() != '\n' );
}


//! Starts time measuring
void begin()
{
    ClockCounter = clock();
}


//! Stops time measuring and prints results
void end()
{
    ClockCounter = clock() - ClockCounter;
    std::cout << "  Measured clock ticks: " << ClockCounter << std::endl;
}


//==============================================================================
/*!
 * Class using mutex for mutual access to its members.
 */
class CMutexLock
{
public:
    void set(int iValue)
    {
        m_Lock.lock();
        m_iValue = iValue;
        m_Lock.unlock();
    }

protected:
    int m_iValue;
    mds::sys::CMutex m_Lock;
};


//==============================================================================
/*!
 * Class using critical section for mutual access to its members.
 */
class CCriticalSectionLock
{
public:
    void set(int iValue)
    {
        m_Lock.enter();
        m_iValue = iValue;
        m_Lock.leave();
    }

protected:
    int m_iValue;
    mds::sys::CCriticalSection m_Lock;
};


//==============================================================================
/*!
 * Class using base::CLockableObject for mutual access to its members.
 */
class CObjectLock : public mds::base::CLockableObject<CObjectLock>
{
public:
    typedef mds::base::CLockableObject<CObjectLock>::CLock tLock;

public:
    void set(int iValue)
    {
        tLock Lock(*this);
        (void)Lock;

        m_iValue = iValue;
    }

protected:
    int m_iValue;
};


//==============================================================================
/*!
 * main
 */
int main(int argc, const char *argv[])
{
    CMutexLock MLock;
    int i;
    std::cout << "Testing mutual access using mutex" << std::endl;
    begin();
    for( i = 0; i < COUNT; ++i )
    {
        MLock.set(i);
    }
    end();
    keypress();

    CCriticalSectionLock CSLock;
    std::cout << "Testing mutual access using critical section" << std::endl;
    begin();
    for( i = 0; i < COUNT; ++i )
    {
        CSLock.set(i);
    }
    end();
    keypress();

    CObjectLock OLock;
    std::cout << "Testing mutual access using mds::CLoackableObject" << std::endl;
    begin();
    for( i = 0; i < COUNT; ++i )
    {
        OLock.set(i);
    }
    end();
    keypress();

    return 0;
}

