//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)    \n
 * Copyright (c) 2003-2005 by Michal Spanel     \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz  \n
 * File:    mdsMatrix.cpp                       \n
 * Section: libBaseTEST                         \n
 * Date:    2003/12/04                          \n
 *
 * $Id:mdsMatrixTEST.cpp 253 2006-02-20 09:56:33Z spanel $
 * 
 * Description:
 * - Testing of the mds::CMatrix template.
 */

#include <MDSTk/Base/mdsSetup.h>
#include <MDSTk/Math/mdsMatrix.h>
#include <MDSTk/Math/mdsVector.h>
#include <MDSTk/Math/mdsMatrixFunctions.h>

// STL
#include <iostream>


//==============================================================================
/*!
 * Prints a given matrix
 */
void printMatrix(mds::math::CDMatrix& m)
{
    std::cout.setf(std::ios_base::fixed);
    for( mds::tSize i = 0; i < m.getNumOfRows(); i++ )
    {
        std::cout << "  ";
        for( mds::tSize j = 0; j < m.getNumOfCols(); j++ )
        {
            std::cout << m(i,j) << " ";
        }
        std::cout << std::endl;
    }
}


//==============================================================================
/*!
 * Prints a given vector
 */
void printVector(mds::math::CDVector& v)
{
    std::cout.setf(std::ios_base::fixed);
    std::cout << "  ";
    for( mds::tSize i = 0; i < v.getSize(); i++ )
    {
        std::cout << v(i) << " ";
    }
    std::cout << std::endl;
}


//==============================================================================
/*!
 * Waiting for a key
 */
void keypress()
{
    while( std::cin.get() != '\n' );
}


//==============================================================================
/*!
 * main
 */
int main(int argc, const char *argv[])
{
    mds::math::CDMatrix m1(4, 4);
    m1.unit();
    std::cout << "Unit matrix 1" << std::endl;
    printMatrix(m1);
    keypress();

    mds::math::CDMatrix m2(4, 4);
    std::cout << "Matrix 2" << std::endl;
    mds::tSize i, j;
    for( i = 0; i < m2.getNumOfRows(); i++ )
    {
        for( j = 0; j < m2.getNumOfCols(); j++ )
        {
            m2(i,j) = i * m2.getNumOfRows() + j;
        }
    }
    printMatrix(m2);
    keypress();

    std::cout << "Operation m1 += m2" << std::endl;
    m1 += m2;
    printMatrix(m1);
    keypress();

    std::cout << "Operation m2 *= 2" << std::endl;
    m2 *= 2;
    printMatrix(m2);
    keypress();

    mds::math::CDMatrix m3(4, 4);
    std::cout << "Operation m3.mult(m1, m2)" << std::endl;
    m3.mult(m1, m2);
    printMatrix(m3);
    keypress();

    std::cout << "Matrix m1" << std::endl;
    for( i = 0; i < m1.getNumOfRows(); i++ )
    {
        for( j = 0; j < m1.getNumOfCols(); j++ )
        {
            m1(i,j) += i * m1.getNumOfCols() + j;
        }
    }
    printMatrix(m1);
    keypress();

    std::cout << "Operation m2.transpose(m1)" << std::endl;
    m2.transpose(m1);
    printMatrix(m2);
    keypress();

    std::cout << "Operation getDeterminant<double>(m2)" << std::endl;
    double dDet = mds::math::getDeterminant<double>(m2);
    std::cout << "  " << dDet << std::endl;
    keypress();

    std::cout << "Operation m1 = m2, inverse(m1)" << std::endl;
    m1 = m2;
    mds::math::inverse(m1);
    printMatrix(m1);
    keypress();

    std::cout << "Operation m3.mult(m1, m2)" << std::endl;
    m3.mult(m1, m2);
    printMatrix(m3);
    keypress();

    std::cout << "Create matrix m4 using constructor parameters (m1, 2, 2, 2, 2, mds::REFERENCE)" << std::endl;
    mds::math::CDMatrix m4(m1, 2, 2, 2, 2, mds::REFERENCE);
    printMatrix(m4);
    keypress();

    std::cout << "Operation m4 += 5" << std::endl;
    m4 += 5;
    printMatrix(m4);
    keypress();

    std::cout << "Matrix m1" << std::endl;
    printMatrix(m1);
    keypress();

    std::cout << "Re-creating matrix m4 using parameters (m1, 1, 1, 2, 2)" << std::endl;
    m4.create(m1, 1, 1, 2, 2);
    printMatrix(m4);
    keypress();

    std::cout << "Operation m4 *= 10" << std::endl;
    m4 *= 10;
    printMatrix(m4);
    keypress();

    std::cout << "Matrix m1" << std::endl;
    printMatrix(m1);
    keypress();

    mds::math::CDMatrix m5(5, 5);
    std::cout << "Matrix 5" << std::endl;
    m5(0,0) = 2; m5(0,1) = 1; m5(0,2) = 0; m5(0,3) = 0; m5(0,4) = 0;
    m5(1,0) = 1; m5(1,1) = 2; m5(1,2) = 1; m5(1,3) = 0; m5(1,4) = 0;
    m5(2,0) = 0; m5(2,1) = 1; m5(2,2) = 2; m5(2,3) = 1; m5(2,4) = 0;
    m5(3,0) = 0; m5(3,1) = 0; m5(3,2) = 1; m5(3,3) = 2; m5(3,4) = 1;
    m5(4,0) = 0; m5(4,1) = 0; m5(4,2) = 0; m5(4,3) = 1; m5(4,4) = 2;
    printMatrix(m5);
    keypress();

    mds::math::CDVector v(5);
    std::cout << "Operation eig(m5, v)" << std::endl;
    mds::math::eig(m5, v);
    std::cout << "Eigenvectors:" << std::endl;
    printMatrix(m5);
    std::cout << "Eigenvalues:" << std::endl;
    printVector(v);
    keypress();

    return 0;
}

