//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)    \n
 * Copyright (c) 2003-2005 by Michal Spanel     \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz  \n
 * File:    mdsVolumeTEST.cpp                   \n
 * Section: libImageTEST                        \n
 * Date:    2004/06/01                          \n
 *
 * $Id: mdsVolumeTEST.cpp 345 2007-06-11 13:23:09Z spanel $
 *
 * Description:
 * - Testing of the mds::CVolume template.
 */

#include <MDSTk/Base/mdsSetup.h>
#include <MDSTk/Image/mdsImage.h>
#include <MDSTk/Image/mdsVolume.h>
#include <MDSTk/Image/mdsVolumeFunctions.h>

// STL
#include <iostream>
#include <string>
#include <ctime>


//==============================================================================
/*!
 * Prints a given image
 */
void printImage(mds::img::CDImage& Image, bool bPrintMargin = false)
{
    std::cout.setf(std::ios_base::fixed);
    mds::tSize Margin = (bPrintMargin) ? Image.getMargin() : 0;
    for( mds::tSize j = -Margin; j < Image.getYSize() + Margin; j++ )
    {
        std::cout << "  ";
        for( mds::tSize i = -Margin; i < Image.getXSize() + Margin; i++ )
        {
            std::cout << Image(i, j) << "  ";
        }
        std::cout << std::endl;
    }
}


//==============================================================================
/*!
 * Prints a given volume
 */
void printVolume(mds::img::CDVolume& Volume, bool bPrintMargin = false)
{
    static char pcPrefix[255];

    std::cout.setf(std::ios_base::fixed);
    mds::tSize Margin = (bPrintMargin) ? Volume.getMargin() : 0;
    for( mds::tSize j = -Margin; j < Volume.getYSize() + Margin; j++ )
    {
        for( mds::tSize k = -Margin; k < Volume.getZSize() + Margin; k++ )
        {
            memset(pcPrefix, (int)' ', Volume.getZSize() - k + 1);
            pcPrefix[Volume.getZSize() - k + 1] = '\0';
            std::cout << pcPrefix;
            for( mds::tSize i = -Margin; i < Volume.getXSize() + Margin; i++ )
            {
                std::cout << Volume(i, j, k) << "  ";
            }
            std::cout << std::endl << std::endl;
        }
    }
}


//==============================================================================
/*!
 * Waiting for a key
 */
void keypress()
{
    while( std::cin.get() != '\n' );
}


//==============================================================================
/*!
 * Clock counter
 */
clock_t ClockCounter;


//==============================================================================
/*!
 * Starts time measuring
 */
void begin()
{
    ClockCounter = clock();
}

//==============================================================================
/*!
 * Stops time measuring and prints result
 */
void end()
{
    ClockCounter = clock() - ClockCounter;
    std::cout << "  Measured clock ticks: " << ClockCounter << std::endl;
}


//==============================================================================
/*!
 * main
 */
int main(int argc, const char *argv[])
{
    mds::img::CDVolumePtr spVolume1(new mds::img::CDVolume(3, 3, 3, (mds::img::CDVolume::tVoxel)123, 1));
    std::cout << "Smart pointer to volume 1" << std::endl;
    printVolume(*spVolume1);
    keypress();

    mds::img::CDVolume Volume2(*spVolume1, 1, 1, 1, 2, 2, 2, mds::REFERENCE);
    std::cout << "Creating volume 2 using constructor parameters (*spVolume1, 1, 1, 1, 2, 2, 2, mds::REFERENCE)" << std::endl;
    std::cout << "- Reference to the volume 1" << std::endl;
    mds::tSize i, j, k;
    for( k = 0; k < Volume2.getZSize(); k++ )
    {
        for( j = 0; j < Volume2.getYSize(); j++ )
        {
            for( i = 0; i < Volume2.getXSize(); i++ )
            {
                Volume2(i, j, k) = k * Volume2.getYSize() * Volume2.getZSize() + j * Volume2.getXSize() + i;
            }
        }
    }
    printVolume(Volume2);
    keypress();

    std::cout << "Volume 1" << std::endl;
    printVolume(*spVolume1);
    keypress();

    std::cout << "Volume slice - plane XY, z = 1" << std::endl;
    mds::img::CDImage Plane(3, 3, (mds::img::CDImage::tPixel)0, 2);
    spVolume1->getPlaneXY(1, Plane);
    printImage(Plane);
    keypress();

    std::cout << "Volume slice - plane XZ, y = 2" << std::endl;
    spVolume1->getPlaneXZ(2, Plane);
    printImage(Plane);
    keypress();

    std::cout << "Operation *spVolume1 += Volume2" << std::endl;
    *spVolume1 += Volume2;
    printVolume(*spVolume1);
    keypress();

    std::cout << "Operation Volume2 *= 2" << std::endl;
    Volume2 *= 2;
    printVolume(Volume2);
    keypress();

    std::cout << "Volume 1" << std::endl;
    printVolume(*spVolume1);
    keypress();

    std::cout << "Operation mds::img::getMin<double>(*spVolume1), getMax(), getSum(), getMean()" << std::endl;
    std::cout << "  " << mds::img::getMin<double>(*spVolume1)
    << "  " << mds::img::getMax<double>(*spVolume1)
    << "  " << mds::img::getSum<double>(*spVolume1)
    << "  " << mds::img::getMean<double>(*spVolume1)
    << std::endl;
    keypress();

    mds::img::CDVolume Volume3(*spVolume1, 0, 0, 0, 2, 2, 2);
    std::cout << "Creating volume 3 using constructor parameters (*spVolume1, 0, 0, 0, 2, 2, 2)" << std::endl;
    std::cout << "- Copy of the volume 1" << std::endl;
    printVolume(Volume3);
    keypress();

    std::cout << "Volume 3" << std::endl;
    printVolume(Volume3);
    keypress();

    std::cout << "Operation Volume3 /= 2" << std::endl;
    Volume3 /= 2;
    printVolume(Volume3);
    keypress();

    std::cout << "Volume 1" << std::endl;
    printVolume(*spVolume1);
    keypress();

    Volume3.create(2, 2, 2, mds::img::CDVolume::tVoxel(0), 1);
    std::cout << "Creating volume 3 with margin (3, 3, 3, tVoxel(0), 1)" << std::endl;
    for( k = 0; k < Volume3.getZSize(); k++ )
    {
        for( j = 0; j < Volume3.getYSize(); j++ )
        {
            for( i = 0; i < Volume3.getXSize(); i++ )
            {
                Volume3(i, j, k) = k * Volume3.getYSize() * Volume3.getZSize() + j * Volume3.getXSize() + i;
            }
        }
    }
    printVolume(Volume3, true);
    keypress();

    std::cout << "Fill margin of the volume 3" << std::endl;
    Volume3.fillMargin(1);
    printVolume(Volume3, true);
    keypress();

    std::cout << "Mirror margin of the volume 3" << std::endl;
    Volume3.mirrorMargin();
    printVolume(Volume3, true);
    keypress();

    std::cout << "Volume 2" << std::endl;
    printVolume(Volume2);
    keypress();

    std::cout << "Iterate volume 2" << std::endl;
    std::cout << "  ";
    {
        mds::img::CDVolume::tIterator it = Volume2.getBegin();
        mds::img::CDVolume::tIterator itEnd = Volume2.getEnd();
        std::cout << "it.getDistance(itEnd): " << it.getDistance(itEnd) << std::endl;
        while( it != itEnd )
        {
            std::cout << *it << "  ";
            ++it;
        }
    }
    std::cout << std::endl;
    keypress();

    std::cout << "Iterate volume 2" << std::endl;
    std::cout << "- Initial iterator position is 1,1,1" << std::endl;
    std::cout << "  ";
    {
        mds::img::CDVolume::tIterator it = Volume2.getIterator(1,1,1);
        mds::img::CDVolume::tIterator itEnd = Volume2.getEnd();
        std::cout << "it.getDistance(itEnd): " << it.getDistance(itEnd) << std::endl;
        while( it != itEnd )
        {
            std::cout << *it << "  ";
            ++it;
        }
    }
    std::cout << std::endl;
    keypress();

    std::cout << "Testing voxel access overload" << std::endl;
    Volume3.create(256, 256, 256, 8);

    std::cout << "  Basic version" << std::endl;
    begin();
    int c, COUNT = 100;
//  int c, COUNT = 5;
    for( c = 0; c < COUNT; ++c )
    {
        for( k = 0; k < Volume3.getZSize(); ++k )
        {
            for( j = 0; j < Volume3.getYSize(); ++j )
            {
                for( i = 0; i < Volume3.getXSize(); ++i )
                {
                    Volume3(i, j, k) = 1234;
                }
            }
        }
    }
    end();
    keypress();

    // Extended version
    std::cout << "  Fill version" << std::endl;
    begin();
    for( c = 0; c < COUNT; ++c )
    {
        Volume3.fill(1234);
    }
    end();
    keypress();

    // Iterator version
    std::cout << "  Iterator version" << std::endl;
    begin();
    for( c = 0; c < COUNT; ++c )
    {
        mds::img::CDVolume::tIterator it = Volume3.getBegin();
        mds::img::CDVolume::tIterator itEnd = Volume3.getEnd();
        while( it != itEnd )
        {
            *it = 1234;
            ++it;
        }
    }
    end();

    return 0;
}

