//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)    \n
 * Copyright (c) 2003-2005 by Michal Spanel     \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz  \n
 * File:    mdsImageTEST.cpp                    \n
 * Section: libImageTEST                        \n
 * Date:    2004/06/01                          \n
 *
 * $Id: mdsImageTEST.cpp 350 2007-06-12 07:42:17Z spanel $
 *
 * Description:
 * - Testing of the mds::CImage template
 */

#include <MDSTk/Base/mdsSetup.h>
#include <MDSTk/Base/mdsGlobalLog.h>
#include <MDSTk/Image/mdsImage.h>
#include <MDSTk/Image/mdsImageFunctions.h>

// STL
#include <iostream>
#include <ctime>


//==============================================================================
/*!
 * Prints a given image
 */
void printImage(const mds::img::CDImage& Image, bool bPrintMargin = false)
{
    std::cout.setf(std::ios_base::fixed);
    mds::tSize Margin = (bPrintMargin) ? Image.getMargin() : 0;
    for( mds::tSize j = -Margin; j < Image.getYSize() + Margin; j++ )
    {
        std::cout << "  ";
        for( mds::tSize i = -Margin; i < Image.getXSize() + Margin; i++ )
        {
            std::cout << Image(i, j) << " ";
        }
        std::cout << std::endl;
    }
}


//==============================================================================
/*!
 * Waiting for a key
 */
void keypress()
{
    while( std::cin.get() != '\n' );
}


//==============================================================================
/*!
 * Clock counter
 */
clock_t ClockCounter;


//==============================================================================
/*!
 * Starts time measuring
 */
void begin()
{
    ClockCounter = clock();
}

//==============================================================================
/*!
 * Stops time measuring and prints result
 */
void end()
{
    ClockCounter = clock() - ClockCounter;
    std::cout << "  Measured clock ticks: " << ClockCounter << std::endl;
}


//==============================================================================
/*!
 * main
 */
int main(int argc, const char *argv[])
{
//    MDS_LOG_INIT_FILE("test.log");
    
    mds::img::CDImagePtr spIm1(new mds::img::CDImage(10, 10, (mds::img::CDImage::tPixel)2, 1));
    std::cout << "Smart pointer to image 1" << std::endl;
    printImage(*spIm1);
    keypress();

    mds::img::CDImage Im2(*spIm1, 5, 5, 3, 3, mds::REFERENCE);
    std::cout << "Creating image 2 using constructor parameters (*spIm1, 5, 5, 3, 3, mds::REFERENCE)" << std::endl;
    std::cout << " - Reference to the image 1" << std::endl;
    mds::tSize i, j;
    for( j = 0; j < Im2.getYSize(); j++ )
    {
        for( i = 0; i < Im2.getXSize(); i++ )
        {
            Im2(i,j) = i * Im2.getYSize() + j;
        }
    }
    printImage(Im2);
    keypress();

    std::cout << "Image 1" << std::endl;
    printImage(*spIm1);
    keypress();

    std::cout << "Operation *spIm1 += Im2" << std::endl;
    *spIm1 += Im2;
    printImage(*spIm1);
    keypress();

    std::cout << "Operation Im2 *= 2" << std::endl;
    Im2 *= 2;
    printImage(Im2);
    keypress();

    std::cout << "Image 1" << std::endl;
    printImage(*spIm1);
    keypress();

    std::cout << "Operation mds::img::getMin<double>(*spIm1), getMax(), getSum(), getMean()" << std::endl;
    std::cout << "  " << mds::img::getMin<double>(*spIm1)
    << "  " << mds::img::getMax<double>(*spIm1)
    << "  " << mds::img::getSum<double>(*spIm1)
    << "  " << mds::img::getMean<double>(*spIm1)
    << std::endl;
    keypress();

    std::cout << "Image 1" << std::endl;
    printImage(*spIm1);
    keypress();

    mds::img::CDImage Im3(*spIm1, 3, 3, 4, 3);
    std::cout << "Creating image 3 using constructor parameters (*spIm1, 3, 3, 4, 3)" << std::endl;
    std::cout << " - Copy of the image 1" << std::endl;
    printImage(Im3);
    keypress();

    std::cout << "Operation Im3 /= 2" << std::endl;
    Im3 /= 2;
    printImage(Im3);
    keypress();

    std::cout << "Image 1" << std::endl;
    printImage(*spIm1);
    keypress();

    Im3.create(3, 3, mds::img::CDImage::tPixel(0), 2);
    std::cout << "Creating image 3 with margin (3, 3, tPixel(0), 2)" << std::endl;
    for( j = 0; j < Im3.getYSize(); j++ )
    {
        for( i = 0; i < Im3.getXSize(); i++ )
        {
            Im3(i,j) = i * Im3.getYSize() + j;
        }
    }
    printImage(Im3, true);
    keypress();

    std::cout << "Fill margin of the image 3" << std::endl;
    Im3.fillMargin(1);
    printImage(Im3, true);
    keypress();

    std::cout << "Mirror margin of the image 3" << std::endl;
    Im3.mirrorMargin();
    printImage(Im3, true);
    keypress();

    std::cout << "Image 2" << std::endl;
    printImage(Im2);
    keypress();

    std::cout << "Iterate image 2" << std::endl;
    std::cout << "  ";
    {
        mds::img::CDImage::tIterator it = Im2.getBegin();
        mds::img::CDImage::tIterator itEnd = Im2.getEnd();
        std::cout << "it.getDistance(itEnd): " << it.getDistance(itEnd) << std::endl;
        while( it != itEnd )
        {
            std::cout << *it << "  ";
            ++it;
        }
        std::cout << std::endl;
        keypress();
    }

    std::cout << "Iterate image 2" << std::endl;
    std::cout << "- Set initial position to 1,1" << std::endl;
    std::cout << "  ";
    {
        mds::img::CDImage::tIterator it;
        it.setPosition(Im2, 1,1);
        mds::img::CDImage::tIterator itEnd = Im2.getEnd();
        while( it != itEnd )
        {
            std::cout << *it << "  ";
            ++it;
        }
        std::cout << std::endl;
        keypress();
    }

    const mds::img::CDImage Im4(Im2, mds::REFERENCE);
    std::cout << "Creating const image 4 using constructor parameters (Im2, mds::REFERENCE)" << std::endl;
    std::cout << " - Reference to the image 2" << std::endl;
    printImage(Im4);
    keypress();

    std::cout << "Iterate image 4" << std::endl;
    std::cout << "- Const iterator" << std::endl;
    std::cout << "  ";
    {
        mds::img::CDImage::tConstIterator it = Im4.getBegin();
        const mds::img::CDImage::tConstIterator itEnd = Im4.getEnd();
        std::cout << "it.getDistance(itEnd): " << it.getDistance(itEnd) << std::endl;
        while( it != itEnd )
        {
            std::cout << *it << "  ";
            ++it;
        }
        std::cout << std::endl;
        keypress();
    }

    std::cout << "Testing image pixel access overload" << std::endl;
    Im3.create(256, 256, 8, (mds::img::CDImage::tPixel)1234);

    std::cout << "  Basic version" << std::endl;
    begin();
    int c, COUNT = 20000;
//  int c, COUNT = 200;
    for( c = 0; c < COUNT; ++c )
    {
        for( j = 0; j < Im3.getYSize(); ++j )
        {
            for( i = 0; i < Im3.getXSize(); ++i )
            {
                mds::img::CDImage::tPixel Temp = Im3(i, j);
                Im3(i, j) = Temp;
            }
        }
    }
    end();
    keypress();

    // Iterator version
    std::cout << "  Iterator version" << std::endl;
    begin();
    for( c = 0; c < COUNT; ++c )
    {
        mds::img::CDImage::tIterator it = Im3.getBegin();
        mds::img::CDImage::tIterator itEnd = Im3.getEnd();
        while( it != itEnd )
        {
            mds::img::CDImage::tPixel Temp = *it;
            *it = Temp;
            ++it;
        }
    }
    end();

    return 0;
}

