//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)    \n
 * Copyright (c) 2003-2005 by Michal Spanel     \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz  \n
 * File:    mdsVolumeThresholding.cpp           \n
 * Section: mVolumeThresholding                 \n
 * Date:    2004/08/19                          \n
 *
 * $Id: mdsVolumeThresholding.cpp 345 2007-06-11 13:23:09Z spanel $
 *
 * File description:
 * - Simple voxel value thresholding.
 */

#include "mdsVolumeThresholding.h"

#include <MDSTk/Image/mdsSlice.h>
#include <MDSTk/Image/mdsDensityVolume.h>

// STL
#include <climits>
#include <iostream>
#include <string>


//==============================================================================
/*
 * Global constants.
 */

//! Module description
const std::string MODULE_DESCRIPTION    = "Module provides a simple voxel value thresholding.";

//! Additional command line arguments
const std::string MODULE_ARGUMENTS      = "min:max";

//! Additional arguments
const std::string MODULE_ARGUMENT_MIN   = "min";
const std::string MODULE_ARGUMENT_MAX   = "max";

//! Default min and max values
const unsigned int DEFAULT_MIN          = mds::img::CPixelTraits<mds::img::CDensityVolume::tVoxel>::getPixelMin();
const unsigned int DEFAULT_MAX          = mds::img::CPixelTraits<mds::img::CDensityVolume::tVoxel>::getPixelMax();


//==============================================================================
/*
 * Implementation of the class CVolumeThresholding.
 */
CVolumeThresholding::CVolumeThresholding(const std::string& sDescription)
    : mds::mod::CModule(sDescription)
{
    allowArguments(MODULE_ARGUMENTS);
}


CVolumeThresholding::~CVolumeThresholding()
{
}


bool CVolumeThresholding::startup()
{
    // Note
    MDS_LOG_NOTE("Module startup");

    // Test of existence of input and output channel
    if( getNumOfInputs() != 1 || getNumOfOutputs() != 1 )
    {
        MDS_CERR('<' << m_sFilename << "> Wrong number of input and output channels" << std::endl);
        return false;
    }

    // Parameters 'min' and 'max'
    m_uMin = DEFAULT_MIN;
    m_Arguments.value(MODULE_ARGUMENT_MIN, m_uMin);
    if( m_uMin > DEFAULT_MAX )
    {
        MDS_CERR('<' << m_sFilename << "> Bad 'min' parameter value: type -h for help" << std::endl);
        printUsage();
        return false;
    }

    // Parameter 'max'
    m_uMax = DEFAULT_MAX;
    m_Arguments.value(MODULE_ARGUMENT_MAX, m_uMax);
    if( m_uMax < m_uMin || m_uMax > DEFAULT_MAX )
    {
        MDS_CERR('<' << m_sFilename << "> Bad 'max' parameter value: type -h for help" << std::endl);
        printUsage();
        return false;
    }

    // O.K.
    return true;
}


bool CVolumeThresholding::main()
{
    // Note
    MDS_LOG_NOTE("Module main function");

    // I/O channels
    mds::mod::CChannel *pIChannel = getInput(0);
    mds::mod::CChannel *pOChannel = getOutput(0);

    // Is any input?
    if( !pIChannel->isConnected() )
    {
        return false;
    }

    // Create a new volume
    mds::img::CDensityVolumePtr spVolume;

    // Wait for data
    if( pIChannel->wait(1000) )
    {
        // Read volume from the input channel
        if( readInput(pIChannel, spVolume) )
        {
            // Interpolated interval
            mds::img::CDensityVolume::tVoxel TMin = (mds::img::CDensityVolume::tVoxel)m_uMin;
            mds::img::CDensityVolume::tVoxel TMax = (mds::img::CDensityVolume::tVoxel)m_uMax;

            // Convert slice pixels
            for( mds::tSize k = 0; k < spVolume->getZSize(); ++k )
            {
                for( mds::tSize j = 0; j < spVolume->getYSize(); ++j )
                {
                    for( mds::tSize i = 0; i < spVolume->getXSize(); ++i )
                    {
                        mds::img::CDensityVolume::tVoxel Value = spVolume->get(i, j, k);
                        if( Value > TMax )
                        {
                            spVolume->set(i, j, k, (mds::img::CDensityVolume::tVoxel)0);
                        }
                        else if( Value < TMin )
                        {
                            spVolume->set(i, j, k, (mds::img::CDensityVolume::tVoxel)0);
                        }
                        else
                        {
                            spVolume->set(i, j, k, (mds::img::CDensityVolume::tVoxel)1);
                        }
                    }
                }
            }

            // Write it to the output channel
            if( !writeOutput(pOChannel, spVolume) )
            {
                MDS_CERR('<' << m_sFilename << "> Failed to write the output volume" << std::endl);
                return false;
            }
        }
        else
        {
            MDS_CERR('<' << m_sFilename << "> Failed to read input volume" << std::endl);
            return false;
        }
    }
    else
    {
        MDS_LOG_NOTE("Wait timeout");
    }

    // Returning 'true' means to continue processing the input channel
    return true;
}


void CVolumeThresholding::shutdown()
{
    // Note
    MDS_LOG_NOTE("Module shutdown");
}


void CVolumeThresholding::writeExtendedUsage(std::ostream& Stream)
{
    MDS_CERR(std::endl);
    MDS_CERR("Extended usage: [-min uValue] [-max uValue]" << std::endl);
    MDS_CERR("Options:" << std::endl);
    MDS_CERR("  -min  Lower boundary of the thresholded interval." << std::endl);
    MDS_CERR("  -max  Higher interval boundary." << std::endl);
}


//==============================================================================
/*
 * Function main() which creates and executes console application.
 */
int main(int argc, char *argv[])
{
    // Creation of a module using smart pointer
    CVolumeThresholdingPtr spModule(new CVolumeThresholding(MODULE_DESCRIPTION));

    // Initialize and execute the module
    if( spModule->init(argc, argv) )
    {
        spModule->run();
    }

    // Console application finished
    return 0;
}

