//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)              \n
 * Copyright (c) 2003-2005 by Michal Spanel               \n
 *
 * Author:   Michal Spanel, spanel@fit.vutbr.cz           \n
 *           Martin Skotnica, xskotn01@stud.fit.vutbr.cz  \n
 * File:     mdsRegionGrowing.cpp                         \n
 * Section:  mVolumeSegRG                                 \n
 * Date:     2005/01/25                                   \n
 *
 * $Id: mdsRegionGrowing.cpp 345 2007-06-11 13:23:09Z spanel $
 *
 * File description:
 * - Simple segmentation method based on region growing and merging.
 */

#include "mdsRegionGrowing.h"

#include <MDSTk/Base/mdsGlobalLog.h>


namespace mds
{
namespace seg
{

//==============================================================================
/*
 * Implementation of the class mds::seg::CVolumeRegionGrowing.
 */
CVolumeRegionGrowing::CVolumeRegionGrowing(double dDiffWeight,
                                           double dLogWeight,
                                           double dThreshold,
                                           bool bMerge
                                           )
    : m_dThreshold(dThreshold)
    , m_dDiffWeight(dDiffWeight)
    , m_dLogWeight(dLogWeight)
    , m_bMerge(bMerge)
{
}


void CVolumeRegionGrowing::createSeeds(const SMySeed& Seed,
                                       CVolumeRegionGrowing::tVolume *pVolume,
                                       int XDim,
                                       int YDim,
                                       int ZDim
                                       )
{
    SMySeed Temp(0, 0, 0, Seed.m_iIndex);
    if( Seed.m_x > 0 )
    {
        Temp.m_x = Seed.m_x - 1;
        Temp.m_y = Seed.m_y;
        Temp.m_z = Seed.m_z;
        if( (int)pVolume->get(Temp.m_x, Temp.m_y, Temp.m_z) != Temp.m_iIndex )
        {
            m_Seeds.push_front(Temp);
        }
    }
    if( Seed.m_y > 0 )
    {
        Temp.m_x = Seed.m_x;
        Temp.m_y = Seed.m_y - 1;
        Temp.m_z = Seed.m_z;
        if( (int)pVolume->get(Temp.m_x, Temp.m_y, Temp.m_z) != Temp.m_iIndex )
        {
            m_Seeds.push_front(Temp);
        }
    }
    if( Seed.m_z > 0 )
    {
        Temp.m_x = Seed.m_x;
        Temp.m_y = Seed.m_y;
        Temp.m_z = Seed.m_z - 1;
        if( (int)pVolume->get(Temp.m_x, Temp.m_y, Temp.m_z) != Temp.m_iIndex )
        {
            m_Seeds.push_front(Temp);
        }
    }
    if( Seed.m_x < XDim - 1 )
    {
        Temp.m_x = Seed.m_x + 1;
        Temp.m_y = Seed.m_y;
        Temp.m_z = Seed.m_z;
        if( (int)pVolume->get(Temp.m_x, Temp.m_y, Temp.m_z) != Temp.m_iIndex )
        {
            m_Seeds.push_front(Temp);
        }
    }
    if( Seed.m_y < YDim - 1 )
    {
        Temp.m_x = Seed.m_x;
        Temp.m_y = Seed.m_y + 1;
        Temp.m_z = Seed.m_z;
        if( (int)pVolume->get(Temp.m_x, Temp.m_y, Temp.m_z) != Temp.m_iIndex )
        {
            m_Seeds.push_front(Temp);
        }
    }
    if( Seed.m_z < ZDim - 1 )
    {
        Temp.m_x = Seed.m_x;
        Temp.m_y = Seed.m_y;
        Temp.m_z = Seed.m_z + 1;
        if( (int)pVolume->get(Temp.m_x, Temp.m_y, Temp.m_z) != Temp.m_iIndex )
        {
            m_Seeds.push_front(Temp);
        }
    }
}


bool CVolumeRegionGrowing::operator()(const CVolumeRegionGrowing::tVolume *pSrcVolume,
                                      CVolumeRegionGrowing::tVolume *pDstVolume
                                      )
{
    // Volume dimensions
    tSize Xdim = pSrcVolume->getXSize();
    tSize Ydim = pSrcVolume->getYSize();
    tSize Zdim = pSrcVolume->getZSize();

    // Check destination image dimensions
    MDS_CHECK(Xdim == pDstVolume->getXSize() && Ydim == pDstVolume->getYSize() && Zdim == pDstVolume->getZSize(), return false);
    MDS_CHECK(Xdim > 0 && Ydim > 0 && Zdim > 0, return false);

    // Vyplni vystup cernou barvnou
    pDstVolume->fill(0);

    // Vypocet odezvy LoG filtru
    tVolume::tSmartPtr spLog(new tVolume(Xdim, Ydim, Zdim));
    mds::img::CVolumeFilter<tVolume, mds::img::VF_LAPLACIAN0, mds::img::VFN_ABS> Filter;
    Filter(*pSrcVolume, *spLog);

    // Zruseni seminek
    m_Seeds.clear();

    // Cislovani regionu musi zacinat od 1
    m_Regions.clear();
    m_Regions.push_back(SMyRegion(0));

    SMySeed Seed;
    int iValue, iAssignedIndex;
    double dDiff, dResp, dResult;

    // Inicializace seminka
    int iColor = 0;
    tSize x = 0, y = 0, z = 0;

    while( z < Zdim )
    {
        // Vyhledani seminka
        do {
            if( (int)pDstVolume->get(x, y, z) == 0 )
            {
                m_Regions.push_back(SMyRegion((long)pSrcVolume->get(x, y, z)));
                m_Seeds.push_back(SMySeed(x, y, z, ++iColor));
            }
            else
            {
                ++x;
                if( x >= Xdim )
                {
                    x = 0;
                    ++y;
                    if( y >= Ydim )
                    {
                        y = 0;
                        ++z;
                    }
                }
            }
        } while( m_Seeds.empty() && z < Zdim );

        // Sireni ze seminka
        while( !m_Seeds.empty() )
        {
            // Odeber prvni seminko
            Seed = m_Seeds.front();
            m_Seeds.pop_front();

            // Vyhledani regionu kam seminko patri
            SMyRegion& Region = m_Regions[Seed.m_iIndex];

            // Hodnota pixelu skryteho za seminkem
            iValue = (int)pSrcVolume->get(Seed.m_x, Seed.m_y, Seed.m_z);
            iAssignedIndex = (int)pDstVolume->get(Seed.m_x, Seed.m_y, Seed.m_z);

            // Je seminko neobarvene?
            if( iAssignedIndex == 0 )
            {
                // Diference stredni hodnoty a pixelu
                dDiff = ABS((double)iValue - Region.getMean());

                // Odezva LoG filtru
                dResp = (double)spLog->get(Seed.m_x, Seed.m_y, Seed.m_z);

                // Kombinace obou vlastnosti
                dResult = m_dDiffWeight * dDiff + m_dLogWeight * dResp;

                // Je vysledek mensi nez povoleny prah?
                if( dResult < m_dThreshold )
                {
                    // Pridani pixelu do regionu
                    Region.insert(iValue);
                    pDstVolume->set(Seed.m_x, Seed.m_y, Seed.m_z, (tVoxel)Seed.m_iIndex);

                    // Vytvoreni seminek ze sousedu
                    createSeeds(Seed, pDstVolume, Xdim, Ydim, Zdim);
                }
            }

            // Seminko narazilo na jiny region
            else if ( iAssignedIndex != Seed.m_iIndex )
            {
                // Vyhledani regionu kam patri
                SMyRegion& Region2 = m_Regions[iAssignedIndex];

                // Je-li rozdil regionu maly, sloucime je
                if( ABS(Region.getMean() - Region2.getMean()) < (m_dDiffWeight * m_dThreshold) )
                {
                    // Slouceni statistik
                    Region.add(Region2.m_dTotal, Region2.m_liCount);
                    Region2.m_dTotal = 0.0;
                    Region2.m_liCount = 0;

                    // Prebarveni
                    pDstVolume->replace((tVoxel)iAssignedIndex, (tVoxel)Seed.m_iIndex);
                }
            }
        }
    }

    // Phase 2: Pripadne spojeni "podobnych" regionu
    if( m_bMerge )
    {
        for( int i = 1; i <= iColor; ++i )
        {
            // Vyhledani regionu kam seminko patri
            SMyRegion& Region1 = m_Regions[i];

            for( int j = i + 1; j <= iColor; ++j )
            {
                // Vyhledani regionu kam seminko patri
                SMyRegion& Region2 = m_Regions[j];

                // Je-li rozdil regionu maly, sloucime je
                if( ABS(Region1.getMean() - Region2.getMean()) < (m_dDiffWeight * m_dThreshold) )
                {
                    // Slouceni statistik
                    Region1.add(Region2.m_dTotal, Region2.m_liCount);
                    Region2.m_dTotal = 0.0;
                    Region2.m_liCount = 0;

                    // Prebarveni
                    pDstVolume->replace((tVoxel)j, (tVoxel)i);
                }
            }
        }
    }

    // Phase 3: Serazeni indexu regionu a odstraneni prazdnych
    std::map<int, int> Indexes;
    Indexes[0] = 0;
    int iNewColor = 1;
    for( int i = 1; i <= iColor; ++i )
    {
        if( m_Regions[i].m_liCount > 1 )
        {
            Indexes[i] = iNewColor;
            ++iNewColor;
        }
    }

    // Preindexovani
    for( z = 0; z < Zdim; ++z )
    {
        for( y = 0; y < Ydim; ++y )
        {
            for( x = 0; x < Xdim; ++x )
            {
                pDstVolume->set(x, y, z, (tVoxel)Indexes[(int)pDstVolume->get(x, y, z)]);
            }
        }
    }

    // Vymazani seminek a regionu
    m_Regions.clear();
    m_Seeds.clear();

    return true;
}


} // namespace seg
} // namespace mds

