//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)    \n
 * Copyright (c) 2003-2004 by Michal Spanel     \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz  \n
 * File:    mdsSliceSubsampling.cpp             \n
 * Section: mSliceSubsampling                   \n
 * Date:    2004/08/19                          \n
 *
 * $Id: mdsSliceSubsampling.cpp 345 2007-06-11 13:23:09Z spanel $
 *
 * File description:
 * - Image subsampling using Gaussian pyramid.
 */

#include "mdsSliceSubsampling.h"

#include <MDSTk/Image/mdsSlice.h>
#include <MDSTk/Image/mdsPyramid.h>

// STL
#include <climits>
#include <iostream>
#include <string>


//==============================================================================
/*
 * Global constants.
 */

//! Module description
const std::string MODULE_DESCRIPTION    = "Module creates a subsampled image using Gaussian pyramid.";

//! Additional command line arguments
const std::string MODULE_ARGUMENTS      = "rate:levels";

// Additional arguments
const std::string MODULE_ARGUMENT_R     = "rate";
const std::string MODULE_ARGUMENT_L     = "levels";

// Default rate argument value
const unsigned int DEFAULT_R            = 2;
const unsigned int DEFAULT_L            = 2;


//==============================================================================
/*
 * Implementation of the class CSliceSubsampling.
 */
CSliceSubsampling::CSliceSubsampling(const std::string& sDescription)
    : mds::mod::CModule(sDescription)
{
    allowArguments(MODULE_ARGUMENTS);
}


CSliceSubsampling::~CSliceSubsampling()
{
}


bool CSliceSubsampling::startup()
{
    // Note
    MDS_LOG_NOTE("Module startup");

    // Test of existence of input and output channel
    if( getNumOfInputs() != 1 || getNumOfOutputs() != 1 )
    {
        MDS_CERR('<' << m_sFilename << "> Wrong number of input and output channels" << std::endl);
        return false;
    }

    // Parameters
    m_uRate = DEFAULT_R;
    m_Arguments.value(MODULE_ARGUMENT_R, m_uRate);
    if( m_uRate < 2 )
    {
        MDS_CERR('<' << m_sFilename << "> Bad " << MODULE_ARGUMENT_R << " parameter value: type -h for help" << std::endl);
        printUsage();
        return false;
    }

    m_uLevels = DEFAULT_L;
    m_Arguments.value(MODULE_ARGUMENT_L, m_uLevels);
    if( m_uLevels < 1 )
    {
        MDS_CERR('<' << m_sFilename << "> Bad " << MODULE_ARGUMENT_L << " parameter value: type -h for help" << std::endl);
        printUsage();
        return false;
    }

    // O.K.
    return true;
}


bool CSliceSubsampling::main()
{
    // Note
    MDS_LOG_NOTE("Module main function");

    // I/O channels
    mds::mod::CChannel *pIChannel = getInput(0);
    mds::mod::CChannel *pOChannel = getOutput(0);

    // Is any input?
    if( !pIChannel->isConnected() )
    {
        return false;
    }

    // Create a new slice
    mds::img::CSlicePtr spSlice;

    // Wait for data
    if( pIChannel->wait(1000) )
    {
        // Read slice from the input channel
        if( readInput(pIChannel, spSlice) )
        {
            // Create image pyramid using a specified subsampling rate
            mds::img::CDImagePyramidPtr spPyramid(new mds::img::CDImagePyramid((mds::tSize)m_uRate,
                                                                               (mds::tSize)m_uLevels,
                                                                               *spSlice
                                                                               ));

            // Create a new slice
            mds::img::CDImagePtr spNewSlice(spPyramid->getImagePtr(m_uLevels - 1));

            // Write it to the output channel
            if( !writeOutput(pOChannel, spNewSlice) )
            {
                MDS_CERR('<' << m_sFilename << "> Failed to write the output slice" << std::endl);
                return false;
            }
        }
        else
        {
            MDS_CERR('<' << m_sFilename << "> Failed to read input slice" << std::endl);
            return false;
        }
    }
    else
    {
        MDS_LOG_NOTE("Wait timeout");
    }

    // Returning 'true' means to continue processing the input channel
    return true;
}


void CSliceSubsampling::shutdown()
{
    // Note
    MDS_LOG_NOTE("Module shutdown");
}


void CSliceSubsampling::writeExtendedUsage(std::ostream& Stream)
{
    MDS_CERR(std::endl);
    MDS_CERR("Extended usage: [-rate uValue] [-levels uValue]" << std::endl);
    MDS_CERR("Options:" << std::endl);
    MDS_CERR("  -rate  Used subsampling rate." << std::endl);
    MDS_CERR("  -levels  The number of levels of the Gaussian pyramid." << std::endl);
}


//==============================================================================
/*
 * Function main() which creates and executes console application.
 */
int main(int argc, char *argv[])
{
    // Creation of a module using smart pointer
    CSliceSubsamplingPtr spModule(new CSliceSubsampling(MODULE_DESCRIPTION));

    // Initialize and execute the module
    if( spModule->init(argc, argv) )
    {
        spModule->run();
    }

    // Console application finished
    return 0;
}

