//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)             \n
 * Copyright (c) 2003-2005 by Michal Spanel              \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz           \n
 *          Martin Skotnica, xskotn01@stud.fit.vutbr.cz  \n
 * File:    mdsRegionGrowing.cpp                         \n
 * Section: mSliceSegRG                                  \n
 * Date:    2004/03/01                                   \n
 *
 * $Id: mdsRegionGrowing.cpp 345 2007-06-11 13:23:09Z spanel $
 *
 * Description:
 * - Simple segmentation method based on region growing and merging.
 */

#include "mdsRegionGrowing.h"

#include <MDSTk/Base/mdsAssert.h>
#include <MDSTk/Base/mdsGlobalLog.h>


namespace mds
{
namespace seg
{

//==============================================================================
/*
 * Implementation of the class mds::seg::CRegionGrowing.
 */
CRegionGrowing::CRegionGrowing(double dDiffWeight,
                               double dLogWeight,
                               double dThreshold,
                               bool bMerge
                               )
    : m_dThreshold(dThreshold)
    , m_dDiffWeight(dDiffWeight)
    , m_dLogWeight(dLogWeight)
    , m_bMerge(bMerge)
{
}


void CRegionGrowing::createSeeds(SMySeed& Seed,
                                 CRegionGrowing::tImage *pImage,
                                 int XDim,
                                 int YDim
                                 )
{
    SMySeed Temp(0, 0, Seed.m_iIndex);
    if( Seed.m_x > 0 )
    {
        Temp.m_x = Seed.m_x - 1;
        Temp.m_y = Seed.m_y;
        if( (int)pImage->get(Temp.m_x, Temp.m_y) != Temp.m_iIndex )
        {
            m_Seeds.push_front(Temp);
        }
    }
    if( Seed.m_y > 0 )
    {
        Temp.m_x = Seed.m_x;
        Temp.m_y = Seed.m_y - 1;
        if( (int)pImage->get(Temp.m_x, Temp.m_y) != Temp.m_iIndex )
        {
            m_Seeds.push_front(Temp);
        }
    }
    if( Seed.m_x < XDim - 1 )
    {
        Temp.m_x = Seed.m_x + 1;
        Temp.m_y = Seed.m_y;
        if( (int)pImage->get(Temp.m_x, Temp.m_y) != Temp.m_iIndex )
        {
            m_Seeds.push_front(Temp);
        }
    }
    if( Seed.m_y < YDim - 1 )
    {
        Temp.m_x = Seed.m_x;
        Temp.m_y = Seed.m_y + 1;
        if( (int)pImage->get(Temp.m_x, Temp.m_y) != Temp.m_iIndex )
        {
            m_Seeds.push_front(Temp);
        }
    }
}


bool CRegionGrowing::operator()(const CRegionGrowing::tImage *pSrcImage,
                                CRegionGrowing::tImage *pDstImage
                                )
{
    // Image dimensions
    tSize Xdim = pSrcImage->getXSize();
    tSize Ydim = pSrcImage->getYSize();

    // Check destination image dimensions
    MDS_CHECK(Xdim == pDstImage->getXSize() && Ydim == pDstImage->getYSize(), return false);
    MDS_CHECK(Xdim > 0 && Ydim > 0, return false);

    // Vyplni vystup cernou barvnou
    pDstImage->fill(0);

    // Vypocet odezvy Laplacianu filtru
    tImage::tSmartPtr spLog(new tImage(Xdim, Ydim));
    mds::img::CImageEdgeDetector<tImage, mds::img::IED_CANNY> Filter(1.0, 0.2, 0.1);
    Filter(*pSrcImage, *spLog);

    // Zruseni seminek
    m_Seeds.clear();

    // Cislovani regionu musi zacinat od 1
    m_Regions.clear();
    m_Regions.push_back(SMyRegion(0));

    SMySeed Seed;
    int iValue, iAssignedIndex;
    double dDiff, dResp, dResult;

    // Inicializace seminka
    int iColor = 0;
    tSize x = 0, y = 0;

    while( y < Ydim )
    {
        // Vyhledani seminka
        do {
            if( (int)pDstImage->get(x, y) == 0 )
            {
                m_Regions.push_back(SMyRegion((long)pSrcImage->get(x, y)));
                m_Seeds.push_back(SMySeed(x, y, ++iColor));
            }
            else
            {
                ++x;
                if( x >= Xdim )
                {
                    x = 0;
                    ++y;
                }
            }
        } while( m_Seeds.empty() && y < Ydim );

        // Sireni ze seminka
        while( !m_Seeds.empty() )
        {
            // Odeber prvni seminko
            Seed = m_Seeds.front();
            m_Seeds.pop_front();

            // Vyhledani regionu kam seminko patri
            SMyRegion& Region = m_Regions[Seed.m_iIndex];

            // Hodnota pixelu skryteho za seminkem
            iValue = (int)pSrcImage->get(Seed.m_x, Seed.m_y);
            iAssignedIndex = (int)pDstImage->get(Seed.m_x, Seed.m_y);

            // Je seminko neobarvene?
            if( iAssignedIndex == 0 )
            {
                // Diference stredni hodnoty a pixelu
                dDiff = ABS((double)iValue - Region.getMean());

                // Odezva LoG filtru
                dResp = (double)spLog->get(Seed.m_x, Seed.m_y);

                // Kombinace obou vlastnosti
                dResult = m_dDiffWeight * dDiff + m_dLogWeight * dResp;

                // Je vysledek mensi nez povoleny prah?
                if( dResult < m_dThreshold )
                {
                    // Pridani pixelu do regionu
                    Region.insert(iValue);
                    pDstImage->set(Seed.m_x, Seed.m_y, (tPixel)Seed.m_iIndex);

                    // Vytvoreni seminek ze sousedu
                    createSeeds(Seed, pDstImage, Xdim, Ydim);
                }
            }

            // Seminko narazilo na jiny region
            else if ( iAssignedIndex != Seed.m_iIndex )
            {
                // Vyhledani regionu kam patri
                SMyRegion& Region2 = m_Regions[iAssignedIndex];

                // Je-li rozdil regionu maly, sloucime je
                if( ABS(Region.getMean() - Region2.getMean()) < (m_dDiffWeight * m_dThreshold) )
                {
                    // Slouceni statistik
                    Region.add(Region2.m_dTotal, Region2.m_liCount);
                    Region2.m_dTotal = 0.0;
                    Region2.m_liCount = 0;

                    // Prebarveni
                    pDstImage->replace((tPixel)iAssignedIndex, (tPixel)Seed.m_iIndex);
                }
            }
        }
    }

    // Phase 2: Pripadne spojeni "podobnych" regionu
    if( m_bMerge )
    {
        for( int i = 1; i <= iColor; ++i )
        {
            // Vyhledani regionu kam seminko patri
            SMyRegion& Region1 = m_Regions[i];

            for( int j = i + 1; j <= iColor; ++j )
            {
                // Vyhledani regionu kam seminko patri
                SMyRegion& Region2 = m_Regions[j];

                // Je-li rozdil regionu maly, sloucime je
                if( ABS(Region1.getMean() - Region2.getMean()) < (m_dDiffWeight * m_dThreshold) )
                {
                    // Slouceni statistik
                    Region1.add(Region2.m_dTotal, Region2.m_liCount);
                    Region2.m_dTotal = 0.0;
                    Region2.m_liCount = 0;

                    // Prebarveni
                    pDstImage->replace((tPixel)j, (tPixel)i);
                }
            }
        }
    }

    // Phase 3: Serazeni indexu regionu a odstraneni prazdnych
    std::map<int, int> Indexes;
    Indexes[0] = 0;
    int iNewColor = 1;
    for( int i = 1; i <= iColor; ++i )
    {
        if( m_Regions[i].m_liCount > 1 )
        {
            Indexes[i] = iNewColor;
            ++iNewColor;
        }
    }

    // Preindexovani
    for( y = 0; y < Ydim; ++y )
    {
        for( x = 0; x < Xdim; ++x )
        {
            pDstImage->set(x, y, (tPixel)Indexes[(int)pDstImage->get(x, y)]);
        }
    }

    // Vymazani seminek a regionu
    m_Regions.clear();
    m_Seeds.clear();

    return true;
}


} // namespace seg
} // namespace mds

