//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)               \n
 * Copyright (c) 2003-2005 by Michal Spanel                \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz             \n
 *          Michal Rozsypalek, xrozsy02@stud.fit.vutbr.cz  \n
 * File:    mdsFuzzyCMeans.h                               \n
 * Section: mSliceSegFCM                                   \n
 * Date:    2005/10/14                                     \n
 *
 * $Id: mdsFuzzyCMeans.h 92 2006-09-24 09:53:39Z spanel $
 *
 * Description:
 * - Fuzzy C-means (FCM) image segmentation algorithm.
 */

#ifndef MDS_FUZZYCMEANS_H
#define MDS_FUZZYCMEANS_H

#include <MDSTk/Base/mdsSetup.h>
#include <MDSTk/Math/mdsBase.h>
#include <MDSTk/Math/mdsRandom.h>
#include <MDSTk/Math/mdsVector.h>
#include <MDSTk/Math/mdsMatrix.h>
#include <MDSTk/Image/mdsImage.h>
#include <MDSTk/Image/mdsImageHistogram.h>


namespace mds
{
namespace seg
{

//==============================================================================
/*
 * Global definitions.
 */

//! If this macro is defined, various informations are logged.
//#ifndef FCM_LOGGING_ENABLED
//#    define FCM_LOGGING_ENABLED
//#endif


//==============================================================================
/*!
 * Class providing Fuzzy C-means (FCM) image segmentation functions.
 */
class CImageFuzzyCMeans
{
public:
    //! Image type.
    typedef mds::img::CDImage tImage;

    //! Image pixel type.
    typedef mds::img::CDImage::tPixel tPixel;

    //! Constant used to enable automatic estimation of the number
    //! of clusters by FCM algorithm.
    enum { UNKNOWN = 0 };

    //! Default weighting factor (>1).
    static const double DEFAULT_WEIGHT;

    //! Minimal required change of the objective function.
    static const double MIN_CHANGE;

    //! Maximal allowed number of iterations.
    static const tSize MAX_ITERS;

public:
    //! Constructor.
    CImageFuzzyCMeans(tSize NumOfClusters = UNKNOWN,
                      double dWeight = DEFAULT_WEIGHT
                      );

    //! Virtual destructor.
    virtual ~CImageFuzzyCMeans() {}

    //! Segmentation of a given image.
    //! - Uses mds::img::CDImage image type.
    //! - Return false on failure.
    bool operator()(const tImage& SrcImage, tImage& DstImage);

protected:
    //! Vector of cluster centers.
    typedef mds::math::CVector<double> tCenters;

    //! Membership matrix.
    typedef mds::math::CMatrix<double> tMembership;

protected:
    //! The number of clusters.
    tSize m_NumOfClusters;

    //! Weighting factor.
    double m_dWeight;

    //! Allowed pixel values.
    tPixel m_PixelMin, m_PixelMax;

    //! Histogram size.
    tSize m_Span;

    //! Helper variables.
    double m_dExponent, m_dInvSpan;

    //! Cluster centers.
    tCenters m_Centers;

    //! Membership matrix.
    tMembership m_Membership;

    //! Helper matrix.
    tMembership m_Powers;

    //! Uniform random number generator.
    mds::math::CUniformPRNG m_Uniform;

    //! Image histogram.
    mds::img::CDImageHistogram m_Histogram;

protected:
    //! Randomly initializes the membership matrix.
    void initMembership();

    //! Checks if the membership matrix is O.K.
    bool checkMembership();

    //! Returns membership of a given pixel in i-th cluster.
    double getMembership(double dValue, tSize i);

    //! Re-computes membership for every pixel.
    void recomputeMembership();

    //! Recomputes matrix of powers of the membership function.
    void recomputePowers();

    //! Compute new positions of cluster centers.
    void recomputeClusterCenters();

    //! Returns value of the objective function.
    double recomputeObjectiveFunction();

    //! Computes the Dunn's partitions coefficient.
    double computeDunnCoefficient();

    //! Computes FCM of a given image.
    void iterateFCM(double dMinChange);

    //! Segments a given input image.
    void segmentImage(const tImage& SrcImage,
                      tImage& DstImage,
                      tSize XSize,
                      tSize YSize
                      );
};


} // namespace seg
} // namespace mds

#endif // MDS_FUZZYCMEANS_H

