//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)               \n
 * Copyright (c) 2003-2005 by Michal Spanel                \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz             \n
 *          Michal Rozsypalek, xrozsy02@stud.fit.vutbr.cz  \n
 * File:    mdsFuzzyCMeans.cpp                             \n
 * Section: mSliceSegFCM                                   \n
 * Date:    2005/10/14                                     \n
 *
 * $Id: mdsFuzzyCMeans.cpp 345 2007-06-11 13:23:09Z spanel $
 *
 * Description:
 * - Fuzzy C-means (FCM) image segmentation algorithm.
 */

#include "mdsFuzzyCMeans.h"

#include <MDSTk/Base/mdsGlobalLog.h>
#include <MDSTk/Math/mdsMatrixFunctions.h>
#include <MDSTk/Image/mdsImageFunctions.h>

#include <float.h>
#include <cmath>


namespace mds
{
namespace seg
{

//==============================================================================
/*
 * Implementation of the CImageFuzzyCMeans class.
 */

//const double CImageFuzzyCMeans::DEFAULT_WEIGHT      = 2;
const double CImageFuzzyCMeans::DEFAULT_WEIGHT      = 1.5;
const double CImageFuzzyCMeans::MIN_CHANGE          = 1.0e-6;
const tSize CImageFuzzyCMeans::MAX_ITERS            = 100;


CImageFuzzyCMeans::CImageFuzzyCMeans(tSize NumOfClusters, double dWeight)
    : m_NumOfClusters(NumOfClusters)
    , m_dWeight(dWeight)
    , m_PixelMin(mds::img::CPixelTraits<tPixel>::getPixelMin())
    , m_PixelMax(mds::img::CPixelTraits<tPixel>::getPixelMax())
    , m_Span(tSize(m_PixelMax - m_PixelMin + 1))
    , m_Centers(NumOfClusters)
    , m_Membership(NumOfClusters, m_Span)
    , m_Powers(NumOfClusters, m_Span)
    , m_Histogram(m_PixelMin, m_PixelMax)
{
    MDS_ASSERT(m_NumOfClusters >= 0 && m_dWeight >= 1.0);

    m_dExponent = 2.0 / (m_dWeight - 1.0);
    m_dInvSpan = 1.0 / m_Span;
}


bool CImageFuzzyCMeans::operator()(const tImage& SrcImage, tImage& DstImage)
{
    // Get the effective size
    tSize XCount = MIN(SrcImage.getXSize(), DstImage.getXSize());
    tSize YCount = MIN(SrcImage.getYSize(), DstImage.getYSize());

    // Compute histogram of input image
    m_Histogram(SrcImage);

    // Is the number of clusters known?
    if( m_NumOfClusters > 0 )
    {
        // Resize the membership matrix
        m_Membership.create(m_NumOfClusters, m_Span);
        m_Powers.create(m_NumOfClusters, m_Span);

        // Resize the vector of cluster centers
        m_Centers.create(m_NumOfClusters);

        // Compute the FCM
        iterateFCM(MIN_CHANGE);

        // Final segmentation
        segmentImage(SrcImage, DstImage, XCount, YCount);

        // O.K.
        return true;
    }

    // Initial Dunn's coefficient
    double dDunnCoeff = 0.0;

    // Test various numbers of clusters
    m_NumOfClusters = 1;
    for( ;; )
    {
        // Resize the membership matrix
        m_Membership.create(m_NumOfClusters, m_Span);
        m_Powers.create(m_NumOfClusters, m_Span);

        // Resize the vector of cluster centers
        m_Centers.create(m_NumOfClusters);

        // Compute the FCM
        iterateFCM(MIN_CHANGE);

        // Compute the Dunn's coefficient
        double dNewValue = computeDunnCoefficient();

        // Estimate changes
        if( dNewValue < dDunnCoeff || dNewValue > 1.0 )
        {
            break;
        }

        // Image segmentation
        segmentImage(SrcImage, DstImage, XCount, YCount);

        // Update current Dunn's coefficient
        dDunnCoeff = dNewValue;

        // Increment the number of clusters
        ++m_NumOfClusters;
    }

    // O.K.
    return true;
}


void CImageFuzzyCMeans::iterateFCM(double dMinChange)
{
    // Random membership function
    initMembership();

    // Initial value of the objective function
    double dObjectiveFunc = 1.0;

    // Iterate while the function converges
    for( int i = 0; i < MAX_ITERS; i++ )
    {
        // Recompute matrix of membership powers
        recomputePowers();

        // Update cluster centers
        recomputeClusterCenters();

#ifdef FCM_LOGGING_ENABLED
        MDS_LOG_NOTE("iterateFCM()");
        for( tSize k = 0; k < m_NumOfClusters; ++k )
        {
            MDS_LOG_NOTE("  Cluster " << k << ": " << m_Centers(k));
        }
#endif // FCM_LOGGING_ENABLED

        // Update the membership matrix
        recomputeMembership();

        // Evaluate the objective function
        double dNewValue = recomputeObjectiveFunction();

        // Estimate change of the objective function
        double dDelta = mds::math::getAbs(dNewValue / dObjectiveFunc - 1.0);

#ifdef FCM_LOGGING_ENABLED
        MDS_LOG_NOTE("iterateFCM()");
        MDS_LOG_NOTE("  Objective Function = " << dNewValue);
        MDS_LOG_NOTE("  Delta = " << dDelta);
#endif // FCM_LOGGING_ENABLED

        // Estimate changes
        if( dDelta < dMinChange || dNewValue < dMinChange )
        {
            break;
        }

        // Update the current value
        dObjectiveFunc = dNewValue;
    }
}


void CImageFuzzyCMeans::initMembership()
{
    MDS_ASSERT(m_NumOfClusters == m_Membership.getNumOfRows());

    m_Membership.zeros();

    for( tSize j = 0; j < m_Span; ++j )
    {
        tSize Index = mds::math::round2Int(m_Uniform.random(0, m_NumOfClusters - 1));
        m_Membership(Index, j) = 1;
    }
}


bool CImageFuzzyCMeans::checkMembership()
{
    double dTemp = mds::math::getSum<double>(m_Membership) * m_dInvSpan;

    return (mds::math::getAbs(dTemp - 1.0) < 0.001);
}


double CImageFuzzyCMeans::getMembership(double dValue, tSize i)
{
    double dCenter = m_Centers(i);
    double dNumerator = mds::math::getAbs(dCenter - dValue);

    double dMembership = 0.0;
    for( tSize j = 0; j < m_NumOfClusters; ++j )
    {
        double dTemp = dNumerator / mds::math::getAbs(m_Centers(j) - dValue);
        dMembership += pow(dTemp, m_dExponent);
    }

    return (dMembership > 0.0) ? 1.0 / dMembership : 1.0;
}


void CImageFuzzyCMeans::recomputeMembership()
{
    for( tSize i = 0; i < m_NumOfClusters; ++i )
    {
        for( tSize j = 0; j < m_Span; ++j )
        {
            m_Membership(i, j) = getMembership(m_Histogram.getLowerBound(j), i);
        }
    }
}


void CImageFuzzyCMeans::recomputePowers()
{
    for( tSize i = 0; i < m_NumOfClusters; ++i )
    {
        for( tSize j = 0; j < m_Span; ++j )
        {
            m_Powers(i, j) = pow(m_Membership(i, j), m_dWeight);
        }
    }
}


void CImageFuzzyCMeans::recomputeClusterCenters()
{
    for( tSize i = 0; i < m_NumOfClusters; ++i )
    {
        m_Centers(i) = 0.0;
        double dDenom = 0.0;

        for( tSize j = 0; j < m_Span; ++j )
        {
            double dTemp = m_Histogram.getCount(j) * m_Powers(i, j);
            m_Centers(i) += dTemp * m_Histogram.getLowerBound(j);
            dDenom += dTemp;
        }

        if( dDenom > 0.0 )
        {
            m_Centers(i) /= dDenom;
        }
    }
}


double CImageFuzzyCMeans::recomputeObjectiveFunction()
{
    double dResult = 0.0;

    for( tSize i = 0; i < m_NumOfClusters; ++i )
    {
        for( tSize j = 0; j < m_Span; ++j )
        {
            double dTemp = m_Centers(i) - m_Histogram.getLowerBound(j);
            dResult += m_Histogram.getCount(j) * m_Powers(i, j) * dTemp * dTemp;
        }
    }

    return dResult;
}


double CImageFuzzyCMeans::computeDunnCoefficient()
{
    double dInvNumOfClusters = 1.0 / m_Membership.getNumOfRows();
    double dInvNumOfSamples = 1.0 / m_Membership.getNumOfCols();

    double dCoefficient = 0.0;
    for( tSize i = 0; i < m_Membership.getNumOfRows(); ++i )
    {
        for( tSize j = 0; j < m_Membership.getNumOfCols(); ++j )
        {
            dCoefficient += m_Membership(i, j) * m_Membership(i, j) * dInvNumOfSamples;
        }
    }

    return (dCoefficient - dInvNumOfClusters) / (1.0 - dInvNumOfClusters + 0.001);
}


void CImageFuzzyCMeans::segmentImage(const tImage& SrcImage,
                                     tImage& DstImage,
                                     tSize XSize,
                                     tSize YSize
                                     )
{
    // Classify input image pixels
    for( tSize y = 0; y < YSize; ++y )
    {
        for( tSize x = 0; x < XSize; ++x )
        {
            // Get index of the histogram bin
            tSize Index = m_Histogram.getIndex(SrcImage(x, y));

            // Find cluster whose membership function is maximal
            tSize Max = 0;
            for( tSize i = 1; i < m_NumOfClusters; ++i )
            {
                if( m_Membership(i, Index) > m_Membership(Max, Index) )
                {
                    Max = i;
                }
            }
            DstImage(x, y) = tPixel(Max);
        }
    }
}


} // namespace seg
} // namespace mds

