//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)    \n
 * Copyright (c) 2003-2005 by Michal Spanel     \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz  \n
 * File:    mdsEMAlgorithm.h                    \n
 * Section: mSliceSegEM                         \n
 * Date:    2005/11/02                          \n
 *
 * $Id: mdsEMAlgorithm.h 92 2006-09-24 09:53:39Z spanel $
 *
 * Description:
 * - Image segmentation algorithm based on gaussian mixture model
 *   and Expectation-Maximization (EM) algorithm.
 */

#ifndef MDS_EMALGORITHM_H
#define MDS_EMALGORITHM_H

#include <MDSTk/Base/mdsSetup.h>
#include <MDSTk/Base/mdsGlobalLog.h>
#include <MDSTk/Math/mdsBase.h>
#include <MDSTk/Math/mdsRandom.h>
#include <MDSTk/Math/mdsVector.h>
#include <MDSTk/Math/mdsMatrix.h>
#include <MDSTk/Image/mdsImage.h>
#include <MDSTk/Image/mdsImageHistogram.h>

#include <cmath>


namespace mds
{
namespace seg
{

//==============================================================================
/*
 * Global definitions.
 */

//! If this macro is defined, various informations are logged.
//#ifndef EM_LOGGING_ENABLED
//#    define EM_LOGGING_ENABLED
//#endif


//==============================================================================
/*!
 * Gaussian mixture model component.
 * - 1D gaussian function.
 */
class CGaussianFunc1D
{
public:
    //! Default constructor.
    CGaussianFunc1D() {}

    //! Constructor.
    CGaussianFunc1D(double dWeight,  double dMean, double dSigma)
        : m_dWeight(dWeight)
        , m_dMean(dMean)
    {
        setSigma(dSigma);
    }

    //! Returns weight of the mixture component.
    double getWeight() { return m_dWeight; }

    //! Sets weight of the component.
    void setWeight(double dWeight) { m_dWeight = dWeight; }

    //! Returns mean value.
    double getMean() { return m_dMean; }

    //! Sets mean value of the gaussian function.
    void setMean(double dMean) { m_dMean = dMean; }

    //! Returns sigma parameter (standard deviation) of the gaussian function.
    double getSigma() { return m_dSigma; }

    //! Sets standard deviation of the gaussian function.
    void setSigma(double dSigma)
    {
        m_dSigma = (mds::math::getAbs(dSigma) > m_dMinSigma) ? dSigma : m_dMinSigma;
    }

    //! Sets minimal allowed value of the standard deviation.
    static void setMinSigma(double dSigma)
    {
        m_dMinSigma = mds::math::getAbs(dSigma);
    }


    //! Returns value of the 1D gaussian function.
    double getValue(double dX)
    {
        double dTemp = dX - m_dMean;
        double dExp = -0.5 * dTemp * dTemp / (m_dSigma * m_dSigma);
        double dResult = 1.0 / (mds::math::SQRT_TWO_PI * m_dSigma) * exp(dExp);

        return dResult;
    }

    //! Returns weighted value of the 1D gaussian function.
    double getWeightedValue(double dX)
    {
        return m_dWeight * getValue(dX);
    }

protected:
    //! Minimal allowed gaussian standard deviation.
    static double m_dMinSigma;

    //! Parameters of the gaussian function.
    double m_dWeight, m_dMean, m_dSigma;
};


//==============================================================================
/*!
 * Class providing image segmentation based on the gaussian mixture model
 * and EM algorithm.
 */
class CImageEMAlgorithm
{
public:
    //! Image type.
    typedef mds::img::CDImage tImage;

    //! Image pixel type.
    typedef mds::img::CDImage::tPixel tPixel;

    //! Constant used to enable automatic estimation of the number
    //! of segments by EM algorithm.
    enum { UNKNOWN = 0 };

    //! Default minimal allowed gaussian standard deviation.
    static const double DEFAULT_MIN_SIGMA;

    //! Minimal required change of the log-likelihood function
    //! which results in component addition.
    static const double DEFAULT_MIN_ADD_CHANGE;

    //! Minimal required change of the log-likelihood function
    //! while iterating.
    static const double MIN_CHANGE;

    //! Coefficient used for gaussian component splitting.
    static const double SPLITTING_COEFF;

    //! Maximal allowed number of iterations.
    static const tSize MAX_ITERS;

    //! Maximal allowed number of iterations.
    static const tSize MAX_ITERS2;

public:
    //! Constructor.
    CImageEMAlgorithm(tSize NumOfComponents = UNKNOWN,
                      double dMinAddChange = DEFAULT_MIN_ADD_CHANGE
                      );

    //! Virtual destructor.
    virtual ~CImageEMAlgorithm() {}

    //! Segmentation of a given image.
    //! - Uses mds::img::CDImage image type.
    //! - Return false on failure.
    bool operator()(const tImage& SrcImage, tImage& DstImage);

protected:
    //! Gaussian functions representing image components/segments.
    typedef mds::math::CVector<CGaussianFunc1D> tComponents;

    //! Support maps.
    typedef mds::math::CMatrix<double> tMaps;

protected:
    //! Number of dimensions of the feature vector.
    enum { NUM_OF_DIMENSIONS = 1 };

protected:
    //! Minimal required change of the log-likelihood function
    //! which results in component addition.
    double m_dMinAddChange;

    //! The number of mixture model components.
    tSize m_NumOfComponents;

    //! Allowed pixel values.
    tPixel m_PixelMin, m_PixelMax;

    //! Histogram size.
    tSize m_Span;

    //! The number of components in the histogram.
    tSize m_Count;

    //! Helper variables.
    double m_dInvSpan, m_dInvCount;

    //! Image segments.
    tComponents m_Components;

    //! Support maps.
    tMaps m_Maps;

    //! Uniform random number generator.
    mds::math::CUniformPRNG m_Uniform;

    //! Image histogram.
    mds::img::CDImageHistogram m_Histogram;

protected:
    //! Clears all support maps.
    void clearSupportMaps();

    //! Random intialization of a component mean value.
    void initComponentMean(tSize i,
                           const tImage& SrcImage,
                           tSize XSize,
                           tSize YSize
                           );

    //! Intialization of a component sigma.
    void initComponentSigma(tSize i);

    //! Intialization of a component weight.
    void initComponentWeight(tSize i, double dWeight);

    //! Initialization of the first mixture component.
    void initFirstComponent();

    //! Random intialization of all components.
    void initComponents(const tImage& SrcImage, tSize XSize, tSize YSize);

    //! Returns value of the objective function.
    double computeLogLikelihood();

    //! E-step of the algorithm.
    void processEStep();

    //! M-step of the algorithm.
    void processMStep();

    //! Iterates both E-step and M-step.
    //! - Returns final value of the log-likelihood function
    double iterateEM(double dMinChange);

    //! Checks degree of separation of two given components.
    bool areSeparated(tSize i, tSize j, int c);

    //! Returns degree of separation.
    int getDegreeOfSeparation(tSize NumOfComponents);

    //! Finds component which can be further splitted.
    //! - Returns index of such componnet.
    tSize findComponent();

    //! Splits a given component.
    void splitComponent(tSize Index, tSize NewIndex);

    //! Checks for similar components.
    //! - If any two components are similar, one of them is reinitialized.
    bool checkForSimilarComponents();

    //! Segments a given input image.
    void segmentImage(const tImage& SrcImage,
                      tImage& DstImage,
                      tSize XSize,
                      tSize YSize
                      );
};


} // namespace seg
} // namespace mds

#endif // MDS_EMALGORITHM_H

