//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)    \n
 * Copyright (c) 2003-2007 by Michal Spanel     \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz  \n
 * File:    mdsRGBImageView.cpp                 \n
 * Section: mRGBImageView                       \n
 * Date:    2007/06/13                          \n
 *
 * $Id:$
 *
 * File description:
 * - Reads RGB image from the input channel and draws it using GLUT and OpenGL.
 */

#include "mdsRGBImageView.h"

// OpenGL
#include <GL/gl.h>
#include <GL/glut.h>

// STL
#include <iostream>
#include <string>


//==============================================================================
/*
 * Global constants.
 */

//! GLUT window title
const char GLUT_WINDOW_TITLE[]              = "MDSTk Image View Window";

//! Module description
const std::string MODULE_DESCRIPTION        = "Module draws input RGB image using GLUT and OpenGL";

//! Additional command line arguments
const std::string MODULE_ARGUMENTS          = "";

//! Additional arguments

//! Default GLUT window position
const int GLUT_WINDOW_XPOS                  = 100;
const int GLUT_WINDOW_YPOS                  = 100;

//! Default GLUT window size
int GLUT_WINDOW_XSIZE                       = 512;
int GLUT_WINDOW_YSIZE                       = 512;

//! Maximum pixel value
const unsigned int PIXEL_MAX                = mds::img::CPixelTraits<mds::img::tPixel8>::getPixelMax();


//==============================================================================
/*
 * Global variables.
 */

//! Current GLUT window size
int iGlutWindowXSize                        = GLUT_WINDOW_XSIZE;
int iGlutWindowYSize                        = GLUT_WINDOW_YSIZE;

//! Texture numbers
GLuint uGLTexture;

//! Size of the used texture part
GLfloat fGLTextureXSize;
GLfloat fGLTextureYSize;


//==============================================================================
/*
 * Global functions.
 */

//! Texture creation
void createTexture(mds::img::CRGBImage& RGBImage);


//==============================================================================
/*
 * GLUT functions.
 */

//! Callback functions
void onInit();
void onResize(int iWidth, int iHeight);
void onDisplay(void);
void onKeyboard(unsigned char ucKey, int iX, int iY);


//==============================================================================
/*
 * Implementation of the class CRGBImageView.
 */
CRGBImageView *CRGBImageView::m_pView = NULL;


CRGBImageView::CRGBImageView(const std::string& sDescription)
    : mds::mod::CView(sDescription)
{
    MDS_ASSERT(m_pView == NULL);
    m_pView = this;

    allowArguments(MODULE_ARGUMENTS);
}


CRGBImageView::~CRGBImageView()
{
}


bool CRGBImageView::startup()
{
    // Note
    MDS_LOG_NOTE("Module startup");

    // Test of existence of input channel
    if( getNumOfInputs() < 1 || getNumOfInputs() > 1 )
    {
        MDS_CERR('<' << m_sFilename << "> Wrong number of input channels" << std::endl);
        return false;
    }

    // O.K.
    return true;
}


bool CRGBImageView::main()
{
    // Note
    MDS_LOG_NOTE("Module main function");

    // Input channel
    mds::mod::CChannel *pIChannel = getInput(0);

    // Is any input?
    if( !pIChannel->isConnected() )
    {
        return false;
    }

    // Create a new RGB image
    mds::img::CRGBImagePtr spImage;

    // Wait for data
    bool bImageRead = false;
    if( pIChannel->wait(1000) )
    {
        // Read the slice from input channel
        if( readInput(pIChannel, m_spRGBImage) )
        {
            bImageRead = true;
        }
        else
        {
            MDS_CERR('<' << m_sFilename << "> Failed to read input image" << std::endl);
            return false;
        }
    }
    else
    {
        MDS_LOG_NOTE("Wait timeout");
    }

    // Draw the read slices
    if( bImageRead )
    {
        // Initilialize the GLUT library
        glutInitDisplayMode(GLUT_RGB | GLUT_DOUBLE | GLUT_DEPTH);
        glutCreateWindow(GLUT_WINDOW_TITLE);
        glutDisplayFunc(onDisplay);
        glutReshapeFunc(onResize);
        glutKeyboardFunc(onKeyboard);

        // Initalize module
        onInit();

        // Run GLUT event processing loop
        glutMainLoop();
    }

    // Do not continue
    return false;
}


void CRGBImageView::shutdown()
{
    // Note
    MDS_LOG_NOTE("Module shutdown");
}


void CRGBImageView::writeExtendedUsage(std::ostream& Stream)
{
    MDS_CERR(std::endl);
    MDS_CERR("Extended usage:" << std::endl);
    MDS_CERR("Options:" << std::endl);
}


//==============================================================================
/*
 * Function main() which creates and executes view application.
 */
int main(int argc, char *argv[])
{
    // Creation of a view module using smart pointer
    CRGBImageViewPtr spView(new CRGBImageView(MODULE_DESCRIPTION));

    // Initialize GLUT library
    glutInit(&argc, argv);

    // Initialize and run the view
    if( spView->init(argc, argv) )
    {
        spView->run();
    }

    // Console application finished
    return 0;
}


//==============================================================================
/*
 * GLUT functions implementation.
 */

// Creation of a GL RGB texture
void createTexture(mds::img::CRGBImage *pRGBImage)
{
    // Set texture image parameters
    glBindTexture(GL_TEXTURE_2D, uGLTexture);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);

    // Image size
    mds::tSize XSize = pRGBImage->getXSize();
    mds::tSize YSize = pRGBImage->getYSize();

    // Texture dimensions
    mds::tSize TexXSize = 2, TexYSize = 2;
    while( TexXSize < XSize || TexYSize < YSize )
    {
        TexXSize *= 2;
        TexYSize *= 2;
    }

    // Allowed maximal texture dimension
    int iMaxTexDim;
    glGetIntegerv(GL_MAX_TEXTURE_SIZE, &iMaxTexDim);

    // Modify the image dimension
    if( TexXSize > iMaxTexDim )
    {
        XSize = TexXSize = iMaxTexDim;
        YSize = TexYSize = iMaxTexDim;
    }

    // Used texture size
    fGLTextureXSize = (GLfloat)XSize / (GLfloat)TexXSize;
    fGLTextureYSize = (GLfloat)YSize / (GLfloat)TexYSize;

    // Allocate texture RGBA data
    unsigned char *pucRGBTexture = new unsigned char[TexXSize * TexYSize * 3];
    memset(pucRGBTexture, 0, TexXSize * TexYSize * 3);

    // Create required texture
    for( mds::tSize j = 0; j < YSize; ++j )
    {
        unsigned char *p = &pucRGBTexture[j * TexXSize * 3];
        for( mds::tSize i = 0; i < XSize; ++i )
        {
            mds::img::tRGBPixel Value = pRGBImage->get(i,j);
            *(p++) = Value.getRed();
            *(p++) = Value.getGreen();
            *(p++) = Value.getBlue();
        }
    }

    // Define 2D texture image
    glTexImage2D(GL_TEXTURE_2D, 0, 3,
                 TexXSize,
                 TexYSize,
                 0, GL_RGB, GL_UNSIGNED_BYTE,
                 pucRGBTexture
                 );

    // Free the memory
    delete[] pucRGBTexture;
}


// GLUT and texture initialization
void onInit()
{
    // Set clear color
    glClearColor(0.0f, 0.0f, 0.0f, 0.0f);

    // Enable z-buffer
    glClearDepth(1.0f);
    glEnable(GL_DEPTH_TEST);
    glDepthFunc(GL_LESS);

    // Enable antialiasing
    glEnable(GL_POINT_SMOOTH);
    glEnable(GL_LINE_SMOOTH);
    glEnable(GL_POLYGON_SMOOTH);

    // Enable the polygon filling
    glPolygonMode(GL_FRONT, GL_FILL);
    glPolygonMode(GL_BACK, GL_FILL);

    // Turn the culling off
    glDisable(GL_CULL_FACE);

    // Set global texture image parameters
    glPixelStorei(GL_UNPACK_ALIGNMENT, 1);

    // Make performace hints
    glHint(GL_PERSPECTIVE_CORRECTION_HINT, GL_NICEST);

    // Generate texture names
    glGenTextures(1, &uGLTexture);

    // RGBImage size
    int XSize = (int)CRGBImageView::getViewPtr()->getRGBImage()->getXSize();
    int YSize = (int)CRGBImageView::getViewPtr()->getRGBImage()->getYSize();

    // Modify glut window size to keep image dimensions ratio
    if( XSize > YSize )
    {
        GLUT_WINDOW_YSIZE = GLUT_WINDOW_XSIZE * YSize / XSize;
    }
    else
    {
        GLUT_WINDOW_XSIZE = GLUT_WINDOW_YSIZE * XSize / YSize;
    }

    // Create the background polygon texture
    createTexture(CRGBImageView::getViewPtr()->getRGBImage());

    // Initialize glut window
    glutPositionWindow(GLUT_WINDOW_XPOS, GLUT_WINDOW_YPOS);
    glutReshapeWindow(GLUT_WINDOW_XSIZE, GLUT_WINDOW_YSIZE);
}


// Sets coordinate system depending on the window size
void onResize(int iWidth, int iHeight)
{
    iGlutWindowXSize = iWidth;
    iGlutWindowYSize = iHeight;

    // Set the viewport
    glViewport(0, 0, iWidth, iHeight);

    // Projection matrix modification
    glMatrixMode(GL_PROJECTION);

    // Clear the projection matrix
    glLoadIdentity();

    // Map abstract coordinates to the window coordinates
    glOrtho(0, iWidth, 0, iHeight, -1, 1);

    // Invert y-axis
    glScalef(1, -1, 1);

    // Translate the origin to the top left corner
    glTranslatef(0, -iHeight, 0);
}


// Displays the window
void onDisplay(void)
{
    // Clear the color buffer and z-buffer
    glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

    // Enable textures
    glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);
    glEnable(GL_TEXTURE_2D);

    // Draw the polygon
    glColor4f(1.0f, 1.0f, 1.0f, 1.0f);
    glBindTexture(GL_TEXTURE_2D, uGLTexture);
    glBegin(GL_QUADS);
        glTexCoord2f(0.0f, 0.0f);
        glVertex3i(0, 0, 0);

        glTexCoord2f(fGLTextureXSize, 0.0f);
        glVertex3i(iGlutWindowXSize, 0, 0);

        glTexCoord2f(fGLTextureXSize, fGLTextureYSize);
        glVertex3i(iGlutWindowXSize, iGlutWindowYSize, 0);

        glTexCoord2f(0.0f, fGLTextureYSize);
        glVertex3i(0, iGlutWindowYSize, 0);
    glEnd();

    // Disable textures
    glDisable(GL_TEXTURE_2D);

    // Draw changes
    glFlush();
    glutSwapBuffers();
}


// Called on ASCII key pressed
void onKeyboard(unsigned char ucKey, int iX, int iY)
{
    // Translate to lower case
    ucKey = (ucKey > 'A' && ucKey <= 'Z') ? ucKey + 'a' - 'A' : ucKey;

    switch( ucKey )
    {
        // ESC, ^C and 'c' exits the application
        case 'c':
        case 27:
        case 3:
            exit(0);
            break;

        // Window mode
        case 'w':
            glutReshapeWindow(GLUT_WINDOW_XSIZE, GLUT_WINDOW_YSIZE);
            glutPositionWindow(GLUT_WINDOW_XPOS, GLUT_WINDOW_YPOS);
            break;

        // Fullscreen mode
        case 'f':
            GLUT_WINDOW_XSIZE = iGlutWindowXSize;
            GLUT_WINDOW_YSIZE = iGlutWindowYSize;
            glutFullScreen();
            break;

        // Help
        case 'h':
            MDS_CERR("Keyboard Shortcuts:" << std::endl);
            MDS_CERR(" h  Shows this help." << std::endl);
            MDS_CERR(" f  Switches to the fullscreen mode." << std::endl);
            MDS_CERR(" w  Switches back to the window." << std::endl);
            MDS_CERR(" c  Exits the program." << std::endl);
            break;
    }
}

