//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)    \n
 * Copyright (c) 2003-2005 by Michal Spanel     \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz  \n
 * File:    mdsMutex.cpp                        \n
 * Section: libSystem                           \n
 * Date:    2003/10/24                          \n
 *
 * $Id: mdsMutex.cpp 64 2006-08-11 08:45:24Z spanel $
 *
 * Description:
 * - Mutual exclusion.
 */

#include <MDSTk/System/mdsMutex.h>

#include <MDSTk/Base/mdsAssert.h>
#include <MDSTk/Base/mdsError.h>
#include <MDSTk/Base/mdsWarning.h>


namespace mds
{
namespace sys
{

//==============================================================================
/*
 * Implementation of the class mds::sys::CMutex.
 * - Windows version.
 */

#ifdef _WIN32

CMutex::CMutex(bool bLocked)
{
    m_hMutex = CreateMutex(NULL, bLocked, NULL);
    if( !m_hMutex )
    {
        MDS_THROW_ERROR("Constructor CMutex::CMutex() failed");
    }
}


CMutex::~CMutex()
{
    BOOL bResult = CloseHandle(m_hMutex);
    if( !bResult )
    {
        MDS_THROW_WARNING("Destructor CMutex::~CMutex() failed");
    }
}


void CMutex::lock()
{
    DWORD dwResult = WaitForSingleObject(m_hMutex, INFINITE);
    if( dwResult != WAIT_OBJECT_0 )
    {
        MDS_THROW_WARNING("Method CMutex::lock() failed");
    }
}


void CMutex::unlock()
{
    BOOL bResult = ReleaseMutex(m_hMutex);
    if( !bResult )
    {
        MDS_THROW_WARNING("Method CMutex::unlock() failed");
    }
}

#endif // _WIN32


//==============================================================================
/*
 * Implementation of the class mds::sys::CMutex.
 * - Linux version.
 */

#ifdef _LINUX

CMutex::CMutex(bool bLocked)
{
    if( pthread_mutex_init(&m_Mutex, NULL) != 0 )
    {
        MDS_THROW_ERROR("Constructor CMutex::CMutex() failed");
    }

    if( bLocked )
    {
        pthread_mutex_lock(&m_Mutex);
    }
}


CMutex::~CMutex()
{
    if( pthread_mutex_destroy(&m_Mutex) != 0 )
    {
        MDS_THROW_WARNING("Destructor CMutex::~CMutex() failed");
    }
}


void CMutex::lock()
{
    if( pthread_mutex_lock(&m_Mutex) != 0 )
    {
        MDS_THROW_WARNING("Method CMutex::lock() failed");
    }
}


void CMutex::unlock()
{
    if( pthread_mutex_unlock(&m_Mutex) != 0 )
    {
        MDS_THROW_WARNING("Method CMutex::unlock() failed");
    }
}

#endif // _LINUX


} // namespace sys
} // namespace mds

