//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)    \n
 * Copyright (c) 2003-2007 by Michal Spanel     \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz  \n
 * File:    mdsSHMSerializer.hxx                \n
 * Section: libModule                           \n
 * Date:    2007/06/21                          \n
 *
 * $Id:$
 *
 * Description:
 * - Binary serialization of objects (data entities) over shared memory.
 */

#include <MDSTk/Module/mdsSHMSerializer.h>

#include <MDSTk/Base/mdsTypes.h>
#include <MDSTk/Base/mdsTypeTraits.h>
#include <MDSTk/Module/mdsBinarySerializer.h>


namespace mds
{
namespace mod
{

//==============================================================================
/*
 * Implementation of the CSHMSerializer class.
 */
CSHMSerializer::CSHMSerializer(CSharedMemChannel *pChannel, unsigned int uiFlags)
    : CChannelSerializer(pChannel, uiFlags | SHM_FLAG)
    , m_spActiveChannel(pChannel)
{
}


CSHMSerializer::~CSHMSerializer()
{
}


bool CSHMSerializer::writeRoot(CSerializable& Object)
{
    // Write the entity header
    if( !testFlag(RAW_DATA) )
    {
        if( !binary_serializer::writeEntityHeader(*m_spChannel) )
        {
            return false;
        }
    }

    // Write the entity data
    Object.serialize(*this);
    if( isError() )
    {
        return false;
    }

    // Write the terminal block
    if( !testFlag(RAW_DATA) )
    {
        return binary_serializer::writeTerminal(*m_spChannel);
    }
    else
    {
        return true;
    }
}


void CSHMSerializer::beginWrite(const char *pcName,
                                EChannelCompression Compression,
                                tSize BlockSize
                                )
{
    if( testFlag(RAW_DATA) )
    {
        return;
    }

    if( !binary_serializer::writeGroupHeader(*m_spChannel, pcName, CC_RAW, 0) )
    {
        setError();
    }
}


void CSHMSerializer::endWrite()
{
}


void CSHMSerializer::writeBool(bool bValue)
{
    if( !m_spActiveChannel->write((char *)&bValue, (int)sizeof(bool)) )
    {
        setError();
    }
}


void CSHMSerializer::writeInt(int iValue)
{
    if( !m_spActiveChannel->write((char *)&iValue, (int)sizeof(int)) )
    {
        setError();
    }
}


void CSHMSerializer::writeUInt(unsigned int uiValue)
{
    if( !m_spActiveChannel->write((char *)&uiValue, (int)sizeof(unsigned int)) )
    {
        setError();
    }
}


void CSHMSerializer::writeFloat(float fValue)
{
    if( !m_spActiveChannel->write((char *)&fValue, (int)sizeof(float)) )
    {
        setError();
    }
}


void CSHMSerializer::writeDouble(double dValue)
{
    if( !m_spActiveChannel->write((char *)&dValue, (int)sizeof(double)) )
    {
        setError();
    }
}


void CSHMSerializer::writeSize(tSize Value)
{
    if( !m_spActiveChannel->write((char *)&Value, (int)sizeof(tSize)) )
    {
        setError();
    }
}


void CSHMSerializer::writeBytes(const void *pData, tSize Length)
{
    if( !m_spActiveChannel->write((char *)pData, (int)Length) )
    {
        setError();
    }
}


bool CSHMSerializer::readRoot(CSerializable& Object)
{
    // Read and check the entity header
    if( !testFlag(RAW_DATA) )
    {
        if( !binary_serializer::readEntityHeader(*m_spChannel) )
        {
            return false;
        }
    }
    
    // Read the entity data
    Object.deserialize(*this);
    if( isError() )
    {
        return false;
    }
    
    // Read the terminal block
    if( !testFlag(RAW_DATA) )
    {
        return binary_serializer::readTerminal(*m_spChannel);
    }
    else
    {
        return true;
    }
}


void CSHMSerializer::beginRead(const char *pcName)
{
    if( testFlag(RAW_DATA) )
    {
        return;
    }

    EChannelCompression Compression;
    mds::tSize BlockSize;
    if( !binary_serializer::readGroupHeader(*m_spChannel,
                                            pcName,
                                            Compression,
                                            BlockSize
                                            )
        || BlockSize != 0 
        || Compression != CC_RAW )
    {
        setError();
    }
}


void CSHMSerializer::endRead()
{
}


void CSHMSerializer::readBool(bool& bValue)
{
    if( m_spActiveChannel->read((char *)&bValue, (int)sizeof(bool)) != (int)sizeof(bool) )
    {
        setError();
    }
}


void CSHMSerializer::readInt(int& iValue)
{
    if( m_spActiveChannel->read((char *)&iValue, (int)sizeof(int)) != (int)sizeof(int) )
    {
        setError();
    }
}


void CSHMSerializer::readUInt(unsigned int& uiValue)
{   
    if( m_spActiveChannel->read((char *)&uiValue, (int)sizeof(unsigned int)) != (int)sizeof(unsigned int) )
    {
        setError();
    }
}


void CSHMSerializer::readFloat(float& fValue)
{
    if( m_spActiveChannel->read((char *)&fValue, (int)sizeof(float)) != (int)sizeof(float) )
    {
        setError();
    }
}


void CSHMSerializer::readDouble(double& dValue)
{
    if( m_spActiveChannel->read((char *)&dValue, (int)sizeof(double)) != (int)sizeof(double) )
    {
        setError();
    }
}


void CSHMSerializer::readSize(tSize& Value)
{
    if( m_spActiveChannel->read((char *)&Value, (int)sizeof(tSize)) != (int)sizeof(tSize) )
    {
        setError();
    }
}


void CSHMSerializer::readBytes(void *pData, tSize Length)
{
    if( m_spActiveChannel->read((char *)pData, (int)Length) != (int)Length )
    {
        setError();
    }
}


} // namespace mod
} // namespace mds

