//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)    \n
 * Copyright (c) 2003-2007 by Michal Spanel     \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz  \n
 * File:    mdsBinarySerializer.cpp             \n
 * Section: libModule                           \n
 * Date:    2007/06/20                          \n
 *
 * $Id:$
 *
 * Description:
 * - Serialization of objects (data entities) via channels.
 */

#include <MDSTk/Module/mdsBinarySerializer.h>

#include <MDSTk/Base/mdsTypes.h>
#include <MDSTk/Base/mdsTypeTraits.h>


namespace mds
{
namespace mod
{

//==============================================================================
/*
 * Global definitions and functions.
 */

namespace binary_serializer
{

//! Data entity magic number.
const mds::sys::tUInt32 DE_MAGIC    = 0x53444d;

//! Name of empty terminal data block.
const char DE_TERMINAL_BLOCK_NAME[] = "TerminalBlock";

//! Alignment of data entity header.
const int DE_HEADER_ALIGNMENT       = 256;


//! Padding of data entity header.
const int DE_ENTITY_HEADER_PADDING  = DE_HEADER_ALIGNMENT - 8;

//! Data entity header.
struct SEntityHeader
{
    //! Magic number.
    mds::sys::tUInt32 m_Magic;

    //! Version of the data entity format.
    mds::sys::tUInt32 m_Version;

    //! Reserved bytes for later use.
    mds::sys::tInt8 m_pcReserved[DE_ENTITY_HEADER_PADDING];
};


//! Padding of entity part header.
const int DE_PART_HEADER_PADDING   = DE_HEADER_ALIGNMENT - CSerializable::MAX_ENTITY_NAME_LENGTH - 8;

//! Entity part header.
struct SGroupHeader
{
    //! Data block name.
    char m_pcName[CSerializable::MAX_ENTITY_NAME_LENGTH];

    //! Used data compression method.
    mds::sys::tInt32 m_Compression;

    //! Size of the data block.
    mds::sys::tInt32 m_BlockSize;

    //! Reserved bytes for later use.
    mds::sys::tInt8 m_pcReserved[DE_PART_HEADER_PADDING];
};


//! Compares two entity names.
//! - Returns true if both names are similar.
bool compareEntityNames(const char *pcName1, const char *pcName2)
{
    for( int i = 0; i < CSerializable::MAX_ENTITY_NAME_LENGTH; ++i )
    {
        if( pcName1[i] != pcName2[i] )
        {
            return false;
        }
        else if ( pcName1[i] == '\0' )
        {
            break;
        }
    }
    
    return true;
}


bool writeEntityHeader(CChannel& Channel)
{
    // Initialize data entity header
    SEntityHeader Header;
    Header.m_Magic = DE_MAGIC;
    Header.m_Version = mds::sys::tUInt32(CSerializable::VERSION);
    memset(Header.m_pcReserved, 0, DE_ENTITY_HEADER_PADDING);

    // Write the header
    return Channel.write((char *)&Header, (int)sizeof(Header));
}


bool readEntityHeader(CChannel& Channel)
{
    // Read the data entity header
    SEntityHeader Header;
    int iHeaderSize = (int)sizeof(Header);
    if( Channel.read((char *)&Header, iHeaderSize) != iHeaderSize )
    {
        return false;
    }

    // Check the entity magic number and version
    return (Header.m_Magic == DE_MAGIC
        && Header.m_Version == mds::sys::tUInt32(CSerializable::VERSION));
}


bool writeGroupHeader(CChannel& Channel,
                      const char *pcName,
                      EChannelCompression eCompression,
                      tSize BlockSize
                      )
{
    // Initialize part header
    SGroupHeader Header;
    strncpy(Header.m_pcName, pcName, CSerializable::MAX_ENTITY_NAME_LENGTH);
    Header.m_Compression = mds::sys::tInt32(eCompression);
    Header.m_BlockSize = mds::sys::tInt32(BlockSize);
    memset(Header.m_pcReserved, 0, DE_PART_HEADER_PADDING);

    // Write the header
    return Channel.write((char *)&Header, (int)sizeof(Header));
}


bool readGroupHeader(CChannel& Channel,
                     const char *pcName,
                     EChannelCompression& eCompression,
                     tSize& BlockSize
                     )
{
    // Data part header and its size
    SGroupHeader Header;
    int iHeaderSize = (int)sizeof(Header);

lRead:
    // Read the part header
    if( Channel.read((char *)&Header, iHeaderSize) != iHeaderSize )
    {
        return false;
    }

    // Check the part name
    if( !compareEntityNames(Header.m_pcName, pcName) )
    {
        // Check if the part is terminal
        if( compareEntityNames(Header.m_pcName, DE_TERMINAL_BLOCK_NAME) )
        {
            // Failed to read required data part
            return false;
        }

        // Check the block size
        if( Header.m_BlockSize > 0 )
        {
            // Skip data of the part
            CBlockChannel BlockChannel(Header.m_BlockSize, &Channel);
            BlockChannel.flush();
        }

        // Try next part
        goto lRead;
    }

    // Compression method
    eCompression = EChannelCompression(Header.m_Compression);

    // Size of the block
    BlockSize = tSize(Header.m_BlockSize);

    // O.K.
    return true;
}


bool writeTerminal(CChannel& Channel)
{
    return writeGroupHeader(Channel, DE_TERMINAL_BLOCK_NAME, CC_RAW, 0);
}


bool readTerminal(CChannel& Channel)
{
    // Data part header
    SGroupHeader Header;
    int iHeaderSize = (int)sizeof(Header);

lRead:
    // Read the part header
    if( Channel.read((char *)&Header, iHeaderSize) != iHeaderSize )
    {
        return false;
    }

    // Check the part name
    if( !compareEntityNames(Header.m_pcName, DE_TERMINAL_BLOCK_NAME) )
    {
        // Check the block size
        if( Header.m_BlockSize > 0 )
        {
            // Skip data of the part
            CBlockChannel BlockChannel(Header.m_BlockSize, &Channel);
            BlockChannel.flush();
        }

        // Try next part
        goto lRead;
    }

    // Terminal has been found
    return true;
}

} // namespace binary_serializer


//==============================================================================
/*
 * Implementation of the CBinarySerializer class.
 */

CBinarySerializer::CBinarySerializer(CChannel *pChannel, unsigned int uiFlags)
    : CChannelSerializer(pChannel, uiFlags)
    , m_spActiveChannel(pChannel)
{
}


CBinarySerializer::~CBinarySerializer()
{
}


bool CBinarySerializer::writeRoot(CSerializable& Object)
{
    // Write the entity header
    if( !testFlag(RAW_DATA) )
    {
        if( !binary_serializer::writeEntityHeader(*m_spChannel) )
        {
            return false;
        }
    }

    // Write the entity data
    Object.serialize(*this);
    if( isError() )
    {
        return false;
    }

    // Write the terminal block
    if( !testFlag(RAW_DATA) )
    {
        return binary_serializer::writeTerminal(*m_spChannel);
    }
    else
    { 
        return true;
    }
}


void CBinarySerializer::beginWrite(const char *pcName,
                                   EChannelCompression Compression,
                                   tSize BlockSize
                                   )
{
    if( testFlag(RAW_DATA) )
    {
        m_spActiveChannel = m_spChannel;
        return;
    }
    
    m_spActiveChannel = CChannelCompressor::create(Compression, new CBlockChannel(BlockSize, m_spChannel));
    
    if( !binary_serializer::writeGroupHeader(*m_spChannel,
                                             pcName,
                                             Compression,
                                             BlockSize
                                             ) )
    {
        setError();
    }
}


void CBinarySerializer::endWrite()
{
    m_spActiveChannel->flush();
}


void CBinarySerializer::writeBool(bool bValue)
{
    if( !m_spActiveChannel->write((char *)&bValue, (int)sizeof(bool)) )
    {
        setError();
    }
}


void CBinarySerializer::writeInt(int iValue)
{
    if( !m_spActiveChannel->write((char *)&iValue, (int)sizeof(int)) )
    {
        setError();
    }
}


void CBinarySerializer::writeUInt(unsigned int uiValue)
{
    if( !m_spActiveChannel->write((char *)&uiValue, (int)sizeof(unsigned int)) )
    {
        setError();
    }
}


void CBinarySerializer::writeFloat(float fValue)
{
    if( !m_spActiveChannel->write((char *)&fValue, (int)sizeof(float)) )
    {
        setError();
    }
}


void CBinarySerializer::writeDouble(double dValue)
{
    if( !m_spActiveChannel->write((char *)&dValue, (int)sizeof(double)) )
    {
        setError();
    }
}


void CBinarySerializer::writeSize(tSize Value)
{
    if( !m_spActiveChannel->write((char *)&Value, (int)sizeof(tSize)) )
    {
        setError();
    }
}


void CBinarySerializer::writeBytes(const void *pData, tSize Length)
{
    if( !m_spActiveChannel->write((char *)pData, (int)Length) )
    {
        setError();
    }
}


bool CBinarySerializer::readRoot(CSerializable& Object)
{
    // Read and check the entity header
    if( !testFlag(RAW_DATA) )
    {
        if( !binary_serializer::readEntityHeader(*m_spChannel) )
        {
            return false;
        }
    }
    
    // Read the entity data
    Object.deserialize(*this);
    if( isError() )
    {
        return false;
    }
    
    // Read the terminal block
    if( !testFlag(RAW_DATA) )
    {
        return binary_serializer::readTerminal(*m_spChannel);
    }
    else
    {
        return true;
    }
}


void CBinarySerializer::beginRead(const char *pcName)
{
    if( testFlag(RAW_DATA) )
    {
        m_spActiveChannel = m_spChannel;
        return;
    }
    
    EChannelCompression Compression;
    mds::tSize BlockSize;
    if( !binary_serializer::readGroupHeader(*m_spChannel,
                                            pcName,
                                            Compression,
                                            BlockSize
                                            )
        || BlockSize < 0 )
    {
        setError();
    }
    
    m_spActiveChannel = CChannelCompressor::create(Compression, new CBlockChannel(BlockSize, m_spChannel));
}


void CBinarySerializer::endRead()
{
    m_spActiveChannel->flush();
}


void CBinarySerializer::readBool(bool& bValue)
{
    if( m_spActiveChannel->read((char *)&bValue, (int)sizeof(bool)) != (int)sizeof(bool) )
    {
        setError();
    }
}


void CBinarySerializer::readInt(int& iValue)
{
    if( m_spActiveChannel->read((char *)&iValue, (int)sizeof(int)) != (int)sizeof(int) )
    {
        setError();
    }
}


void CBinarySerializer::readUInt(unsigned int& uiValue)
{   
    if( m_spActiveChannel->read((char *)&uiValue, (int)sizeof(unsigned int)) != (int)sizeof(unsigned int) )
    {
        setError();
    }
}


void CBinarySerializer::readFloat(float& fValue)
{
    if( m_spActiveChannel->read((char *)&fValue, (int)sizeof(float)) != (int)sizeof(float) )
    {
        setError();
    }
}


void CBinarySerializer::readDouble(double& dValue)
{
    if( m_spActiveChannel->read((char *)&dValue, (int)sizeof(double)) != (int)sizeof(double) )
    {
        setError();
    }
}


void CBinarySerializer::readSize(tSize& Value)
{
    if( m_spActiveChannel->read((char *)&Value, (int)sizeof(tSize)) != (int)sizeof(tSize) )
    {
        setError();
    }
}


void CBinarySerializer::readBytes(void *pData, tSize Length)
{
    if( m_spActiveChannel->read((char *)pData, (int)Length) != (int)Length )
    {
        setError();
    }
}


} // namespace mod
} // namespace mds

