//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)    \n
 * Copyright (c) 2003-2005 by Michal Spanel     \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz  \n
 *          Premysl Krsek, krsek@fit.vutbr.cz   \n
 * File:    mdsDicomSlice.cpp                   \n
 * Section: libImageIO                          \n
 * Date:    2003/12/15                          \n
 *
 * $Id: mdsDicomSlice.cpp 341 2007-06-02 16:47:01Z xsarav00 $
 *
 * Description:
 * - Manipulation with DICOM files.
 */

#include <MDSTk/ImageIO/mdsDicomSlice.h>

// STL
#include <cstring>
#include <cmath>


namespace mds
{
namespace img
{

//==============================================================================
/*
 * Implementation of the class mds::img::CDicomSlice.
 */
CDicomSlice::CDicomSlice()
    : m_iSeriesNumber(0)
    , m_iSliceNumber(0)
    , m_iWindowCenter(1000)
    , m_iWindowWidth(1000)
    , m_bExplicitTransfer(false)
    , m_usPixelRepresentation(0)
    , m_ImagePosition(0, 0, 0)
    , m_ImageOrientationX(1, 0, 0)
    , m_ImageOrientationY(0, 1, 0)
{
}

//==============================================================================

CDicomSlice::CDicomSlice(tSize XSize,
                         tSize YSize
                         )
    : CSlice(XSize, YSize)
    , m_iSeriesNumber(0)
    , m_iSliceNumber(0)
    , m_iWindowCenter(1000)
    , m_iWindowWidth(1000)
    , m_bExplicitTransfer(false)
    , m_usPixelRepresentation(0)
    , m_ImagePosition(0, 0, 0)
    , m_ImageOrientationX(1, 0, 0)
    , m_ImageOrientationY(0, 1, 0)
{
}

//==============================================================================

CDicomSlice::CDicomSlice(tSize XSize,
                         tSize YSize,
                         const CDImage::tPixel& Value
                         )
    : CSlice(XSize, YSize, Value)
    , m_iSeriesNumber(0)
    , m_iSliceNumber(0)
    , m_iWindowCenter(1000)
    , m_iWindowWidth(1000)
    , m_bExplicitTransfer(false)
    , m_usPixelRepresentation(0)
    , m_ImagePosition(0, 0, 0)
    , m_ImageOrientationX(1, 0, 0)
    , m_ImageOrientationY(0, 1, 0)
{
}

//==============================================================================

CDicomSlice::CDicomSlice(const CDImage& Image)
    : CSlice(Image)
    , m_iSeriesNumber(0)
    , m_iSliceNumber(0)
    , m_iWindowCenter(1000)
    , m_iWindowWidth(1000)
    , m_bExplicitTransfer(false)
    , m_usPixelRepresentation(0)
    , m_ImagePosition(0, 0, 0)
    , m_ImageOrientationX(1, 0, 0)
    , m_ImageOrientationY(0, 1, 0)
{
}

//==============================================================================

CDicomSlice::CDicomSlice(const CDImage& Image, EMakeRef MakeRef)
    : CSlice(Image, REFERENCE)
    , m_iSeriesNumber(0)
    , m_iSliceNumber(0)
    , m_iWindowCenter(1000)
    , m_iWindowWidth(1000)
    , m_bExplicitTransfer(false)
    , m_usPixelRepresentation(0)
    , m_ImagePosition(0, 0, 0)
    , m_ImageOrientationX(1, 0, 0)
    , m_ImageOrientationY(0, 1, 0)
{
}

//==============================================================================

CDicomSlice::CDicomSlice(const CSlice& Slice)
    : CSlice(Slice)
    , m_iSeriesNumber(0)
    , m_iSliceNumber(0)
    , m_iWindowCenter(1000)
    , m_iWindowWidth(1000)
    , m_bExplicitTransfer(false)
    , m_usPixelRepresentation(0)
    , m_ImagePosition(0, 0, 0)
    , m_ImageOrientationX(1, 0, 0)
    , m_ImageOrientationY(0, 1, 0)
{
}

//==============================================================================

CDicomSlice::CDicomSlice(const CSlice& Slice, EMakeRef MakeRef)
    : CSlice(Slice, REFERENCE)
    , m_iSeriesNumber(0)
    , m_iSliceNumber(0)
    , m_iWindowCenter(1000)
    , m_iWindowWidth(1000)
    , m_bExplicitTransfer(false)
    , m_usPixelRepresentation(0)
    , m_ImagePosition(0, 0, 0)
    , m_ImageOrientationX(1, 0, 0)
    , m_ImageOrientationY(0, 1, 0)
{
}

//==============================================================================

CDicomSlice::CDicomSlice(const CDicomSlice& Slice)
    : CSlice(Slice)
    , m_sPatientName(Slice.m_sPatientName)
    , m_sPatientId(Slice.m_sPatientId)
    , m_sPatientBirthday(Slice.m_sPatientBirthday)
    , m_sPatientSex(Slice.m_sPatientSex)
    , m_sPatientDescription(Slice.m_sPatientDescription)    
    , m_sStudyUid(Slice.m_sStudyUid)
    , m_sStudyId(Slice.m_sStudyId)        
    , m_sStudyDate(Slice.m_sStudyDate)
    , m_sStudyDescription(Slice.m_sStudyDescription)  
    , m_sSeriesUid(Slice.m_sSeriesUid)
    , m_iSeriesNumber(Slice.m_iSeriesNumber)
    , m_sModality(Slice.m_sModality)
    , m_sSeriesDate(Slice.m_sSeriesDate)
    , m_sSeriesTime(Slice.m_sSeriesTime)
    , m_sSeriesDescription(Slice.m_sSeriesDescription)
    , m_iSliceNumber(Slice.m_iSliceNumber)
    , m_iWindowCenter(Slice.m_iWindowCenter)
    , m_iWindowWidth(Slice.m_iWindowWidth)
    , m_sPatientPosition(Slice.m_sPatientPosition)
    , m_bExplicitTransfer(Slice.m_bExplicitTransfer)
    , m_usPixelRepresentation(Slice.m_usPixelRepresentation)
    , m_ImagePosition(Slice.m_ImagePosition)
    , m_ImageOrientationX(Slice.m_ImageOrientationX)
    , m_ImageOrientationY(Slice.m_ImageOrientationY)
{
}

//==============================================================================

CDicomSlice::CDicomSlice(const CDicomSlice& Slice, EMakeRef MakeRef)
    : CSlice(Slice, REFERENCE)
    , m_sPatientName(Slice.m_sPatientName)
    , m_sPatientId(Slice.m_sPatientId)
    , m_sPatientBirthday(Slice.m_sPatientBirthday)
    , m_sPatientSex(Slice.m_sPatientSex)
    , m_sPatientDescription(Slice.m_sPatientDescription)        
    , m_sStudyUid(Slice.m_sStudyUid)
    , m_sStudyId(Slice.m_sStudyId)    
    , m_sStudyDate(Slice.m_sStudyDate)
    , m_sStudyDescription(Slice.m_sStudyDescription) 
    , m_sSeriesUid(Slice.m_sSeriesUid)
    , m_iSeriesNumber(Slice.m_iSeriesNumber)
    , m_sModality(Slice.m_sModality) 
    , m_sSeriesDate(Slice.m_sSeriesDate)
    , m_sSeriesTime(Slice.m_sSeriesTime)
    , m_sSeriesDescription(Slice.m_sSeriesDescription)
    , m_iSliceNumber(Slice.m_iSliceNumber)
    , m_iWindowCenter(Slice.m_iWindowCenter)
    , m_iWindowWidth(Slice.m_iWindowWidth)
    , m_sPatientPosition(Slice.m_sPatientPosition)
    , m_bExplicitTransfer(Slice.m_bExplicitTransfer)
    , m_usPixelRepresentation(Slice.m_usPixelRepresentation)
    , m_ImagePosition(Slice.m_ImagePosition)
    , m_ImageOrientationX(Slice.m_ImageOrientationX)
    , m_ImageOrientationY(Slice.m_ImageOrientationY)
{
}

//==============================================================================

CDicomSlice::~CDicomSlice()
{
}

//==============================================================================

CDicomSlice& CDicomSlice::operator =(const CDImage& Image)
{
    *((CDImage *)this) = Image;

    return *this;
}

//==============================================================================

CDicomSlice& CDicomSlice::operator =(const CSlice& Slice)
{
    *((CSlice *)this) = Slice;

    return *this;
}

//==============================================================================

CDicomSlice& CDicomSlice::operator =(const CDicomSlice& Slice)
{
    *((CSlice *)this) = (const CSlice&)Slice;

    m_sPatientName = Slice.m_sPatientName;
    m_sPatientId = Slice.m_sPatientId;
    m_sPatientBirthday = Slice.m_sPatientBirthday;
    m_sPatientSex = Slice.m_sPatientSex;
    m_sPatientDescription = Slice.m_sPatientDescription;    
    m_sStudyUid = Slice.m_sStudyUid;
    m_sStudyId = Slice.m_sStudyId;    
    m_sStudyDate = Slice.m_sStudyDate;
    m_sStudyDescription = Slice.m_sStudyDescription;
    m_sModality = Slice.m_sModality;
    m_iSeriesNumber = Slice.m_iSeriesNumber;
    m_sSeriesUid = Slice.m_sSeriesUid;
    m_sSeriesDate = Slice.m_sSeriesDate;
    m_sSeriesTime = Slice.m_sSeriesTime;
    m_sSeriesDescription = Slice.m_sSeriesDescription;
    m_iSliceNumber = Slice.m_iSliceNumber;
    m_iWindowCenter = Slice.m_iWindowCenter;
    m_iWindowWidth = Slice.m_iWindowWidth;
    m_sPatientPosition = Slice.m_sPatientPosition;
    m_bExplicitTransfer = Slice.m_bExplicitTransfer;
    m_usPixelRepresentation = Slice.m_usPixelRepresentation;
    m_ImagePosition = Slice.m_ImagePosition;
    m_ImageOrientationX = Slice.m_ImageOrientationX;
    m_ImageOrientationY = Slice.m_ImageOrientationY;

    return *this;
}

//==============================================================================
/*
 * IMplementation of the class mds::CDicomSlice
 * - loading DICOM slices
 */
bool CDicomSlice::loadDicom(mds::mod::CChannel *pChannel)
{
    MDS_CHECK(pChannel, return false);

    // Read and check DICOM header
    if ( ! readDicomHeader(pChannel) )
        return false;

    // Read DICOM data
    return ( readDicom(pChannel) );
}


//==============================================================================
/*
 * DICOM data manipulation functions
 */
bool CDicomSlice::readDicom(mds::mod::CChannel *pChannel)
{
    // Helper variables
    char              pcType[5], *pcTemp, *pcTemp1, *pcElemData;
    unsigned short    usGroup, usElem, usWord16;
    unsigned long     ulOffset, ulLength;
    double            dValue;
    unsigned          uXSize = 0, uYSize = 0;

    CPoint3D zero_point(0, 0, 0);
    CVector3D normala_image, position_vector;

    // inicializace parametru rezu
    m_dPosition = 0.0;
    m_dDX = 1.0;
    m_dDY = 1.0;
    m_dThickness = 1.0;

    m_iSeriesNumber    = 0;
    m_iSliceNumber = 0;
    m_iWindowCenter = 1000;
    m_iWindowWidth = 1000;
    m_usPixelRepresentation = 0;

    m_sPatientName.erase();
    m_sPatientId.erase();
    m_sPatientBirthday.erase();
    m_sPatientSex.erase();
    m_sPatientDescription.erase();
    m_sStudyUid.erase();
    m_sStudyId.erase();    
    m_sStudyDate.erase();
    m_sStudyDescription.erase();
    m_sModality.erase();
    m_sSeriesUid.erase();
    m_sSeriesDate.erase();
    m_sSeriesTime.erase();
    m_sPatientPosition.erase();

    m_ImageOrientationX.setXYZ(1, 0, 0);
    m_ImageOrientationY.setXYZ(0, 1, 0);

    // cyklus cteni elementu souboru
    do {
        // nacteni tagu
        if ( ! readElementTag(pChannel, &usGroup, &usElem) )
            return false;

        // nacteni poctu bytu hodnoty elementu
        if ( ! readValueLength(pChannel, pcType, m_bExplicitTransfer, &ulLength, &ulOffset) )
            return false;

        // pokud ma tag nulovou velikost, pokracuje se dalsim
        if( ulLength == 0UL )
            continue;

        // pokud na tag nedefinovanou delku, pokracuje se ve cteni podle kodovani VR = SQ
        if ( ulLength == 0xFFFFFFFF )
        {
            // nacteni sekvence podle kodovani VR = SQ
            if ( ! readSQDataElement(pChannel))
                return false;

            continue;
        }

        // nacteni data elementu
        if ( ! (pcElemData = readValueData(pChannel, ulLength)) )
            return false;

        // zpracovani hodnoty elementu
        switch( usGroup )
        {
        case 0x8:
            switch( usElem )
            {
            case 0x12:                 // create date
                // ulozeni textu
                m_sSeriesDate = pcElemData;
                break;
            case 0x13:                    // create time
                // ulozeni textu
                m_sSeriesTime = pcElemData;
                break;
            case 0x20:                    // datum study
                // ulozeni textu
                m_sStudyDate = pcElemData;
                break;
            case 0x21:                    // datum serie
                // ulozeni textu
                m_sSeriesDate = pcElemData;
                break;
            case 0x31:                    // cas serie
                // ulozeni textu
                m_sSeriesTime = pcElemData;
                break;
            case 0x60:                    // modalita rezu
                // ulozeni textu
                m_sModality = pcElemData;
                break;
            case 0x1030:                  // popis study
                // ulozeni textu
                m_sStudyDescription = pcElemData;
                break;
            case 0x103E:                  // popis serie
                // ulozeni textu
                m_sSeriesDescription = pcElemData;
                break;
            }
            break;

        case 0x10:
            switch( usElem )
            {
            case 0x10:                    // jmeno pacienta
                // ulozeni textu
                m_sPatientName = pcElemData;
                break;
            case 0x20:                    // id pacienta
                // ulozeni textu
                m_sPatientId = pcElemData;
                break;
            case 0x30:                    // datum narozeni pacienta
                // ulozeni textu
                m_sPatientBirthday = pcElemData;
                break;
            case 0x40:                    // pohlavi pacienta
                // ulozeni textu
                m_sPatientSex = pcElemData;
                break;
            case 0x4000:                  // popis pacienta
                // ulozeni textu
                m_sPatientDescription = pcElemData;
                break;
            }
            break;

        case 0x18:
            switch( usElem )
            {
            case 0x50:                    // tloustka rezu
                // ziskani cisla z textu
                sscanf(pcElemData, "%lf", &m_dThickness);
                break;
            case 0x5100:                    // patient position
                // ulozeni textu
                m_sPatientPosition = pcElemData;
                break;
            }
            break;

        case 0x20:
            switch( usElem )
            {
            case 0xD:                     // uid study
                // ulozeni textu
                m_sStudyUid = pcElemData;
                break;
            case 0xE:                     // uid serie
                // ulozeni textu
                m_sSeriesUid = pcElemData;
                break;
            case 0x10:                    // id study
                // ziskani cisla z textu
                m_sStudyId = pcElemData;
                break;
            case 0x11:                    // cislo serie
                // ziskani cisla z textu
                sscanf(pcElemData, "%d", &m_iSeriesNumber);
                break;
            case 0x13:                    // cislo rezu
                // ziskani cisla z textu
                sscanf(pcElemData, "%d", &m_iSliceNumber);
                m_iIndex = m_iSliceNumber;    // ???
                break;
            case 0x32:                    // pozice rezu pacienta
                // ziskani souradnice Z rezu z textu
                dValue = 0;
                pcTemp = strrchr(pcElemData, '\\');
                sscanf((pcTemp+1), "%lf", &dValue);
                m_ImagePosition.setZ((tCoordinate)dValue);
                // zruseni posledniho lomitka
                *pcTemp = 0;
                pcTemp1 = pcTemp;
                // ziskani souradnice Y rezu z textu
                dValue = 0;
                pcTemp = strrchr(pcElemData, '\\');
                *pcTemp1 = '\\';
                sscanf((pcTemp+1), "%lf", &dValue);
                m_ImagePosition.setY((tCoordinate)dValue);
                // zruseni posledniho lomitka
                *pcTemp = 0;
                // ziskani souradnice X rezu z textu
                dValue = 0;
                sscanf(pcElemData, "%lf", &dValue);
                *pcTemp = '\\';
                m_ImagePosition.setX((tCoordinate)dValue);
                break;
            case 0x37:                    // image orientation pacienta
                // ziskani souradnice Z vektoru y orientace rezu z textu
                dValue = 0;
                pcTemp = strrchr(pcElemData, '\\');
                sscanf((pcTemp+1), "%lf", &dValue);
                m_ImageOrientationY.setZ((tCoordinate)dValue);
                // zruseni posledniho lomitka
                *pcTemp = 0;
                pcTemp1 = pcTemp;
                // ziskani souradnice Y vektoru y orientace rezu z textu
                dValue = 0;
                pcTemp = strrchr(pcElemData, '\\');
                *pcTemp1 = '\\';
                sscanf((pcTemp+1), "%lf", &dValue);
                m_ImageOrientationY.setY((tCoordinate)dValue);
                // zruseni posledniho lomitka
                *pcTemp = 0;
                pcTemp1 = pcTemp;
                // ziskani souradnice X vektoru y orientace rezu z textu
                dValue = 0;
                pcTemp = strrchr(pcElemData, '\\');
                *pcTemp1 = '\\';
                sscanf((pcTemp+1), "%lf", &dValue);
                m_ImageOrientationY.setX((tCoordinate)dValue);
                // zruseni posledniho lomitka
                *pcTemp = 0;
                pcTemp1 = pcTemp;
                // ziskani souradnice Z vektoru x orientace rezu z textu
                dValue = 0;
                pcTemp = strrchr(pcElemData, '\\');
                *pcTemp1 = '\\';
                sscanf((pcTemp+1), "%lf", &dValue);
                m_ImageOrientationX.setZ((tCoordinate)dValue);
                // zruseni posledniho lomitka
                *pcTemp = 0;
                pcTemp1 = pcTemp;
                // ziskani souradnice Y vektoru x orientace rezu z textu
                dValue = 0;
                pcTemp = strrchr(pcElemData, '\\');
                *pcTemp1 = '\\';
                sscanf((pcTemp+1), "%lf", &dValue);
                m_ImageOrientationX.setY((tCoordinate)dValue);
                // zruseni posledniho lomitka
                *pcTemp = 0;
                pcTemp1 = pcTemp;
                // ziskani souradnice X vektoru x orientace rezu z textu
                dValue = 0;
                sscanf(pcElemData, "%lf", &dValue);
                *pcTemp1 = '\\';
                m_ImageOrientationX.setX((tCoordinate)dValue);

                // vypocet normaly roviny rezu a jeho normalizace
                normala_image.getVectorProduct(m_ImageOrientationX, m_ImageOrientationY);
                normala_image.normalize();
                // vypocet vektoru pozice rezu, jeho delky a normalizace
                position_vector. create(zero_point, m_ImagePosition);
                m_dPosition = position_vector.getLength();
                position_vector.normalize();

                // test modality MR
                if ( (strncmp(m_sModality.c_str(), "MR", 2) == 0) || (strncmp(m_sModality.c_str(), "mr", 2) == 0) )
                {
                    // zruseni tiltu pro MR data
                    m_ImageOrientationX.setXYZ(1, 0, 0);
                    m_ImageOrientationY.setXYZ(0, 1, 0);
                    // vypocet pozice rezu z vektoru roviny rezu a jeho 3D pozice
                    m_dPosition = normala_image.getScalarProduct(normala_image, position_vector) * m_dPosition;
                }
                else
                {
                    m_dPosition = m_ImagePosition.getZ();
                }
                break;
            case 0x1041:                    // pozice rezu
                // ziskani cisla z textu
                //sscanf(pcElemData, "%lf", &m_dPosition);
                break;
            }
            break;

        case 0x28:
            switch( usElem )
            {
            case 0x10:                    // rozmer matice v y
                // ziskani cisla
                memcpy(&usWord16, pcElemData, 2);
                // ulozeni hodnoty
                uYSize = usWord16;
                break;
            case 0x11:                    // rozmer matice v x
                // ziskani cisla
                memcpy(&usWord16, pcElemData, 2);
                // ulozeni hodnoty
                uXSize = usWord16;
                break;
            case 0x30:                    // rozmery pixelu v x a y
                // ziskani cisla
                pcTemp = strchr(pcElemData, '\\');
                // vymazani lomitka mezi hodnotami
                *pcTemp = 0;
                sscanf(pcElemData, "%lf", &m_dDX);
                // ziskani cisla
                sscanf((pcTemp+1), "%lf", &m_dDY);
                // vraceni lomitka mezi hodnoty
                *pcTemp = '\\';
                break;
            case 0x103:
                m_usPixelRepresentation = 0;
                memcpy(&usWord16, pcElemData, 2);
                // ulozeni hodnoty
                m_usPixelRepresentation = usWord16;
                break;
            case 0x1050:                    // stred densitniho okenka rezu
                dValue = 0;
                // ziskani cisla z textu
                sscanf(pcElemData, "%lf", &dValue);
                // ulozeni cisla do attributu
                m_iWindowCenter = (int)floor(dValue);
                break;
            case 0x1051:                    // sirka densitniho okenka rezu
                dValue = 0;
                // ziskani cisla z textu
                sscanf(pcElemData, "%lf", &dValue);
                // ulozeni cisla do attributu
                m_iWindowWidth = (int)floor(dValue);
                break;
            }
            break;

        case 0x7fe0:
            if( usElem == 0x10 )                // pixel data rezu
            {
                // kontrola rozmeru dat obrazu
                MDS_ASSERT((uXSize * uYSize * sizeof(unsigned short)) == ulLength);

                // vytvoreni obrazu
                if ( ! CDImage::create((tSize)uXSize, (tSize)uYSize, (CDImage::tPixel)0, m_Margin ) )
                    return false;

                /** @todo Pozor na inicializaci dx, dy pri tvorbe CSlice. */

                // kopie dat
                for( tSize j = 0; j < m_YSize; ++j )
                {
                    setRow(j, (CDImage::tPixel *)&pcElemData[j * m_XSize * sizeof(CDImage::tPixel)]);
                }

                // Uvolneni nactenych dat
                delete pcElemData;

                /** @todo Uprava dat matice podle pixel rep, posunu dat, atd. */
                // ukonceni funkce
                return true;
            }
            break;

        } // switch( usGroup )

        // Uvolneni nactenych dat
        delete pcElemData;

    } while( true );
}

//==============================================================================

bool CDicomSlice::readDicomHeader(mds::mod::CChannel *pChannel)
{
    // Helper variables
    char pcPrefix[5], pcType[5], *pcElemData;
    unsigned short usGroup, usElem;
    unsigned long ulOffset, ulHeaderLimit, ulLength;


    // preskoceni preamble
    char pcDump[128];
    if ( pChannel->read(pcDump, 128) != 128 )
        return false;

    // ziskani hodnoty prefixu
    if ( pChannel->read(pcPrefix, 4)!= 4 )
        return false;

    // kontrola pcPrefix
    if( strncmp(pcPrefix, "DICM", 4) )
        return false;

    // ziskani hodnoty prvniho tagu
    if ( ! readElementTag(pChannel, &usGroup, &usElem) )
        return false;

    // kontrola kodu prvniho elementu
    if( usGroup != 2 || usElem != 0 )
        return false;

    // nacteni poctu bytu hodnoty elementu
    if ( ! readValueLength(pChannel, pcType, true, &ulLength, &ulOffset) )
        return false;

    // kontrola delky hodnoty elementu
    if( ulLength != 4UL )
        return false;

    // ziskani poctu bytu hlavicky
    if ( pChannel->read((char *)&ulHeaderLimit, 4) != 4 )
        return false;

    // pro urceni konce hlavicky
    ulOffset = 0UL;

    // cteni datovych elementu hlavicky
    do {
        // ziskani hodnoty dalsiho tagu
        if ( ! readElementTag(pChannel, &usGroup, &usElem) )
            return false;
        ulOffset += 4UL;

        // kontrola skupiny hlavicky
        if( usGroup != 2 )
            return false;

        // nacteni poctu bytu hodnoty elementu
        if ( ! readValueLength(pChannel, pcType, true, &ulLength, &ulOffset) )
            return false;

        // pokud je delka tagu nulova, prejde se na dalsi
        if( ulLength == 0UL )
            continue;

        // nacteni data elementu
        if ( ! (pcElemData = readValueData(pChannel, ulLength)) )
            return false;
        ulOffset += ulLength;

        // zpracovani hodnoty elementu
        switch( usElem )
        {
            case 0x10:                        // Transfer syntax UID
                // vyhodnoceni hodnoty transfer syntax
                if( !strncmp(pcElemData, "1.2.840.10008.1.2", ulLength) )
                {
                    m_bExplicitTransfer = false;    // implicitni transfer syntax
                }
                else if( !strncmp(pcElemData, "1.2.840.10008.1.2.1", ulLength) )
                {
                    m_bExplicitTransfer = true;    // explicitni transfer syntax
                }
                else                               // neznamy transfer syntax
                    return false;
    
                break;
        }

        // Uvolneni nactenych dat
        delete pcElemData;

    } while( ulOffset < ulHeaderLimit );

    // DICOM header readed succesfuly
    return true;
}

//==============================================================================

bool CDicomSlice::readValueLength(mds::mod::CChannel *pChannel, char *pcType, bool bExplicit, unsigned long *ulLength, unsigned long *ulOffset)
{
    MDS_ASSERT(pcType);
    MDS_ASSERT(pcType);
    MDS_ASSERT(ulLength);
    MDS_ASSERT(ulOffset);

    *ulLength = 0UL;                // ziskana delka hodnoty elementu

    // test implicitniho transfer syntax
    if( !bExplicit )
    {
        // ziskani delky hodnoty polozky
        if ( pChannel->read((char *)ulLength, 4) != 4 )
            return false;
        *ulOffset += 4UL;
    }
    else
    {
        // ziskani typu vr
        if ( pChannel->read(pcType, 2) != 2 )
            return false;
        *ulOffset += 2UL;

        // ziskani dvou doplnkovych byte nebo nacteni delky
        if ( pChannel->read((char *)ulLength, 2) != 2 )
            return false;
        *ulOffset += 2UL;

        // konverze znaku VR na mala pismena
        pcType[0] = tolower(pcType[0]);
        pcType[1] = tolower(pcType[1]);

        // overeni spravnosti typu vr
        if( !strncmp(pcType,"ob",2) || !strncmp(pcType,"ow",2) || !strncmp(pcType,"sq",2) || !strncmp(pcType,"un",2) )
        {
            // ziskani delky hodnoty polozky
            if ( pChannel->read((char *)ulLength, 4) != 4 )
                return false;
            *ulOffset += 4UL;
        }
    }

    // vraceni ziskane hodnoty delky hodnoty elementu
    return true;
}

//==============================================================================

char * CDicomSlice::readValueData(mds::mod::CChannel *pChannel, unsigned long ulLength)
{
    // Allocate memory
    char *pcData = new char[ulLength + 1];
    MDS_ASSERT(pcData);

    // Read pcData
    if( pChannel->read(pcData, (int)ulLength) != (int)ulLength )
    {
        delete[] pcData;
        return NULL;
    }

    pcData[ulLength] = '\0';
    return pcData;
}

//==============================================================================

bool CDicomSlice::readElementTag(mds::mod::CChannel *pChannel, unsigned short *usGroup, unsigned short *usElem )
{
    // Read an element tag
    if( pChannel->read((char *)usGroup, 2) != 2 )
        return false;

    if( pChannel->read((char *)usElem, 2) != 2 )
        return false;

    return true;
}

//==============================================================================

bool CDicomSlice::readSQDataElement(mds::mod::CChannel *pChannel)
{
    char              pcType[5], * pcElemData;
    unsigned short    usGroup, usElem;
    unsigned long     ulOffset, ulLength;

    // nekonecny cyklus cteni
    while(true)
    {
        // nacteni tagu
        if ( ! readElementTag(pChannel, &usGroup, &usElem))
            return false;

        // nacteni poctu bytu hodnoty elementu
        if ( ! readValueLength(pChannel, pcType, false, &ulLength, &ulOffset))
            return false;

        // test hodnoty skupiny tagu
        if (usGroup != 0xFFFE)
            return false;

        // test hodnoty elementu tagu
        if (usElem == 0xE000)
        {
            // test delky dat elementu
            if (ulLength != 0xFFFFFFFF)
            {
                // test nulove delky
                if (ulLength != 0)
                {
                    // nacteni dat elementu
                    if ( ! (pcElemData = readValueData(pChannel, ulLength)) )
                        return false;
                    // uvolneni pameti dat elementu
                    delete pcElemData;
                }
            }
            else
            {
                // nekonecny cyklus vlozenych elementu
                while(true)
                {
                    // nacteni tagu
                    if ( ! readElementTag(pChannel, &usGroup, &usElem))
                        return false;

                    // nacteni poctu bytu hodnoty elementu
                    if ( ! readValueLength(pChannel, pcType, m_bExplicitTransfer, &ulLength, &ulOffset))
                        return false;

                    // test ukoncovaciho tagu sekvence
                    if ((usGroup == 0xFFFE) && (usElem == 0xE00D))
                        break;

                    // test delky dat elementu
                    if (ulLength != 0xFFFFFFFF)
                    {
                        // test nulove delky
                        if (ulLength != 0)
                        {
                            // nacteni dat elementu
                            if ( ! (pcElemData = readValueData(pChannel, ulLength)) )
                                return false;
                            // uvolneni pameti dat elementu
                            delete pcElemData;
                        }
                    }
                    else
                    {
                        // rekursivni volani funkce pro cteni SQ sekvence nedefinovane delky
                        if ( ! readSQDataElement(pChannel))
                            return false;
                    }
                }
            }
        }
        else if (usElem == 0xE0DD)
            break;
        else
            return false;
    }

    return true;
}

} // namespace img
} // namespace mds

