//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)    \n
 * Copyright (c) 2003-2005 by Michal Spanel     \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz  \n
 * File:    mdsSlice.cpp                        \n
 * Section: libImage                            \n
 * Date:    2003/12/14                          \n
 *
 * $Id: mdsSlice.cpp 383 2007-06-21 12:17:06Z spanel $
 * 
 * Description:
 * - Density slice.
 */

#include <MDSTk/Image/mdsSlice.h>


namespace mds
{
namespace img
{

//==============================================================================
/*
 * Global constants.
 */

//! Size of the slice margin.
const tSize SLICE_MARGIN    = 16;


//==============================================================================
/*
 * Implementation of the class mds::img::CSlice.
 */
CSlice::CSlice()
    : CDImage(0, 0, SLICE_MARGIN)
    , m_dDX(1.0)
    , m_dDY(1.0)
    , m_dThickness(1.0)
    , m_dPosition(0.0)
    , m_iIndex(0)
    , m_eOrientation(PLANE_XY)
{
}


CSlice::CSlice(tSize XSize, tSize YSize)
    : CDImage(XSize, YSize, SLICE_MARGIN)
    , m_dDX(1.0)
    , m_dDY(1.0)
    , m_dThickness(1.0)
    , m_dPosition(0.0)
    , m_iIndex(0)
    , m_eOrientation(PLANE_XY)
{
}


CSlice::CSlice(tSize XSize, tSize YSize, const CDImage::tPixel& Value)
    : CDImage(XSize, YSize, Value, SLICE_MARGIN)
    , m_dDX(1.0)
    , m_dDY(1.0)
    , m_dThickness(1.0)
    , m_dPosition(0.0)
    , m_iIndex(0)
    , m_eOrientation(PLANE_XY)
{
}


CSlice::CSlice(const CSlice& Slice)
    : CDImage(Slice)
    , m_dDX(Slice.m_dDX)
    , m_dDY(Slice.m_dDY)
    , m_dThickness(Slice.m_dThickness)
    , m_dPosition(Slice.m_dPosition)
    , m_iIndex(Slice.m_iIndex)
    , m_eOrientation(Slice.m_eOrientation)
{
}


CSlice::CSlice(const CSlice& Slice, EMakeRef MakeRef)
    : CDImage(Slice, REFERENCE)
    , m_dDX(Slice.m_dDX)
    , m_dDY(Slice.m_dDY)
    , m_dThickness(Slice.m_dThickness)
    , m_dPosition(Slice.m_dPosition)
    , m_iIndex(Slice.m_iIndex)
    , m_eOrientation(Slice.m_eOrientation)
{
}


CSlice::CSlice(const CDImage& Image)
    : CDImage(Image)
    , m_dDX(1.0)
    , m_dDY(1.0)
    , m_dThickness(1.0)
    , m_dPosition(0.0)
    , m_iIndex(0)
    , m_eOrientation(PLANE_XY)
{
}


CSlice::CSlice(const CDImage& Image, EMakeRef MakeRef)
    : CDImage(Image, REFERENCE)
    , m_dDX(1.0)
    , m_dDY(1.0)
    , m_dThickness(1.0)
    , m_dPosition(0.0)
    , m_iIndex(0)
    , m_eOrientation(PLANE_XY)
{
}


CSlice::~CSlice()
{
}


bool CSlice::create(tSize XSize, tSize YSize)
{
    // Create image
    bool bResult = CDImage::create(XSize, YSize, SLICE_MARGIN);

    // Initialize the slice parameters
    m_dThickness = 1.0;
    m_dPosition = 0.0;
    m_iIndex = 0;
    m_eOrientation = PLANE_XY;

    return bResult;
}


bool CSlice::create(tSize XSize, tSize YSize, double dDX, double dDY)
{
    // Create image
    bool bResult = CDImage::create(XSize, YSize, SLICE_MARGIN);

    // Initialize the slice parameters
    m_dDX = dDX;
    m_dDY = dDY;
    m_dThickness = 1.0;
    m_dPosition = 0.0;
    m_iIndex = 0;
    m_eOrientation = PLANE_XY;

    return bResult;
}


bool CSlice::create(tSize XSize,
                    tSize YSize,
                    const CDImage::tPixel& Value
                    )
{
    // Create image
    bool bResult = CDImage::create(XSize, YSize, Value, SLICE_MARGIN);

    // Initialize the slice parameters
    m_dThickness = 1.0;
    m_dPosition = 0.0;
    m_iIndex = 0;
    m_eOrientation = PLANE_XY;

    return bResult;
}


bool CSlice::create(tSize XSize,
                    tSize YSize,
                    const CDImage::tPixel& Value,
                    double dDX,
                    double dDY
                    )
{
    // Create image
    bool bResult = CDImage::create(XSize, YSize, Value, SLICE_MARGIN);

    // Initialize the slice parameters
    m_dDX = dDX;
    m_dDY = dDY;
    m_dThickness = 1.0;
    m_dPosition = 0.0;
    m_iIndex = 0;
    m_eOrientation = PLANE_XY;

    return bResult;
}


bool CSlice::create(const CSlice& Slice)
{
    // Copy the image data
    CDImage::create(Slice);

    // Slice properties
    m_dDX = Slice.m_dDX;
    m_dDY = Slice.m_dDY;
    m_dThickness = Slice.m_dThickness;
    m_dPosition = Slice.m_dPosition;
    m_iIndex = Slice.m_iIndex;
    m_eOrientation = Slice.m_eOrientation;

    return true;
}


bool CSlice::create(const CSlice& Slice, EMakeRef MakeRef)
{
    // Make reference to the image data
    CDImage::create(Slice, REFERENCE);

    // Slice properties
    m_dDX = Slice.m_dDX;
    m_dDY = Slice.m_dDY;
    m_dThickness = Slice.m_dThickness;
    m_dPosition = Slice.m_dPosition;
    m_iIndex = Slice.m_iIndex;
    m_eOrientation = Slice.m_eOrientation;

    return true;
}


CSlice& CSlice::operator =(const CSlice& Slice)
{
    *((CDImage *)this) = (const CDImage&)Slice;

    m_dDX = Slice.m_dDX;
    m_dDY = Slice.m_dDY;
    m_dThickness = Slice.m_dThickness;
    m_dPosition = Slice.m_dPosition;
    m_iIndex = Slice.m_iIndex;
    m_eOrientation = Slice.m_eOrientation;

    return *this;
}


//==============================================================================
/*
 * - Slice data serialization functions.
 */

void CSlice::serialize(mds::mod::CChannelSerializer& Writer)
{
    // Serialize data of the parent class
    CDImage::serialize(Writer);

    // Begin of data serialization block
    MDS_DE_SERIALIZE_BEGIN;

        // Extended slice info
        Writer.writeDouble(m_dDX);
        Writer.writeDouble(m_dDY);
        Writer.writeDouble(m_dThickness);
        Writer.writeDouble(m_dPosition);
        Writer.writeInt(m_iIndex);
        Writer.writeInt(m_eOrientation);

    // End of the block
    MDS_DE_SERIALIZE_END;
}


void CSlice::deserialize(mds::mod::CChannelSerializer& Reader)
{
    // Deserialize data of the parent class
    CDImage::deserialize(Reader);

    // Begin of data deserialization block
    MDS_DE_DESERIALIZE_BEGIN;

        // Read the extended slice info struct
        Reader.readDouble(m_dDX);
        Reader.readDouble(m_dDY);
        Reader.readDouble(m_dThickness);
        Reader.readDouble(m_dPosition);
        Reader.readInt(m_iIndex);

        int iOrientation;
        Reader.readInt(iOrientation);
        m_eOrientation = (EPlane)iOrientation;

    // End of the block
    MDS_DE_DESERIALIZE_END;
}


} // namespace img
} // namespace mds

