//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)    \n
 * Copyright (c) 2003-2005 by Michal Spanel     \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz  \n
 * File:    mdsKernel.cpp                       \n
 * Section: libImage                            \n
 * Date:    2004/04/19                          \n
 *
 * $Id: mdsKernel.cpp 64 2006-08-11 08:45:24Z spanel $
 * 
 * Description:
 * - Convolution kernel definition.
 */

#include <MDSTk/Image/mdsKernel.h>

#include <MDSTk/Base/mdsMemory.h>
#include <MDSTk/Math/mdsBase.h>

#include <cstring>


namespace mds
{
namespace img
{

//==============================================================================
/*
 * Implementation of the class mds::img::CKernel1D.
 */
CKernel1D::CKernel1D(tSize Size)
    : mds::base::CData<float>(Size)
{
    MDS_ASSERT((Size % 2) == 1);
}


CKernel1D::CKernel1D(tSize Size, const CKernel3D::tData& Value)
    : mds::base::CData<float>(Size)
{
    MDS_ASSERT((Size % 2) == 1);

    // Zero the kernel data
    memSet(m_pData, Value, m_Size);
}


CKernel1D::CKernel1D(CKernel1D::tConstDataPtr pData, tSize Size)
    : mds::base::CData<float>(Size)
{
    MDS_ASSERT((Size % 2) == 1);

    // Copy the kernel data
    memCopy(m_pData, pData, m_Size);
}


CKernel1D::CKernel1D(const CKernel1D& k)
    : mds::base::CData<float>(k)
{
}


void CKernel1D::resize(tSize Size)
{
    MDS_ASSERT((Size % 2) == 1);

    // Reallocate the memory
    data::resize(Size);
}


CKernel1D& CKernel1D::operator =(const CKernel1D& k)
{
    if( &k == this )
    {
        return *this;
    }

    tSize Count = mds::math::getMin(m_Size, k.m_Size);
    memCopy(m_pData, k.m_pData, Count);

    return *this;
}


//==============================================================================
/*
 * Implementation of the class mds::img::CKernel2D.
 */
CKernel2D::CKernel2D(tSize Size)
    : mds::base::CData<float>(Size * Size)
    , m_YOffset(Size)
{
    MDS_ASSERT((Size % 2) == 1);
}


CKernel2D::CKernel2D(tSize Size, const CKernel3D::tData& Value)
    : mds::base::CData<float>(Size * Size)
    , m_YOffset(Size)
{
    MDS_ASSERT((Size % 2) == 1);

    // Initialize the kernel data
    memSet(m_pData, Value, m_Size);
}


CKernel2D::CKernel2D(CKernel2D::tConstDataPtr pData, tSize Size)
    : mds::base::CData<float>(Size * Size)
    , m_YOffset(Size)
{
    MDS_ASSERT((Size % 2) == 1);

    // Copy the kernel data
    memCopy(m_pData, pData, m_Size);
}


CKernel2D::CKernel2D(const CKernel2D& k)
    : mds::base::CData<float>(k)
    , m_YOffset(k.m_YOffset)
{
}


void CKernel2D::resize(tSize Size)
{
    MDS_ASSERT((Size % 2) == 1);

    // Reallocate the memory
    data::resize(Size * Size);
    m_YOffset = Size;
}


CKernel2D& CKernel2D::operator =(const CKernel2D& k)
{
    if( &k == this )
    {
        return *this;
    }

    tSize Count = mds::math::getMin(m_YOffset, k.m_YOffset);
    for( tSize j = 0; j < Count; ++j )
    {
        memCopy(m_pData + j * m_YOffset, k.m_pData + j * k.m_YOffset, Count);
    }

    return *this;
}


//==============================================================================
/*
 * Implementation of the class mds::img::CKernel3D.
 */
CKernel3D::CKernel3D(tSize Size)
    : mds::base::CData<float>(Size * Size * Size),
    m_YOffset(Size),
    m_ZOffset(Size * Size)
{
    MDS_ASSERT((Size % 2) == 1);
}


CKernel3D::CKernel3D(tSize Size, const CKernel3D::tData& Value)
    : mds::base::CData<float>(Size * Size * Size),
    m_YOffset(Size),
    m_ZOffset(Size * Size)
{
    MDS_ASSERT((Size % 2) == 1);

    // Initialize the kernel data
    memSet(m_pData, Value, m_Size);
}


CKernel3D::CKernel3D(CKernel3D::tConstDataPtr pData, tSize Size)
    : mds::base::CData<float>(Size * Size * Size),
    m_YOffset(Size),
    m_ZOffset(Size * Size)
{
    MDS_ASSERT((Size % 2) == 1);

    // Copy the kernel data
    memCopy(m_pData, pData, m_Size);
}


CKernel3D::CKernel3D(const CKernel3D& k)
    : mds::base::CData<float>(k)
    , m_YOffset(k.m_YOffset)
    , m_ZOffset(k.m_ZOffset)
{
}


void CKernel3D::resize(tSize Size)
{
    MDS_ASSERT((Size % 2) == 1);

    // Reallocate the memory
    data::resize(Size * Size * Size);
    m_YOffset = Size;
    m_ZOffset = Size * Size;
}


CKernel3D& CKernel3D::operator =(const CKernel3D& k)
{
    if( &k == this )
    {
        return *this;
    }

    tSize Count = mds::math::getMin(m_YOffset, k.m_YOffset);

    for( tSize i = 0; i < Count; ++i )
    {
        for( tSize j = 0; j < Count; ++j )
        {
            memCopy(m_pData + i * m_ZOffset + j * m_YOffset,
                    k.m_pData + i * k.m_ZOffset + j * k.m_YOffset,
                    Count
                   );
        }
    }
    return *this;
}


} // namespace img
} // namespace mds

