//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)    \n
 * Copyright (c) 2003-2005 by Michal Spanel     \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz  \n
 * File:    mdsDensityVolume.cpp                \n
 * Section: libImage                            \n
 * Date:    2004/07/11                          \n
 *
 * $Id: mdsDensityVolume.cpp 383 2007-06-21 12:17:06Z spanel $
 * 
 * Description:
 * - Manipulation with the real volume data.
 */

#include <MDSTk/Image/mdsDensityVolume.h>


namespace mds
{
namespace img
{

//==============================================================================
/*
 * Global constants.
 */

//! Size of the volume margin.
const tSize VOLUME_MARGIN   = 3;


//==============================================================================
/*
 * Implementation of the class mds::img::CDensityVolume.
 */
CDensityVolume::CDensityVolume()
    : CDVolume(0, 0, 0, VOLUME_MARGIN)
    , m_dDX(1.0)
    , m_dDY(1.0)
    , m_dDZ(1.0)
{
}


CDensityVolume::CDensityVolume(tSize XSize,
                               tSize YSize,
                               tSize ZSize
                               )
    : CDVolume(XSize, YSize, ZSize, VOLUME_MARGIN)
    , m_dDX(1.0)
    , m_dDY(1.0)
    , m_dDZ(1.0)
{
}


CDensityVolume::CDensityVolume(tSize XSize,
                               tSize YSize,
                               tSize ZSize,
                               const CDVolume::tVoxel& Value
                               )
    : CDVolume(XSize, YSize, ZSize, Value, VOLUME_MARGIN)
    , m_dDX(1.0)
    , m_dDY(1.0)
    , m_dDZ(1.0)
{
}


CDensityVolume::CDensityVolume(const CDensityVolume& Volume)
    : CDVolume(Volume)
    , m_dDX(Volume.m_dDX)
    , m_dDY(Volume.m_dDY)
    , m_dDZ(Volume.m_dDZ)
{
}


CDensityVolume::CDensityVolume(const CDensityVolume& Volume, EMakeRef MakeRef)
    : CDVolume(Volume, REFERENCE)
    , m_dDX(Volume.m_dDX)
    , m_dDY(Volume.m_dDY)
    , m_dDZ(Volume.m_dDZ)
{
}


CDensityVolume::CDensityVolume(const CDVolume& Volume)
    : CDVolume(Volume)
    , m_dDX(1.0)
    , m_dDY(1.0)
    , m_dDZ(1.0)
{
}


CDensityVolume::CDensityVolume(const CDVolume& Volume, EMakeRef MakeRef)
    : CDVolume(Volume, REFERENCE)
    , m_dDX(1.0)
    , m_dDY(1.0)
    , m_dDZ(1.0)
{
}


CDensityVolume::~CDensityVolume()
{
}


bool CDensityVolume::create(tSize XSize,
                            tSize YSize,
                            tSize ZSize
                            )
{
    // Create the volume
    bool bResult = CDVolume::create(XSize, YSize, ZSize, VOLUME_MARGIN);

    return bResult;
}


bool CDensityVolume::create(tSize XSize,
                            tSize YSize,
                            tSize ZSize,
                            double dDX,
                            double dDY,
                            double dDZ
                            )
{
    // Create the volume
    bool bResult = CDVolume::create(XSize, YSize, ZSize, VOLUME_MARGIN);

    // Initialize the volume parameters
    m_dDX = dDX;
    m_dDY = dDY;
    m_dDZ = dDZ;

    return bResult;
}


bool CDensityVolume::create(tSize XSize,
                            tSize YSize,
                            tSize ZSize,
                            const CDVolume::tVoxel& Value
                            )
{
    // Create the volume
    bool bResult = CDVolume::create(XSize, YSize, ZSize, Value, VOLUME_MARGIN);

    return bResult;
}


bool CDensityVolume::create(tSize XSize,
                            tSize YSize,
                            tSize ZSize,
                            const CDVolume::tVoxel& Value,
                            double dDX,
                            double dDY,
                            double dDZ
                            )
{
    // Create the volume
    bool bResult = CDVolume::create(XSize, YSize, ZSize, Value, VOLUME_MARGIN);

    // Initialize the volume parameters
    m_dDX = dDX;
    m_dDY = dDY;
    m_dDZ = dDZ;

    return bResult;
}


bool CDensityVolume::create(const CDensityVolume& Volume)
{
    // Copy the volume data
    CDVolume::create(Volume);

    // Volume properties
    m_dDX = Volume.m_dDX;
    m_dDY = Volume.m_dDY;
    m_dDZ = Volume.m_dDZ;

    return true;
}


bool CDensityVolume::create(const CDensityVolume& Volume, EMakeRef MakeRef)
{
    // Make reference to the volume data
    CDVolume::create(Volume, REFERENCE);

    // Volume properties
    m_dDX = Volume.m_dDX;
    m_dDY = Volume.m_dDY;
    m_dDZ = Volume.m_dDZ;

    return true;
}


CDensityVolume& CDensityVolume::operator =(const CDensityVolume& Volume)
{
    *((CDVolume *)this) = (const CDVolume&)Volume;

    m_dDX = Volume.m_dDX;
    m_dDY = Volume.m_dDY;
    m_dDZ = Volume.m_dDZ;

    return *this;
}


bool CDensityVolume::getSliceXY(tSize z, CSlice& Plane)
{
    // Cut data from CVolume
    if (CDVolume::getPlaneXY(z, Plane))
    {
        Plane.setIndex(z);
        Plane.setPosition(z * m_dDZ);
        return true;
    }

    return false;
}


bool CDensityVolume::getSliceXZ(tSize y, CSlice& Plane)
{
    // Cut data from CVolume
    if( CDVolume::getPlaneXZ(y, Plane) )
    {
        Plane.setIndex(y);
        Plane.setPosition(y * m_dDY);
        return true;
    }

    return false;
}


bool CDensityVolume::getSliceYZ(tSize x, CSlice& Plane)
{
    // Cut data from CVolume
    if( CDVolume::getPlaneYZ(x, Plane) )
    {
        Plane.setIndex(x);
        Plane.setPosition(x * m_dDX);
        return true;
    }

    return false;
}


bool CDensityVolume::getSliceXY(double dZ, CSlice& Plane)
{
    // Index calculating from given real position
    // tSize Index = tSize(dZ / m_dDZ + 0.5);
    tSize Index = tSize(dZ / m_dDZ);

    // Cut data from CVolume
    if( CDVolume::getPlaneXY(Index, Plane) )
    {
        Plane.setIndex(Index);
        Plane.setPosition(dZ);
        return true;
    }

    return false;
}


bool CDensityVolume::getSliceXZ(double dY, CSlice& Plane)
{
    // index calculating from given real position
    // tSize Index = tSize(dY / m_dDY + 0.5);
    tSize Index = tSize(dY / m_dDY);

    // Cut data from CVolume
    if( CDVolume::getPlaneXZ(Index, Plane) )
    {
        Plane.setIndex(Index);
        Plane.setPosition(dY);
        return true;
    }

    return false;
}


bool CDensityVolume::getSliceYZ(double dX, CSlice& Plane)
{
    // Index calculating from given real position
    // tSize Index = tSize(dX / m_dDX + 0.5);
    tSize Index = tSize(dX / m_dDX);

    // Cut data from CVolume
    if( CDVolume::getPlaneYZ(Index, Plane) )
    {
        Plane.setIndex(Index);
        Plane.setPosition(dX);
        return true;
    }

    return false;
}


bool CDensityVolume::getSlice(double dPosition, CSlice& Plane)
{
    tSize Index;        // Calculated index

    switch (Plane.getOrientation())
    {
        case PLANE_XY:
            // Index calculating from given real position
            Index = tSize(dPosition / m_dDZ);
    
            // Cut data from CVolume
            if( CDVolume::getPlaneXY(Index, Plane) )
            {
                Plane.setIndex(Index);
                Plane.setPosition(dPosition);
                return true;
            }
            break;
    
        case PLANE_XZ:
            // index calculating from given real position
            Index = tSize(dPosition / m_dDY);
    
            // Cut data from CVolume
            if( CDVolume::getPlaneXZ(Index, Plane) )
            {
                Plane.setIndex(Index);
                Plane.setPosition(dPosition);
                return true;
            }
            break;
    
        case PLANE_YZ:
            // Index calculating from given real position
            Index = tSize(dPosition / m_dDX);
    
            // Cut data from CVolume
            if( CDVolume::getPlaneYZ(Index, Plane) )
            {
                Plane.setIndex(Index);
                Plane.setPosition(dPosition);
                return true;
            }
            break;
    
        default:
            break;
    }

    return false;
}


void CDensityVolume::initSlice(CSlice &Slice, EPlane eOrientation)
{
    // Oritentation selection
    switch( eOrientation )
    {
        case PLANE_XY:
            // Slice data creation by volume parameters
            Slice.create(m_XSize, m_YSize, m_dDX, m_dDY);
    
            // Set other slice parameters by volume parameters
            Slice.setPosition(0.0);
            Slice.setIndex(0);
            Slice.setThickness(m_dDZ);
            Slice.setOrientation(PLANE_XY);
            break;
    
        case PLANE_XZ:
            // Slice data creation by volume parameters
            Slice.create(m_XSize, m_ZSize, m_dDX, m_dDZ);
    
            // Set other slice parameters by volume parameters
            Slice.setPosition(0.0);
            Slice.setIndex(0);
            Slice.setThickness(m_dDY);
            Slice.setOrientation(PLANE_XZ);
            break;
    
        case PLANE_YZ:
            // Slice data creation by volume parameters
            Slice.create(m_YSize, m_ZSize, m_dDY, m_dDZ);
    
            // Set other slice parameters by volume parameters
            Slice.setPosition(0.0);
            Slice.setIndex(0);
            Slice.setThickness(m_dDX);
            Slice.setOrientation(PLANE_YZ);
            break;
    
        default:
            break;
    }
}


//==============================================================================
/*
 * - Density volume serialization functions.
 */

void CDensityVolume::serialize(mds::mod::CChannelSerializer& Writer)
{
    // Serialize data of the parent class
    CDVolume::serialize(Writer);

    // Begin of data serialization block
    MDS_DE_SERIALIZE_BEGIN;

        // Extended volume info
        Writer.writeDouble(m_dDX);
        Writer.writeDouble(m_dDY);
        Writer.writeDouble(m_dDZ);
    
    // End of the block
    MDS_DE_SERIALIZE_END;
}


void CDensityVolume::deserialize(mds::mod::CChannelSerializer& Reader)
{
    // Deserialize data of the parent class
    CDVolume::deserialize(Reader);

    // Begin of data deserialization block
    MDS_DE_DESERIALIZE_BEGIN;

        // Density volume info
        Reader.readDouble(m_dDX);
        Reader.readDouble(m_dDY);
        Reader.readDouble(m_dDZ);

    // End of the block
    MDS_DE_DESERIALIZE_END;
}


} // namespace img
} // namespace mds

