////////////////////////////////////////////////////////////
// $Id: mcvertices.h 383 2007-06-21 12:17:06Z spanel $
////////////////////////////////////////////////////////////

#ifndef MCVERTICES_H
#define MCVERTICES_H

////////////////////////////////////////////////////////////
// include soubory

#include "mcentities.h"

#include <cstring>

////////////////////////////////////////////////////////////
// definice maker

////////////////////////////////////////////////////////////
// definice namespace vctl

namespace vctl
{
/**
  * Kontejner 3D vrcholu typu MCVertex.
  * Je odvozeno od obecneho kontejneru entit MCEntitieS.
  * <br><br>
  * Pokud se pouziva testovani existence vrcholu vkladanych do 
  * tohoto kontejneru, automaticky se podle jejich souradnic 
  * generuje identifikacni HASH kod pro testovani existence.
  * Po vlozeni vrcholu do kontejneru se proto nesmi menit 
  * jejich souradnice, jinak cely mechanismus selze.
  * <br><br>
  * Testovani existence vrcholu se aktivuje nastavenim 
  * SetTestExistence() na TRUE.
  */

class MCVerticeS: public MCEntitieS<MCVertex>
{
    ////////////////////////////////////////////////////////////
    // attributy tridy

private:

    bool                                test_existence;        /**< Priznak pro provadeni testovani existence vytvarenych vrcholu, defaultni stav je false. */
    MCVertex                            work_node;             /**< Pracovni, pomocny objekt vrcholu. */
    MCQueue<unsigned int, MCVertex>     exist_fronta;          /**< Fronta pro overovani existence vytvarenych vrcholu. */

    ////////////////////////////////////////////////////////////
    // macros for the class data serialisation and deserialisation, by CDataEntity class

public:

    //! Standard method getEntityName().
    MDS_ENTITY_NAME(MCVerticeS);

    //! Standard method getEntityCompression().
    MDS_ENTITY_COMPRESSION(mds::mod::CC_RAW);

    ////////////////////////////////////////////////////////////
    // funkce tridy

public:

    /** Konstruktor tridy MCVerticeS. */
    MCVerticeS()                                           { test_existence = false; };
    /** Destruktor tridy MCVerticeS. */
    ~MCVerticeS()                                          { };

    /** Writes the vertices container. */
    void serialize(mds::mod::CChannelSerializer& Writer);
    /** Reads the vertices container. */
    void deserialize(mds::mod::CChannelSerializer& Reader);

    /** Ziskani stavu priznaku pro provadeni testu existence vytvarenych vrcholu.
     *  @return priznak provadeni testu existence vytvarenych vrcholu. */
    bool GetTestExistence()                                { return test_existence; };
    /** Nastaveni stavu priznaku pro provadeni testu existence vytvarenych vrcholu.
     *  @param _value - novy stav priznaku provadeni testu existence vytvarenych vrcholu. */
    void SetTestExistence(bool _value)                     { test_existence = _value; };

    /** Vytvoreni noveho vrcholu na zaklade daneho 3D bodu.
     *  Provadi testovani existence vrcholu podle daneho bodu a podle priznaku testovani.
     *  Podle vysledku testu vraci nalezeny nebo novy vrchol.
     *  @param new_bod - ukazatel na 3D bod, podle jehoz souradnic je vytvoren novy vrchol.
     *  @return ukazatel na objekt noveho vrcholu nebo nalezeneho totozneho vrcholu. */
    MCVertex * New( MCPoint3D & new_bod );
    /** Vytvoreni noveho vrcholu na zaklade danych souradnic.
     *  Provadi testovani existence vrcholu podle danych souradnic a podle priznaku testovani.
     *  Podle vysledku testu vraci nalezeny nebo novy vrchol.
     *  @param new_x, new_y, new_z - hodnoty souradnic noveho vrcholu.
     *  @return ukazatel na objekt noveho vrcholu nebo nalezeneho totozneho vrcholu. */
    MCVertex * New( double new_x, double new_y, double new_z );
    /** Vytvoreni noveho vrcholu na zaklade danych souradnic.
     *  Provadi testovani existence vrcholu podle danych souradnic a podle priznaku testovani.
     *  Podle vysledku testu vraci nalezeny nebo novy vrchol.
     *  @param new_x, new_y, new_z - ukazatele na souradnice noveho vrcholu.
     *  @return ukazatel na objekt noveho vrcholu nebo nalezeneho totozneho vrcholu. */
    //MCVertex * New( double & new_x, double & new_y, double & new_z );
    /** Vytvoreni noveho vrcholu na zaklade pracovniho vrcholu kontejneru.
     *  Provadi testovani existence vrcholu podle danych souradnic a podle priznaku testovani.
     *  Podle vysledku testu vraci nalezeny nebo novy vrchol.
     *  @return ukazatel na objekt noveho vrcholu nebo nalezeneho totozneho vrcholu. */
    MCVertex * NewFromWorkNode();

    /** Vymazani daneho vrcholu z kontejneru, uvolneni jeho pameti.
     *  Provadi uvolneni pameti vymazaneho vrcholu.
     *  Provadi odtrideni mazane entity z fronty.
     *  @param del_ent - ukazatel na objekt mazaneho vrcholu. */
    void Erase( MCVertex * del_ent);

    /** Vytvoreni pole ukazatelu na vrcholy kontejneru podle jejich poradi.
     *  Projde seznam vrcholu s seznamu MCList kontejneru a ulozi jejich ukazatele do daneho vektoru.
     *  @param pole_node - reference na vektor ukazatelu vrcholu kontejneru. */
    void MakeIndexVector( std::vector<MCVertex *> & pole_node );
    /** Ocislovani vsech vrcholu kontejneru podle jejich poradi.
     *  Nastavi u kazdeho vrcholu kontejneru hodnotu index podle jeho poradi. */
    void Indexing();

    /** Test existence daneho vrcholu v kontejneru.
     *  @param test - ukazatel na testovany vrchol.
     *  @return vraci ukazatel na nalezeny totozny vrchol nebo NULL, pokud takovy neexistuje. */
    MCVertex * TestExistence(MCVertex * _test)              { assert(_test != NULL); return ( (MCVertex *) exist_fronta.Exist(_test->GetHashCode(), _test) ); };

    /** Znovu zatrideni vsech vrcholu kontejneru do exist fronty pro kontrolu jejich existence. */
    void RegenExistence();

    /** Zjisteni rozsahu vsech vrcholu kontejneru v prostoru.
     *  Prohleda prostor vrcholu kontejneru, nalezne a vrati jejich min a max souradnice.
     *  @param bod_min - ukazatel na 3D bod minimalniho bodu rozsahu.
     *  @param bod_max - ukazatel na 3D bod maximalniho bodu rozsahu. */
    void GetRange( MCPoint3D & bod_min, MCPoint3D & bod_max );

    /** Vymazani vsech entit kontejneru. Implementace ciste virtualni funkce. 
      * Provadi take uvolneni existencni fronty. */
    void ClearAll()         { MCEntitieS<MCVertex>::ClearAll(); exist_fronta.Clear(); };
};
}

#endif

////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////
