////////////////////////////////////////////////////////////
// $Id: mcvector3d.h 436 2007-08-07 06:59:02Z spanel $
////////////////////////////////////////////////////////////

#ifndef MCVECTOR3D_H
#define MCVECTOR3D_H

////////////////////////////////////////////////////////////
// include soubory

// MDSTk
#include <MDSTk/Base/mdsSetup.h>
#include <MDSTk/Base/mdsSmallObject.h>

#include "mccoordinate3d.h"
#include "mcpoint3d.h"

#include <cmath>

////////////////////////////////////////////////////////////
// definice maker

////////////////////////////////////////////////////////////
// definice namespace vctl

namespace vctl
{
/**
 * Zapouzdreni obsluhy a prace s vektorem ve 3D prostoru se souradnicemi typu double.
 * Odvozeno od sablony MCCoordinate3D pro 3D souradnice typu double.
 * Alokace jako maly objekt pres Loki::SmallObject.
 */

class MCVector3D: public MCCoordinate3D<double>, public mds::base::CSmallObject<>
{
    ////////////////////////////////////////////////////////////
    // attributy tridy

protected:

    ////////////////////////////////////////////////////////////
    // funkce tridy

public:

    /** Konstruktor objektu MCVector3D.
     *  Provadi nulovani souradnic. */
    MCVector3D()                                          { };
    /** Konstruktor objektu MCVector3D.
     *  Provadi inicializaci souradnic podle daneho vektoru.
     *  @param v - vektor, podle nehoz jsou inicializovany souradnice noveho vektoru. */
    MCVector3D(MCVector3D * v)                            { x = v->x; y = v->y; z = v->z; };
    /** Konstruktor objektu MCVector3D.
     *  Provadi inicializaci souradnic podle ukazatelu na dane hodnoty.
     *  @param _x - ukazatel na novou souradnici v ose X.
     *  @param _y - ukazatel na novou souradnici v ose Y.
     *  @param _z - ukazatel na novou souradnici v ose Z. */
    MCVector3D(double * _x, double * _y, double * _z)     { x = *_x; y = *_y; z = *_z; };
    /** Konstruktor objektu MCVector3D.
     *  Provadi inicializaci souradnic danych hodnot.
     *  @param _x - hodnota nove souradnice v ose X.
     *  @param _y - hodnota nove souradnice v ose Y.
     *  @param _z - hodnota nove souradnice v ose Z. */
    MCVector3D(double _x, double _y, double _z)           { x = _x; y = _y; z = _z; };
    /** Konstruktor objektu MCVector3D.
     *  Provadi inicializaci souradnic  podle ukazatelu na dva body tvorici vektor.
     *  @param p1 - ukazatel na pocatecni bod vektoru typu MCPoint3D.
     *  @param p2 - ukazatel na koncovy bod vektoru typu MCPoint3D. */
    MCVector3D(MCPoint3D * p1, MCPoint3D * p2)            { Make(p1, p2); };

    /** Destruktor objektu MCVector3D .*/
    ~MCVector3D()                                         {};

    /** Vytvoreni vektoru, vypocet souradnic, ze dvou referenci danych bodu.
     *  @param p1 - reference na pocatecni bod MCPoint3D vektoru.
     *  @param p2 - reference na koncovy bod MCPoint3D vektoru. */
    void Make(MCPoint3D & p1, MCPoint3D & p2)             { x = p2.GetX() - p1.GetX(); y = p2.GetY() - p1.GetY(); z = p2.GetZ() - p1.GetZ(); };
    /** Vytvoreni vektoru ze dvou ukazatelu danych bodu.
     *  @param p1 - ukazatel na pocatecni bod MCPoint3D vektoru.
     *  @param p2 - ukazatel na koncovy bod MCPoint3D vektoru. */
    void Make(MCPoint3D * p1, MCPoint3D * p2)             { x = p2->GetX() - p1->GetX(); y = p2->GetY() - p1->GetY(); z = p2->GetZ() - p1->GetZ(); };

    /** Operator souctu dvou vektoru.
     *  @param v - reference na pricitany vektor.
     *  @return vysledny vektor dany soucten. */
    MCVector3D operator + (MCVector3D & v)                { MCVector3D  _v( x + v.x, y + v.y , z + v.z ); return _v; };
    /** Operator rozdilu dvou vektoru.
     *  @param v - reference na odecitany vektor.
     *  @return vysledny vektor dany rozdilem. */
    MCVector3D operator - (MCVector3D & v)                { MCVector3D  _v( x - v.x, y - v.y , z - v.z ); return _v; };
    /** Operator zaporneho znamenka.
     *  Realizuje prevod na opacny, zaporny vektor.
     *  @return vysledny opacny vektor. */
    MCVector3D operator - ()                              { MCVector3D  _v(-x, -y, -z); return _v; };

    /** Operator pricteni daneho vektoru k danemu vektoru.
     *  @param v - reference na pricitany vektor. */
    void operator += (MCVector3D & v)                     { x += v.x; y += v.y; z += v.z; };
    /** Operator odecteni daneho vektoru od danemu vektoru.
     *  @param v - reference na odecitany vektor. */
    void operator -= (MCVector3D & v)                     { x -= v.x; y -= v.y; z -= v.y; };
    /** Operator nasobeni daneho vektoru danym realnym cislem.
     *  @param cislo - double hodnota, kterou je vektor nasoben. */
    void operator *= (double cislo)                       { x *= cislo; y *= cislo; z *= cislo; };

    /** Operator vektoroveho soucinu.
     *  @param v - reference na druhy vektor, se kterym se provadi vektorovy soucin.
     *  @return vysledny vektor dany vektorovym soucinem. */
    MCVector3D operator % (MCVector3D & v)                { MCVector3D  _v( y * v.z - z * v.y, z * v.x - x * v.z, x * v.y - y * v.x ); return _v; };
    /** Vektorovy soucin dvou danych vektoru.
     * Vysledek operace je ulozen do aktualniho vektoru.
     *  @param u - reference na prvni vektor, se kterym se provadi vektorovy soucin.
     *  @param v - reference na druhy vektor, se kterym se provadi vektorovy soucin. */
    void VectorProduct(MCVector3D & u, MCVector3D & v)    { x = u.y * v.z - u.z * v.y; y = u.z * v.x - u.x * v.z; z = u.x * v.y - u.y * v.x; };

    /** Operator skalarniho soucinu.
     *  @param v - reference na vektor, se kterym se provadi skalarni soucin.
     *  @return double hodnota vysledku skalarniho soucinu. */
    double operator * (MCVector3D & v)                    { return ( x * v.x + y * v.y + z * v.z ); };
    /** Staticka funkce skalarniho soucinu dvou danych vektoru.
     *  @param u - reference na prvni vektor, se kterym se provadi skalarni soucin.
     *  @param v - reference na druhy vektor, se kterym se provadi skalarni soucin.
     *  @return double hodnota vysledku skalarniho soucinu. */
    static double DotProduct(MCVector3D & u, MCVector3D & v)  { return ( u.x * v.x + u.y * v.y + u.z * v.z ); };

    /** Nasobeni souradnic vektoru danym realnym cislem.
     *  @param cislo - double hodnota, kterou je vektor nasoben.
     *  @return vysledny vektor dany vynasobenim danym cislem. */
    MCVector3D operator *(double cislo)                   { MCVector3D  _v( x*cislo, y*cislo, z*cislo ); return _v; };
    /** Deleni souradnic vektoru danym realnym cislem.
     *  @param cislo - double hodnota, kterou je vektor delen.
     *  @return vysledny vektor dany delenim danym cislem. */
    MCVector3D operator / (double cislo)                  { MCVector3D  _v( x/cislo, y/cislo, z/cislo ); return _v; };

    /** Vypocet delky vektoru.
     *  @return double hodnota delky vektoru. */
    double Length()                                       { return sqrt(x*x + y*y + z*z); };
    /** Vypocet druhe mocniny delky vektoru.
     *  @return double hodnota druhe mocniny delky vektoru. */
    double SquareLength()                                 { return (x*x + y*y + z*z); };

    /** Normalizace vektoru.
     *  Prevede tento vektor na vektor jednotkove delky.
     *  @return vysledny vektor, tento vektor po normalizaci. */
    MCVector3D & Normalization()
    {
        double d = Length();          // delka vektoru

        // kontrola nulove delky
        if (d == 0)
        { x = 0; y = 0; z = 0; }    // nulovy vektor
        else
        { x /= d; y /= d; z /= d; } // jednotkovy vektor

        return *this;                 // vraceni aktualniho jednotkoveho vektoru
    };

    /** Staticke funkce pro vypocet normaloveho vektoru z roviny podle danych tri bodu.
     *  @param p0 - ukazatel na prvni 3D bod roviny.
     *  @param p1 - ukazatel na druhy 3D bod roviny.
     *  @param p2 - ukazatel na treti 3D bod roviny.
     *  @param normala - reference na vysledny vektor vypoctene normaly.
     *  @return reference na vysledny vektor vypoctene normaly, shodne s parametrem normala. */
    static MCVector3D & MakeNormal( MCPoint3D * p0, MCPoint3D * p1, MCPoint3D * p2, MCVector3D & normala )
    {
        MCVector3D      v01(p0, p1), v02(p0, p2);     // pomocne vektory

        // vypocet normaly, vektorovy soucin
        normala.VectorProduct(v01, v02);
        // normalizace normaly
        normala.Normalization();

        // vraceni reference vysledneho vektoru
        return normala;
    };
};
}
#endif

////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////
