////////////////////////////////////////////////////////////
// $Id: mctetras.h 383 2007-06-21 12:17:06Z spanel $
////////////////////////////////////////////////////////////

#ifndef MCTETRAS_H
#define MCTETRAS_H

////////////////////////////////////////////////////////////
// include soubory

#include "mcvertices.h"
#include "mcedges.h"
#include "mctris.h"

////////////////////////////////////////////////////////////
// definice maker

////////////////////////////////////////////////////////////
// definice namespace vctl

namespace vctl
{
/**
  * Kontejner 3D tetrahedru typu MCTetra.
  * Je odvozeno od obecneho kontejneru entit MCEntitieS.
  */

class MCTetraS: public MCEntitieS<MCTetra>
{
    ////////////////////////////////////////////////////////////
    // attributy tridy

private:

    MCVerticeS                * vertices;                     /**< Ukazatel na kontejner vrcholu site, na kterych je sit Tetra postavena. */
    MCEdgeS                   * edges;                        /**< Ukazatel na paralelni kontejner hran MCEdgeS, ktery muze zdilet se siti Tetra vrcholy. */
    MCTriS                    tris;                           /**< Paralelni kontejner Tri MCTriS, ktery muze zdilet se siti Tetra vrcholy. */
    unsigned int              actual_value_flag;              /**< Actual value flag for signing the tetrahedrons mesh. */

    ////////////////////////////////////////////////////////////
    // macros for the class data serialisation and deserialisation, by CDataEntity class

public:

    //! Standard method getEntityName().
    MDS_ENTITY_NAME(MCTetraS);

    //! Standard method getEntityCompression().
    MDS_ENTITY_COMPRESSION(mds::mod::CC_RAW);

    ////////////////////////////////////////////////////////////
    // funkce tridy

public:

    /** Konstruktor tridy MCTetraS. */
    MCTetraS()                                           { vertices = tris.GetVerticeS(); edges = tris.GetEdgeS(); actual_value_flag = 0; };
    /** Destruktor tridy MCTetraS. */
    ~MCTetraS()                                          { };

    /** Writes the tetrahedral mesh. */
    void serialize(mds::mod::CChannelSerializer& Writer);
    /** Reads the tetrahedral mesh. */
    void deserialize(mds::mod::CChannelSerializer& Reader);

    /** Get value of actual value flag.
      * @return value of actual value flag. */
    unsigned int GetActualValueFlag()                                           { return actual_value_flag; };
    /** Set new value of actual value flag.
      * @param _value - new value of actual value flag. */
    void SetActualValueFlag(unsigned int _value)                                { actual_value_flag = _value; };

    /** Vytvoreni noveho Tetra na zaklade danych vrcholu.
     *  Provadi testovani existence Tetra podle danych vrcholu.
     *  Podle vysledku testu vraci nalezeny nebo novy Tetra.
     *  @param _n0 - ukazatel na prvni vrchol noveho Tetra.
     *  @param _n1 - ukazatel na druhy vrchol noveho Tetra.
     *  @param _n2 - ukazatel na treti vrchol noveho Tetra.
     *  @param _n3 - ukazatel na ctvrty vrchol noveho Tetra.
     *  @return ukazatel na objekt noveho Tetra nebo nalezeneho totozneho Tetra. */
    MCTetra * New(MCVertex * _u0, MCVertex * _u1, MCVertex * _u2, MCVertex * _u3);

    /** Vymazani daneho Tetra z kontejneru.
     *  Provadi uvolneni vazeb na vrcholy a pameti vymazaneho Tetra.
     *  Provadi odtrideni mazane entity z fronty.
     *  @param del_ent - ukazatel na objekt mazaneho Tetra. */
    void Erase( MCTetra * del_ent);

    /** Odregistrovani vsech Tetra kontejneru z jejich vrcholu. */
    void DeRegistrationAll();

    /** Ziskani vsech Tetra kolem hrany dane dvema jejimi vrcholy.
     *  Hledaji se Tetra, ktere zdileji oba dane vrcholy.
     *  @param _u0 - ukazatel na prvni dany vrchol testovane hrany 
     *  @param _u1 - ukazatel na druhy dany vrchol testovane hrany. 
     *  @param tetra_pole - vektor ukazatelu na Tetra zdilejici oba dane vrcholy. */
    void GetTetraEdge( MCVertex * _u0, MCVertex * _u1, std::vector<MCTetra *> & tetra_pole );
    /** Ziskani poctu Tetra kolem hrany dane dvema jejimi vrcholy.
     *  Pocitaji se Tetra, ktere zdileji oba dane vrcholy.
     *  @param u0 - ukazatel na prvni dany vrchol testovane hrany. 
     *  @param u1 - ukazatel na druhy dany vrchol testovane hrany. 
     *  @return pocet Tetra zdilejicich oba dane vrcholy. */
    int GetTetraEdgeNumber( MCVertex * _u0, MCVertex * _u1 );

    /** Test existence hrany dane dvema vrcholy v siti Tetra kontejneru.
     *  @param _u0 - ukazatel na prvni vrchol testovane hrany.
     *  @param _u1 - ukazatel na druhy vrchol testovane hrany.
     *  @return priznak o existenci testovane hrany v siti Tetra kontejneru. */
    bool TestEdgeExistence( MCVertex * _u0, MCVertex * _u1 );
    /** Test existence Tri daneho tremi vrcholy v siti Tetra. */
    /** Test existence Tri daneho tremi vrcholy v siti Tetra kontejneru.
     *  @param _u0 - ukazatel na prvni vrchol testovaneho Tri.
     *  @param _u1 - ukazatel na druhy vrchol testovaneho Tri.
     *  @param _u2 - ukazatel na treti vrchol testovaneho Tri.
     *  @return priznak o existenci testovaneho Tri v siti Tetra kontejneru. */
    bool TestTriExistence( MCVertex * _u0, MCVertex * _u1, MCVertex * _u2 );
    /** Test existence Tetra daneho ctyrmi vrcholy v kontejneru.
     *  @param _u0 - ukazatel na prvni vrchol testovaneho Tetra.
     *  @param _u1 - ukazatel na druhy vrchol testovaneho Tetra.
     *  @param _u2 - ukazatel na treti vrchol testovaneho Tetra.
     *  @param _u3 - ukazatel na ctvrty vrchol testovaneho Tetra.
     *  @return ukazatel na nalezeny Tetra nebo NULL, pokud v kontejneru neexistuje takove Tetra. */
    MCTetra * TestExistence(MCVertex * _u0, MCVertex * _u1, MCVertex * _u2, MCVertex * _u3);

    /** Ziskani ukazatele na kontejner vrcholu, na nichz je sit Tetra postavena.
     *  @return ukazatel na kontejner vrcholu pro tento kontejner. */
    MCVerticeS * GetVerticeS()                           { return vertices; };
    /** Ziskani ukazatele na paralelni kontejner Hran, ktery je postaveny na stejnem kontejneru vrcholu.
     *  @return ukazatel na paralelni kontejner Hran pro tento kontejner. */
    MCEdgeS * GetEdgeS()                                 { return edges; };
    /** Ziskani ukazatele na paralelni kontejner Tri, ktery je postaveny na stejnem kontejneru vrcholu.
     *  @return ukazatel na paralelni kontejner Tri pro tento kontejner. */
    MCTriS * GetTriS()                                   { return &tris; };

    /** Vytvoreni hran pro vsechny tetra site. */
    void MakeAllTetrasEdges();
    /** Vytvoreni hran daneho tetra.
      * @param _tetra - ukazatel na tetra jehoz hrany budou vytvareny. */
    void MakeTetraEdges(MCTetra * _tetra);

    /** Vymazani vrcholu site tetra, ktere nemaji registrovan zadny tetra. */
    void EraseVerticesNoTetras();
    /** Vymazani hran site tetra, ktere nezdileji zadny tetra. */
    void EraseEdgesNoTetras();
};
}

#endif

////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////
