////////////////////////////////////////////////////////////
// $Id: mcqueue.h 164 2006-12-08 20:26:48Z krsek $
////////////////////////////////////////////////////////////

#ifndef MCQUEUE_H
#define MCQUEUE_H

////////////////////////////////////////////////////////////
// include soubory

//#include "mcentity.h"

#include <map>
#include <stdexcept>

////////////////////////////////////////////////////////////
// definice maker

////////////////////////////////////////////////////////////
// definice namespace vctl

namespace vctl
{
/**
  * Sablona fronty pro trideni objektu odvozenych od MCEntity.
  * Zapouzdruje STL Multimap <key, type>.
  * Parametrem sablony je typ klice mapy a vlozena hodnota, 
  * coz by mel byt objekt odvozeny od MCEntity.
  */

template <typename KEY_TYPE, typename VALUE_TYPE> class MCQueue
{
    ////////////////////////////////////////////////////////////
    // typy tridy

public:

    typedef std::multimap<KEY_TYPE, VALUE_TYPE *>                              MT_TYPE;    /**< Typ stromu fronty. */
    typedef typename std::multimap<KEY_TYPE, VALUE_TYPE *>::iterator           MT_ITER;    /**< Typ iteratoru fronty. */
    typedef typename std::multimap<KEY_TYPE, VALUE_TYPE *>::reverse_iterator   MT_RITER;   /**< Typ reverzniho iteratoru fronty. */

    ////////////////////////////////////////////////////////////
    // attributy tridy

private:

    MT_TYPE         strom;          /**< Multimapa obsluhujici frontu. */

    ////////////////////////////////////////////////////////////
    // funkce tridy

public:

    /** Konstruktor tridy MCQueue. */
    MCQueue()                                              {};
    /** Destruktor tridy MCQueue. */
    ~MCQueue()                                             {};

    /** Zatrideni dane entity s danym klicem do fronty.
     *  @param klic - hodnota klice dane entity pro zatrideni.
     *  @param  entita - ukazatel na danou entitu pro zatrideni. */
    void Sort( KEY_TYPE klic, VALUE_TYPE * entita )                 { strom.insert(std::make_pair(klic, entita)); };
    /** Odtrideni dane entity s danym klicem  z fronty.
     *  @param klic - hodnota klice dane entity pro odtrideni.
     *  @param entita - ukazatel na danou entitu pro odtrideni. */
    void UnSort( KEY_TYPE klic, VALUE_TYPE * entita )
    {
        MT_ITER          it;       // iterator mazane entity

        // test, jestli je strom prazdny
        if ( ! strom.empty())
        {
            // nalezeni iteratoru pro mazanou entitu
            if ( (it = GetIterPtr( klic, entita )) == strom.end() )
                throw std::logic_error("Chyba mazani Entity podle klice z fronty, neni ve fronte");
            // zruseni entity z fronty
            strom.erase(it);
        }
    };

    /** Ziskani entity z fronty s minimalnim klicem.
     *  @return ukazatel na nalezenou entitu s minimalnim klicem. */
    VALUE_TYPE * GetMin()
    {
        MT_ITER        it = strom.begin();       // iterator prvni polozky fronty

        // kontrola prazdne fronty
        if (it != strom.end())
        {
            // kontrola existence ukazatele na prvni polozku
            assert(it->second != NULL);
            // vraci prvni entitu fronty, s nejmensim klicem
            return (it->second);
        }

        return NULL;
    };

    /** Ziskani entity z fronty s maximalnim klicem.
     *  @return ukazatel na nalezenou entitu s maximalnim klicem. */
    VALUE_TYPE * GetMax()
    {
        MT_RITER        it = strom.rbegin();  // iterator na posledni polozku

        // kontrola rezimu trideni a prazdne fronty
        if (it != strom.rend())
        {
            // kontrola existence ukazatele na posledni polozku
            assert(it->second != NULL);
            // vraci posledni entitu fronty, s nejvetsim klicem
            return (it->second);
        }

        return NULL;
    };

    /** Ziskani iteratoru na entitu z fronty s minimalnim klicem.
     *  @return iterator na nalezenou entitu s minimalnim klicem. */
    MT_ITER GetMinIter()                                   { return (strom.begin()); };
    /** Ziskani iteratoru na entitu z fronty s maximalnim klicem.
     *  @return iterator na nalezenou entitu s maximalnim klicem. */
    MT_RITER GetMaxIter()                                  { return (strom.rbegin()); };

    /** Test daneho itaratoru, jetli je roven end() iteratoru kontejneru.
     *  @param _it - porovnavany iterator. 
     *  @return vysledek testu, true = dany iterator je na konci kontejneru. */
    KEY_TYPE TestEndIter(MT_ITER _it)                             { return (_it == strom.end()); };
    /** Test daneho itaratoru, jetli je roven rend() reverzniho iteratoru kontejneru.
     *  @param _it - porovnavany reverzni iterator. 
     *  @return vysledek testu, true = dany iterator je na konci kontejneru. */
    bool TestEndIter(MT_RITER _it)                         { return (_it == strom.rend()); };

    /** Vycisteni fronty od vsech zatridenych entit. */
    void Clear()                                           { strom.clear(); };

    /** Test existence dane entity ve fronte.
     *  @param klic - hodnota klice testovane entity.
     *  @param entita - ukazatel na testovanou entitu.
     *  @return ukazatel na existujici totoznou entitu nebo NULL pokud takova ve fronte neexistuje. */
    VALUE_TYPE * Exist( KEY_TYPE klic, VALUE_TYPE * entita )
    {
        MT_ITER    it;           // iterator na totoznou polozku

        // hledani iteratoru dane entity podle totoznosti
        if ((it = GetIterIdentity(klic, entita)) != strom.end())
        {
            // kontrola existence ukazatele na polozku
            assert(it->second != NULL);
            // vraceni ukazatele na nalezenou polozku
            return (it->second);
        }

        return NULL;
    };

private:

    /** Ziskani iteratoru na prvek ve fronte, ktery je totozny s danym prvkem.
     *  @param klic - hodnota klice testovane entity.
     *  @param entita - ukazatel na testovanou entitu.
     *  @return iterator multimapy nalezene entity nebo end() pokud takova ve fronte neexistuje. */
    MT_ITER GetIterIdentity( KEY_TYPE klic, VALUE_TYPE * entita )
    {
        // rozsah iteratoru pro dany klic
        std::pair<MT_ITER, MT_ITER>    rozsah_it;

        // ziskani rozsahu it pro dany klic
        rozsah_it = strom.equal_range(klic);

        // cyklus iteratoru pro dany klic
        for (MT_ITER it = rozsah_it.first; it != rozsah_it.second; ++it)
        {
            // kontrola existence ukazatele na polozku
            assert(it->second != NULL);
            // kontrola nalezeneho iter, jestli obsahuje totoznou entitu
            if ((it->second)->TestIdentity(entita))
                return it;      // nalezeny iter obsahuje hledanou totoznou entitu
        }

        // vraceni koncoveho iter, jako signal neuspesnosti hledani
        return (strom.end());
    };

    /** Ziskani iteratoru na prvek ve fronte, ktery ma stejny ukazatel s danym prvkem.
     *  @param klic - hodnota klice testovane entity.
     *  @param entita - ukazatel na testovanou entitu.
     *  @return iterator multimapy nalezene entity nebo end() pokud takova ve fronte neexistuje. */
    MT_ITER GetIterPtr( KEY_TYPE klic, VALUE_TYPE * entita )
    {
        // rozsah iteratoru pro dany klic
        std::pair<MT_ITER, MT_ITER>    rozsah_it;

        // ziskani rozsahu it pro dany klic
        rozsah_it = strom.equal_range(klic);

        // cyklus iteratoru pro dany klic
        for (MT_ITER it = rozsah_it.first; it != rozsah_it.second; ++it)
        {
            // kontrola nalezeneho iter, jestli obsahuje entita
            if (it->second == entita)
                return it;      // nalezeny iter obsahuje hledanou entitu
        }

        // vraceni koncoveho iter, jako signal neuspesnosti hledani
        return (strom.end());
    };
};
}

#endif

////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////
