/*! \file mcplane.h
	\brief Definiton of plane
*/

#ifndef MCPLANE_H
#define MCPLANE_H


#include "mcentity.h"
#include <MDSTk/Base/mdsSmallObject.h>



// namespace
namespace vctl
{

/*! \brief Plane defined with its equation	coefficients

	The coefficients of the equation (A*x + B*y + C*z + D = 0) need not to be normalized.
	The coefficients A, B, and C can not all equal to zero.
*/
class MCPlane: public mds::base::CSmallObject<>
{
public:
	/*! \brief Relative position of an edge and a plane
	*/
    enum MCRelativePosition
    {
        INTERSECTS,				/*!< The edge intersects the plane*/
        FIRST_VERTEX_IN_PLANE,		/*!< First endpoint of the edge lies in the plane*/
        SECOND_VERTEX_IN_PLANE,		/*!< Second endpoint of the edge lies in the plane*/
        BOTH_IN_PLANE,			/*!< Both endpoints of the edge lie in the plane*/
        OUT_OF_PLANE			/*!< The edge does not intersect the plane*/
    };

	/*! \brief Types of coordinate planes
	*/
    enum MCCoordinatePlaneType
    {
        XY_PLANE,				/*!< normal vector (0,0,1)*/
        XZ_PLANE,				/*!< normal vector (0,1,0)*/
        YZ_PLANE				/*!< normal vector (1,0,0)*/
    };

	//=====================================================================
	// constructors and destructor

	/*! \brief Default constructor.

		Makes a XY coordinate plane
	*/
    MCPlane() 
        : _A(0.0), _B(0.0), _C(1.0), _D(0.0) {};

	/*! \brief A constructor with plane equation coefficients.
	*/
    MCPlane(double A, double B, double C, double D) 
        : _A(A), _B(B), _C(C), _D(D) {NormalizeVector();};

	/*! \brief A constructor with a normal vector of the plane and a point lying in the plane.
	*/
    MCPlane(MCVertex * vertex, MCVector3D * normal) 
    {
        _A = normal->GetX(); 
        _B = normal->GetY(); 
        _C = normal->GetZ(); 
        _D = -(_A * vertex->GetX() + _B * vertex->GetY() + _C * vertex->GetZ());
        NormalizeVector();
    };
	/*! \brief A constructor with three points lying in the plane.
	*/
    MCPlane(MCVertex * v1, MCVertex * v2, MCVertex * v3) 
    {
        MCVector3D vec1(v1, v2);
        MCVector3D vec2(v1, v3);
        MCVector3D normal = vec1 % vec2;
        _A = normal.GetX(); 
        _B = normal.GetY(); 
        _C = normal.GetZ(); 
        _D = -(_A * v1->GetX() + _B * v1->GetY() + _C * v1->GetZ());
        NormalizeVector();
    };
	/*! \brief A constructor of a coordinate plane.
	*/
    MCPlane(MCCoordinatePlaneType cplane)
        : _A(0.0), _B(0.0), _C(0.0), _D(0.0)
    {
        switch(cplane)
        {
            case XY_PLANE : _C = 1.0; break;
            case XZ_PLANE : _B = 1.0; break;
            case YZ_PLANE : _A = 1.0; break;
        }
    };
	
	/*! \brief A constructor of a coordinate plane.

		\a vertex lies in the plane.
	*/
    MCPlane(MCCoordinatePlaneType cplane, MCVertex * vertex)
        : _A(0.0), _B(0.0), _C(0.0)
    {
        switch(cplane)
        {
            case XY_PLANE : _C = 1.0; _D = -vertex->GetZ(); break;
            case XZ_PLANE : _B = 1.0; _D = -vertex->GetY(); break;
            case YZ_PLANE : _A = 1.0; _D = -vertex->GetX(); break;
        }
    };

	/*! \brief A constructor of a coordinate plane.

		The plane is in the \a distance from coordinate origin.
	*/
	MCPlane(MCCoordinatePlaneType cplane, double distance)
        : _A(0.0), _B(0.0), _C(0.0), _D(distance)
    {
        switch(cplane)
        {
            case XY_PLANE : _C = 1.0; break;
            case XZ_PLANE : _B = 1.0; break;
            case YZ_PLANE : _A = 1.0; break;
        }
    };

    ~MCPlane() {};
    
    //=====================================================================
	// Set and Get
	/*@{*/
	/*! \brief Sets the coefficient */
    void SetA(double A) {_A = A;};
	void SetB(double B) {_B = B;};
	void SetC(double C) {_C = C;};
	void SetD(double D) {_D = D;};
	/*@}*/
	
	/*@{*/
	/*! \brief Returns the coefficient */
    double GetA() {return _A;};
	double GetB() {return _B;};
	double GetC() {return _C;};
	double GetD() {return _D;};
	/*@}*/

	/*! \brief Returns the normal vector of a plane*/
    MCVector3D GetNormal() {return MCVector3D(_A, _B, _C);};
	/*! \brief Moves the plane into a point*/
    void MoveTo(MCVertex * vertex) {_D = -(_A * vertex->GetX() + _B * vertex->GetY() + _C * vertex->GetZ());}
	/*! \brief Moves the plane by a \a distance*/
	void MoveBy(double distance) {_D += distance;}
    
    //=====================================================================
	   
    /*@{*/
	/*! \brief Position of an edge and the plane.
		
		\param[in] v1 Frist endpoint of an edge
		\param[in] v1 Second endpoint of an edge
		\param[out] intersection Reference to a vertex : \n
								 the intersection coordinates are set if the edge intersects the plane (returns INTERSECTS) \n
								 or the endpoint coordinates are set if one end point of the edge is in the plane (returns FIRST_VERTEX_IN_PLANE or SECOND_VERTEX_IN_PLANE) \n
								 otherwise the coordinates are not set (returns BOTH_IN_PLANE or OUT_OF_PLANE)

		\return an enum type of the relative edge and plane position
	*/
    MCRelativePosition Position(MCEdge * edge, MCVertex & intersection) {return Position(edge->GetVertex(0), edge->GetVertex(1), intersection);};
    MCRelativePosition Position(MCVertex * v1, MCVertex * v2, MCVertex & intersection);
    /*@}*/

    /*! \brief Tests identity*/
    bool TestIdentity( MCPlane * test_plane )
    { 
        return (((fabs(_A - test_plane->GetA()) < DOUBLE_MIN) 
              && (fabs(_B - test_plane->GetB()) < DOUBLE_MIN) 
              && (fabs(_C - test_plane->GetC()) < DOUBLE_MIN)
              && (fabs(_D - test_plane->GetD()) < DOUBLE_MIN)) ? true : false );
    };

	/*! \brief Substitutes a point into the plane equation
		
		When the normal vector is normalised it returns the distance of the point from the plane

		\return A result of the equation \n
				A*x + B*y + C*z + D \n
				where x, y and z are coordinates of the vertex v
	*/
    double ToPlaneEquation(MCVertex * v) {return _A * v->GetX() + _B * v->GetY() + _C * v->GetZ() + _D;};
	
	/*! \brief Normalizes the normal vector of the plane*/
    void NormalizeVector() {IsValid(); double d = sqrt(_A * _A + _B * _B + _C * _C); _A /= d; _B /= d; _C /= d; _D /= d;}
    
protected:
	/*@{*/ 
	/*! \brief Equation coefficient*/
    double _A, _B, _C, _D;
    /*@}*/ 

	/*! \brief Checks that at least one coefficient is non-zero*/
    void IsValid() {assert(_A != 0.0 || _B != 0.0 || _C != 0.0);};    
	
private:
	MCVector3D edgeVector;	
};

} // namespace
#endif
