////////////////////////////////////////////////////////////
// $Id: mcentity.h 476 2007-09-24 12:45:40Z krsek $
////////////////////////////////////////////////////////////

#ifndef MCENTITY_H
#define MCENTITY_H

////////////////////////////////////////////////////////////
// include soubory

// MDSTk
#include <MDSTk/Base/mdsSetup.h>
#include <MDSTk/Base/mdsSmallObject.h>
#include <MDSTk/Module/mdsSerializer.h>

#include "mclistnode.h"
#include "mcpoint3d.h"
#include "mcvector3d.h"

#include <memory.h>
#include <vector>
#include <stdexcept>
#include <fstream>
#include <cmath>
#include <iostream>
#include <cassert>
#include <sstream>

////////////////////////////////////////////////////////////
// definice maker

#define         DOUBLE_MAX                 1000000000             // 1 000 000 000
//#define         DOUBLE_MAX                 1000
#define         DOUBLE_MIN                 0.000000001            // 0.000 000 001
//#define         DOUBLE_MIN                 0.001

#define         SQRT_3                    1.73205080756887729353

////////////////////////////////////////////////////////////
// definice namespace vctl

namespace vctl
{
/** Pojmenovani stavu entit, enum, 32 bits. */
enum MEEntityFlags
{
    VERTEX_CORNER_FLAG = 2,
    ENTITY_MARK_FLAG = 1,
    NO_FLAG = 0
};

////////////////////////////////////////////////////////////
// odkazy na pozdeji deklarovanych tridy

class MCEdge;
class MCTri;
class MCTetra;

////////////////////////////////////////////////////////////
// hlavicka tridy MCEntity

/**
  * Sablona obecneho zakladu vektorove entity.
  * Zdruzuje obecne vlastnosti vektorovych entit.
  * Odvozeno od prvku spojoveho seznamu MCListNode.
  */

template <typename ENTITY_TYPE> class MCEntity: public MCListNode<ENTITY_TYPE>
{
    ////////////////////////////////////////////////////////////
    // attributy tridy

protected:

    int                   flag;                 /**< Priznak stavu entity, typ MEEntityFlags, 32 bitovy. */
    int                   object;               /**< Cislo objektu ke kteremu entita patri. */
    double                value;                /**< Hodnota entity, attribut. */
    void                  * value_ptr;          /**< Ukazatel na pripojenou hodnotu, typ * void, nestara se o uvolneni pameti, pred koncem nastavit na null. */

    ////////////////////////////////////////////////////////////
    // funkce tridy

public:

    /** Konstruktor tridy MCEntity. */
    MCEntity()                                                { flag = NO_FLAG; value = 0; value_ptr = NULL; object = -1; };
    /** Destruktor tridy MCEntity. */
    virtual ~MCEntity()                                       { assert(value_ptr == NULL); };

    /** Writes the Entity. */
    void SerializeEntity(mds::mod::CChannelSerializer& Writer)
    {
      // write entity flag
      Writer.writeInt(flag);

      // write entity object index
      Writer.writeInt(object);
    };

    /** Reads the Entity. */
    void DeserializeEntity(mds::mod::CChannelSerializer& Reader)
    {
      // read entity object flag
      Reader.readInt(flag);

      // read entity object index
      Reader.readInt(object);
    };

    /** Ziskani cisla objektu entity.
     *  @return aktualni cislo objektu entity. */
    int GetObject()                                           { return object; };
    /** Nastaveni noveho cisla objektu entity.
     *  @param _value - nove cislo objektu entity. */
    void SetObject(int _value)                                { object = _value; };

    /** Nastaveni hodnoty entity.
     *  @param _value - nova hodnota entity, double. */
    void SetValue( double _value )                            { value = _value; };
    /** Ziskani nastavene hodnoty entity.
     *  @return aktualni hodnota entity, double. */
    double GetValue()                                         { return value; };

    /** Nastaveni ukazatele na pripojenou hodnotu.
     *  @param _ptr - novy adresa pripojene hodnoty, void, probehne pretypovani. */
    void SetValuePtr( void *  _ptr )                          { value_ptr = _ptr; };
    /** Ziskani nastaveneho ukazatele na pripojenou hodnotu.
     *  @return aktualni adresa pripojene hodnoty, void, nutne pretypovani. */
    void * GetValuePtr()                                      { return value_ptr; };

    /** Nastaveni stavu entity. Binarni kombinace aktualniho stavu a dane masky.
     *  @param set_flag - hodnota stavu entity, ktera je binarne pridana k aktualni hodnote stavu, 32 bit. */
    void SetFlag( int set_flag )                              { flag |= set_flag; };
    /** Ziskani stavu (hodnoty stavu) entity.
     *  @return hodnota stavu entity, stavove cislo, 32 bit. */
    int GetFlag()                                             { return flag; };
    /** Prepsani aktualni stavu entity.
     *  Aktualni stav je nahrazen novym podle dane hodnoty, prirazeni nove hodnoty do flagu.
     *  @param new_flag - nova hodnota stavu entity, zcela nahrazuje puvodni hodnotu stavu, 32 bit. */
    void ReplaceFlag( int new_flag )                          { flag = new_flag; };
    /** Maskovani stavu entity. Vypnuti bitu flagu podle dane masky.
     *  @param mask_flag - hodnota masky, jeji bity nastavene na 1 urcuji bity pro vypnuti. */
    void MaskFlag( int mask_flag )                            { flag &= (0xffff - mask_flag); };
    /** Test stavu entity. Test nastaveni bitu flagu podle dane masky.
     *  @param test_flag - hodnota testovaci masky, s jejimi bity nastavene na 1 se porovnava stav entity, 32 bit.
     *  @return vysledek testovani dane masky s bity stavu entity, bool. */
    bool TestFlag( int test_flag )                            { return ((flag & test_flag) ? true : false); };

    /** Copies all entity attributes.
     *  @param  */
    void SetEntityAttributes(MCEntity<ENTITY_TYPE> *entity)   { assert(entity); flag = entity->flag; object = entity->object; value = entity->value; };

    /** Test totoznosti aktualni entity a dane entity.
     *  Ciste virtualni funkce.
     *  @param test_entity - ukazatel na entitu se kterou se provadi testovani identity s aktualni entitou.
     *  @return vysledek testovani identity, bool. */
    virtual bool TestIdentity( ENTITY_TYPE * test_entity ) = 0;

    /** Ziskani hash kodu entity.
     *  Ciste virtualni funkce. 
     *  @return hodnota HASH kodu vygenerovana pro aktualni entitu. */
    virtual unsigned int GetHashCode() = 0;

    /** Prevedeni parametru entity na text ve forme string.
     *  Ciste virtualni funkce.
     *  @return textovy retezec, na ktery je aktualni entita prevedena, do ktereho je vypsana. */
    virtual std::string ToString() = 0;

    /** Generovani hash kodu pro dane pole byte, textovy retezec.
     *  @param hash_string - ukazatel na pole byte, textovy retezec pro generovani hash kodu.
     *  @param hash_string_size - pocet byte daneho pole byte, textoveho retezce. 
     *  @return vysledna hodnota hash kodu pro dane pole byte. */
    unsigned int MakeStringHashCode(const char * hash_string, int hash_string_size)
    {
        static const unsigned int       hash_shift = 6;                           // hodnota posunu pro vypocet hash kodu
        static const unsigned int       hash_mask = ~0U << (32U - hash_shift);    // maska pro vypocet hash kodu
        unsigned int                    hash_code = 0;                            // vysledny hash kod

        assert(hash_string != NULL);

        // cyklus byte daneho stringu pro vypocet hash kodu
        for (int i = 0; i < hash_string_size; i++)
            hash_code = (hash_code & hash_mask) ^ (hash_code << hash_shift) ^ hash_string[i];

        return hash_code;
    };
};

////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////
// hlavicka tridy MCVertex

/**
  * Objektove zapouzdreni obsluhy 3D vrcholu polygonalni site.
  * Je odvozeno od MCPoint3D a MCEntity.
  */

class MCVertex: public MCEntity<MCVertex>, public MCPoint3D
{
    ////////////////////////////////////////////////////////////
    // attributy tridy

protected:

    unsigned int        index;                      /**< Cislo uzlu (index) pro ulozeni pripojenych entit do souboru. */
    unsigned int        m_hash_code;                /**< Hash kod vrcholu, vypocita se ze souradnic. */
    MCEdge              * edge_list;                /**< Ukazatel na prvni hranu zdilejici dany vrchol. */
    MCTri               * tri_list;                 /**< Ukazatel na prvni tri zdilejici dany vrchol. */
    MCTetra             * tetra_list;               /**< Ukazatel na prvni tetra zdilejici dany vrchol. */

    ////////////////////////////////////////////////////////////
    // funkce tridy

public:

    /** Konstruktor tridy MCVertex. */
    MCVertex()                                                        { index = 0; edge_list = NULL; tri_list = NULL; tetra_list = NULL; };
    /** Konstruktor tridy MCVertex.
     *  @param _point - ukazatel na 3D bod podle ktereho jsou nastaveny souradnice noveho vrcholu. */
    MCVertex(MCPoint3D & _point):MCPoint3D(_point)                    { index = 0; edge_list = NULL; tri_list = NULL; tetra_list = NULL; };
    /** Konstruktor tridy MCVertex.
     *  @param _x - reference na souradnici noveho vrcholu v ose X, double.
     *  @param _y - reference na souradnici noveho vrcholu v ose Y, double.
     *  @param _z - reference na souradnici noveho vrcholu v ose Z, double. */
    //MCVertex(double & _x, double & _y, double & _z):MCPoint3D(_x, _y, _z)   { index = 0; edge_list = NULL; tri_list = NULL; tetra_list = NULL; };
    /** Konstruktor tridy MCVertex.
     *  @param _x - souradnici noveho vrcholu v ose X, double.
     *  @param _y - souradnici noveho vrcholu v ose Y, double.
     *  @param _z - souradnici noveho vrcholu v ose Z, double. */
    MCVertex(double _x, double _y, double _z):MCPoint3D(_x, _y, _z)   { index = 0; edge_list = NULL; tri_list = NULL; tetra_list = NULL; };

    /** Destruktor tridy MCVertex. */
    ~MCVertex()                                                       {};

    /** Writes the Vertex data. */
    void SerializeEntity(mds::mod::CChannelSerializer& Writer);
    /** Reads the Vertex data. */
    void DeserializeEntity(mds::mod::CChannelSerializer& Reader);

    /** Nastaveni indexu vrcholu pro potreby ulozeni navazujicich entit.
     *  @param _index - noba hodnota indexu vrcholu. */
    void SetIndex( unsigned int _index )                              { index = _index; };
    /** Ziskani indexu vrcholu.
     *  @return aktualni hodnota indexu vrcholu. */
    unsigned int GetIndex()                                           { return index; };

    /** Registrace dane hrany zavesene na vrchol.
     *  @param _edge - ukazatel na hranu k registraci. */
    void SetRegisteredEdge(MCEdge * _edge)                            { edge_list = _edge; };
    /** Ziskani ukazatele na registrovanou hranu.
     *  @return ukazatel na prvni hranu registrovanou u vrcholu. */
    MCEdge * GetRegisteredEdge()                                      { return edge_list; };

    /** Registrace daneho tri zaveseneho na vrchol.
     *  @param _tri - ukazatel na tri k registraci. */
    void SetRegisteredTri(MCTri * _tri)                               { tri_list = _tri; };
    /** Ziskani ukazatele na registrovany tri.
     *  @return ukazatel na prvni tri registrovany u vrcholu. */
    MCTri * GetRegisteredTri()                                        { return tri_list; };

    /** Registrace daneho tetra zaveseneho na vrchol.
     *  @param _tetra - ukazatel na tetra k registraci. */
    void SetRegisteredTetra(MCTetra * _tetra)                         { tetra_list = _tetra; };
    /** Ziskani ukazatele na registrovany tetra.
     *  @return ukazatel na prvni tetra registrovany u vrcholu. */
    MCTetra * GetRegisteredTetra()                                    { return tetra_list; };

    /** Ziskani seznamu hran zavesenych na vrchol.
     *  @param _edge_list - reference na vektor ukazatelu na hrany zavesene na vrcholu.  */
    void GetRegisteredEdgeList(std::vector<MCEdge *> & _edge_list);
    /** Ziskani seznamu tri zavesenych na vrchol.
     *  @param _tri_list - reference na vektor ukazatelu na tri zavesene na vrcholu.  */
    void GetRegisteredTriList(std::vector<MCTri *> & _tri_list);
    /** Ziskani seznamu tetra zavesenych na vrchol.
     *  @param _tetra_list - reference na vektor ukazatelu na tetra zavesene na vrcholu.  */
    void GetRegisteredTetraList(std::vector<MCTetra *> & _tetra_list);

    /** Test totoznosti daneho vrcholu s aktualnim.
     *  Definuje ciste vistualni funkci z MCEntity.
     *  @param _test - ukazatel na porovnavany vrchol.
     *  @return vysledek porovnani, true = totozny, false = neni totozny. */
    bool TestIdentity( MCVertex * _test )
    { assert(_test); 
        #ifndef NO_VERTEX_IDENTITY
          return ( ( (fabs(x - _test->GetX()) < DOUBLE_MIN) && (fabs(y - _test->GetY()) < DOUBLE_MIN) && (fabs(z - _test->GetZ()) < DOUBLE_MIN) ) ? true : false );
        #else
          return false;
        #endif
    };

    /** Ziskani hash kodu vrcholu vypocitneho podle jeho souradnic a v nem ulozeneho.
     *  Definuje ciste virtualni funkci MCEntity.
     *  @return hodnota hash kodu vrcholu. */
    unsigned int GetHashCode()                                        { return m_hash_code; };
    /** Nastaveni hash kodu vrcholu podle daneho vrcholu.
      * Definuje ciste virtualni funkci MCEntity. */
    void SetHashCode(MCVertex & _vertex)                      { m_hash_code = _vertex.m_hash_code; };

    /** Vypocet hash kodu vrcholu podle jeho souradnic.
     *  Ziskana hodnota hash kodu vrcholu se ulozi do jeho atributu m_hash_code.
     *  @return hodnota hash kodu vrcholu. */
    void MakeHashCode()
    {
        double            double_array[3];                 // pole souradnic zpracovavaneho vrcholu

        // zaokrouhleni souradnic vrcholu a ulozeni vysledku do pracovniho pole pro vypocet hash kodu
        double_array[0] = ( floor( x * DOUBLE_MAX ) );
        double_array[1] = ( floor( y * DOUBLE_MAX ) );
        double_array[2] = ( floor( z * DOUBLE_MAX ) );

        // vypocet hash kodu a vraceni vysledku
        m_hash_code = MakeStringHashCode( ((char *) double_array), sizeof(double)*3);
    };

    /** Prevedeni parametru vrcholu na text ve forme string.
     *  Definuje ciste vistualni funkci z MCEntity.
     *  @return string obsahujici text popisu vrcholu. */
    std::string ToString()
    { std::ostringstream text_buff; text_buff << "Vertex " << this << " : " << x << " / " << y << " / " << z; return text_buff.str(); };

};

////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////
// hlavicka tridy MCEdge

/**
  * Objektove zapouzdreni obsluhy 3D hrany polygonalni site.
  * Je odvozeno od MCEntity. 
  * Hrana je definovana dvema vrcholy, jejich ukazateli.
  * Alokace jako maly objekt pres Loki::SmallObject.
  */

class MCEdge: public MCEntity<MCEdge>, public mds::base::CSmallObject<>
{
    ////////////////////////////////////////////////////////////
    // attributy tridy

protected:

    MCVertex            * u[2];                 /**< Pole ukazatelu na uzly tvorici hranu. */
    MCEdge              * s[2];                 /**< Pole ukazatelu na hrany zdilejici dany uzel. */

    ////////////////////////////////////////////////////////////
    // funkce tridy

public:

    /** Konstruktor tridy MCEdge. */
    MCEdge()                                               { u[0] = NULL; u[1] = NULL; s[0] = NULL; s[1] = NULL; };
    /** Konstruktor tridy MCEdge.
     *  @param _hrana - ukazatel na hranu, podle ktere jsou nastaveny vrcholy nove hrany. */
    MCEdge(MCEdge * _hrana)                                { assert(_hrana != NULL); _hrana->GetVerticeS(u); s[0] = NULL; s[1] = NULL; };
    /** Konstruktor tridy MCEdge.
     *  @param _u0 - ukazatel na prvni vrchol nove hrany.
     *  @param _u1 - ukazatel na druhy vrchol nove hrany. */
    MCEdge(MCVertex * _u0, MCVertex * _u1)                 { assert((_u0 != NULL) && (_u1 != NULL)); u[0] = _u0; u[1] = _u1;  s[0] = NULL; s[1] = NULL; };
    /** Konstruktor tridy MCEdge.
     *  @param _u - pole ukazatelu na vrcholy nove harny. */
    MCEdge(MCVertex * _u[2])                               { assert((_u[0] != NULL) && (_u[1] != NULL)); u[0] = _u[0]; u[1] = _u[1];  s[0] = NULL; s[1] = NULL; };

    /** Destruktor tridy MCEdge. */
    ~MCEdge()                                              {};

    /** Writes the Edge data. */
    void SerializeEntity(mds::mod::CChannelSerializer& Writer);
    /** Reads the Edge data. */
    void DeserializeEntity(mds::mod::CChannelSerializer& Reader, std::vector<MCVertex *> & index_array);

    /** Nastaveni ukazatelu na vrchol hrany podle daneho indexu a daneho ukazatele.
     *  @param index - index nastavovaneho vrcholu.
     *  @param new_uzel - novy ukazatel na prislusny vrchol. */
    void SetVertex( int index, MCVertex * new_uzel )       { assert((index >= 0) && (index < 2) && (new_uzel != NULL)); u[index] = new_uzel; };
    /** Nastaveni ukazatelu na vrcholy hrany podle danych ukazatelu na vrcholy.
     *  @param _u0 - novy ukazatel na prvni vrchol hrany.
     *  @param _u1 - novy ukazatel na druhy vrchol hrany. */
    void SetVerticeS( MCVertex * _u0, MCVertex * _u1 )     { assert((_u0 != NULL) && (_u1 != NULL)); u[0] = _u0; u[1] = _u1; };
    /** Nastaveni ukazatelu na vrcholy hrany podle hodnot v danem poli ukazatelu.
     *  @param _u - pole novych ukazatelu na vrcholy hrany. */
    void SetVerticeS( MCVertex * _u[2] )                   { assert((_u[0] != NULL) && (_u[1] != NULL)); u[0] = _u[0]; u[1] = _u[1]; };

    /** Ziskani ukazatele na vrchol hrany podle daneho indexu.
     *  @param index - index ziskaneho vrcholu.
     *  @return ukazatel na ziskany vrchol. */
    MCVertex * GetVertex( int index )                      { assert((index >= 0) && (index < 2)); return u[index]; };
    /** Ziskani ukazatelu na vrcholy hrany.
     *  @param _u0 - ukazatel na ukazatel na prvni vrchol hrany.
     *  @param _u1 - ukazatel na ukazatel na druhy vrchol hrany. */
    void GetVerticeS( MCVertex ** _u0, MCVertex ** _u1 )   { assert((_u0 != NULL) && (_u1 != NULL)); *_u0 = u[0]; *_u1 = u[1]; };
    /** Ziskani ukazatelu na vrcholy hrany
     *  @param _u - pole ukazatelu na vrcholu hrany. */
    void GetVerticeS( MCVertex * _u[2] )                   { _u[0] = u[0]; _u[1] = u[1]; };

    /** Vymena daneho uzlu za novy uzel dane hodnoty.
     *  @param old_uzel - puvodni ukazatel na vrchol.
     *  @param new_uzel - novy ukazatel na vrchol. */
    void ChangeVertex(MCVertex * old_uzel, MCVertex * new_uzel)
    { int old_index = IsVertex(old_uzel); assert(old_index != -1); assert(new_uzel != NULL); u[old_index] = new_uzel; };

    /** Test daneho vrcholu jestli je obsazen hranou s vracenim jeho indexu v ramci hrany.
     *  @param test_uzel - ukazatel na testovany vrchol.
     *  @return index daneho vcholu v hrane s ohledem na orientaci, pokud je nalezen, -1 pokud neni nalezen. */
    int IsVertex( MCVertex * test_uzel )                   { assert(test_uzel != NULL); return ( (test_uzel == u[0]) ? 0 : ((test_uzel == u[1]) ? 1 : -1) ); };
    /** Test daneho vrcholu jestli je obsazen hranou s vracenim logickeho stavu true/false obsazeni.
     *  @param test_uzel - ukazatel na testovany vrchol.
     *  @return vysledek hledani uzlu v hrane, true = hrana obsahuje dany vrchol, false = neobsahuje. */
    bool IsVertexBool( MCVertex * test_uzel )              { assert(test_uzel != NULL); return ( ((test_uzel == u[0]) || (test_uzel == u[1])) ? true : false); };

    /** Ziskani zbyvajiciho uzlu vuci danemu uzlu.
     *  @param _uzel - ukazatel na dany vrchol hrany.
     *  @return ukazatel na zbyly vrchol hrany proti danemu. */
    MCVertex * GetRestVertex( MCVertex * _uzel )           { assert(_uzel != NULL); return( (u[0] == _uzel) ? u[1] : ((u[1] == _uzel) ? u[0] : NULL) ); };

    /** Ziskani geometrickeho stredu hrany.
     *  @param _stred - ukazatel na 3D bod ziskaneho stredu hrany. */
    void GetCenter( MCPoint3D & _stred )                   { _stred.SetXYZ( (u[0]->GetX()+u[1]->GetX())*0.5, (u[0]->GetY()+u[1]->GetY())*0.5, (u[0]->GetZ()+u[1]->GetZ())*0.5 ); };
    /** Ziskani delky hrany.
     *  @return vypoctena delka hrany. */
    double GetLength()                                     { return u[0]->Distance(*u[1]); };
    /** Ziskani bodu na hrane v danem pomeru mezi jejimy vrcholy. */
    void GetEdgePoint(MCPoint3D & bod_h, double pomer)
    {
        bod_h.SetX(u[0]->GetX() + pomer*(u[1]->GetX() - u[0]->GetX()));
        bod_h.SetY(u[0]->GetY() + pomer*(u[1]->GetY() - u[0]->GetY()));
        bod_h.SetZ(u[0]->GetZ() + pomer*(u[1]->GetZ() - u[0]->GetZ()));
    };

    /** Registrace aktualni hrany pro jeji vrcholy. */
    void Registration()                                    { s[0] = u[0]->GetRegisteredEdge(); s[1] = u[1]->GetRegisteredEdge(); u[0]->SetRegisteredEdge(this); u[1]->SetRegisteredEdge(this); };
    /** Uvolneni registrovanych vazeb hrany u jejich vrcholu. */
    void DeRegistration();

    /** @todo Odstraneni pretypovani z TestIdentity, souvisi s predelanim MCQueue na typ polozky fronty krome typu klice. */
    /** Test totoznosti dane hrany s aktualni. Definuje ciste vistualni funkci MCEntity. */
    bool TestIdentity( MCEdge * test_edge )
    { assert(test_edge != NULL); return( (IsVertexBool(test_edge->GetVertex(0)) && (IsVertexBool(test_edge->GetVertex(1)))) ? true : false ); };

    /** @todo Doplnit implementaci generovani hash kodu. */
    /** Ziskani hash kodu hrany.
     *  Definuje ciste vistualni funkci z MCEntity.
     *  @return hodnota hash kodu vrcholu. */
    unsigned int GetHashCode()                             { return 0; };

    /** Prevedeni parametru hrany na text ve forme string.
     *  Definuje ciste vistualni funkci z MCEntity.
     *  @return string obsahujici text popisu hrany. */
    virtual std::string ToString()
    { std::ostringstream text_buff; text_buff << "Edge " << this << " : " << u[0] << " / " << u[1]; return text_buff.str(); };

    /** Ziskani hrany, ktera je registrovana pro dany vrchol. */
    MCEdge * GetVertexEdge( MCVertex * _uzel )             { int index = IsVertex(_uzel); return((index != -1) ? s[index] : NULL); };
    /** Nastaveni dane hrany, ktera je registrovana danemu vrchol. */
    void SetVertexEdge(MCVertex * _uzel, MCEdge * _hrana)  { int index = IsVertex(_uzel); assert(index != -1); s[index] = _hrana; };

private:

protected:
};

////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////
// hlavicka tridy MCTri

/**
  * Objektove zapouzdreni obsluhy 3D trojuhelnika polygonalni site.
  * Je odvozeno od MCEntity. 
  * Trojuhelnik je definovan tremi vrcholy, jejich ukazateli.
  * Alokace jako maly objekt pres Loki::SmallObject.
  */

class MCTri: public MCEntity<MCTri>, public mds::base::CSmallObject<>
{

    ////////////////////////////////////////////////////////////
    // attributy tridy

protected:

    MCVertex            * u[3];                 /**< Pole ukazatelu na uzly tvorici trojuhelnik. */
    MCTri               * s[3];                 /**< Pole ukazatelu na tri zdilejici dany uzel. */

    ////////////////////////////////////////////////////////////
    // funkce tridy

public:

    /** Konstruktor tridy MCTri, bez parametru, provadi nulovani hodnot. */
    MCTri()                                                { u[0] = NULL; u[1] = NULL; u[2] = NULL; s[0] = NULL; s[1] = NULL; s[2] = NULL; };
    /** Konstruktor tridy MCTri, podle ukazatle daneho tri nastavuje vrcholy tri. */
    MCTri(MCTri * _tri)                                    { assert(_tri != NULL); u[0] = _tri->u[0]; u[1] = _tri->u[1]; u[2] = _tri->u[2]; s[0] = NULL; s[1] = NULL; s[2] = NULL; };
    /** Konstruktor tridy MCTri, podle danych ukazatelu na vrcholy nastavuje vrcholy tri. */
    MCTri(MCVertex * _u0, MCVertex * _u1, MCVertex * _u2)  { assert((_u0 != NULL) && (_u1 != NULL) && (_u2 != NULL)); u[0] = _u0; u[1] = _u1; u[2] = _u2; for (int i = 0; i < 3; i++) s[i] = NULL; };
    /** Konstruktor tridy MCTri, podle pole danych ukazatelu na vrcholy nastavuje vrcholy tri. */
    MCTri(MCVertex * _u[3])                                { assert((_u[0] != NULL) && (_u[1] != NULL) && (_u[2] != NULL)); u[0] = _u[0]; u[1] = _u[1]; u[2] = _u[2]; s[0] = NULL; s[1] = NULL; s[2] = NULL; };

    /** Destruktor tridy MCTri. */
    ~MCTri()                                               {};

    /** Ulozeni tri do daneho proudu. Navazuje na jiz nactene vrcholy. */
    void Load(std::ifstream & proud, std::vector<MCVertex *> & _pole);
    /** Nacteni tri z daneho proudu. Predpoklada, ze vrcholy jsou ocislovany, indexovany. */
    void Save(std::ofstream & proud);

    /** Writes the Triangle data. */
    void SerializeEntity(mds::mod::CChannelSerializer& Writer);
    /** Reads the Triangle data. */
    void DeserializeEntity(mds::mod::CChannelSerializer& Reader, std::vector<MCVertex *> & index_array);

    /** Nastaveni ukazatelu na vrchol tri podle daneho indexu a daneho ukazatele. */
    void SetVertex( int index, MCVertex * new_uzel )       { assert((index >= 0) && (index < 3) && (new_uzel != NULL)); u[index] = new_uzel; };
    /** Nastaveni ukazatelu na vrcholy tri podle danych ukazatelu na vrcholy. */
    void SetVerticeS( MCVertex * _u0, MCVertex * _u1, MCVertex * _u2 ) { assert((_u0 != NULL) && (_u1 != NULL) && (_u2 != NULL)); u[0] = _u0; u[1] = _u1; u[2] = _u2; };
    /** Nastaveni ukazatelu na vrcholy tri podle hodnot v danem poli ukazatelu. */
    void SetVerticeS( MCVertex * _u[3] )                   { assert((_u[0] != NULL) && (_u[1] != NULL) && (_u[2] != NULL)); u[0] = _u[0]; u[1] = _u[1]; u[2] = _u[2]; };

    /** Ziskani ukazatele na vrchol tri podle daneho indexu. */
    MCVertex * GetVertex( int index )                      { assert((index >= 0) && (index < 3)); return u[index]; };
    /** Ziskani ukazatelu na vrcholy tri, vraci pres tri dane ukazatele na ukazatele vrcholu. */
    void GetVerticeS( MCVertex ** _u0, MCVertex ** _u1, MCVertex ** _u2 ) { assert((_u0 != NULL) && (_u1 != NULL) && (_u2 != NULL)); *_u0 = u[0]; *_u1 = u[1]; *_u2 = u[2]; };
    /** Ziskani ukazatelu na vrcholy tri, vraci pres pole ukazatelu na ukazatele vrcholu. */
    void GetVerticeS( MCVertex * _u[3] )                   { _u[0] = u[0]; _u[1] = u[1]; _u[2] = u[2]; };

    /** Vymena daneho uzlu za novy uzel dane hodnoty.
     *  Neprovadi zmenu registrace tri u vrcholu, 
     *  nutno pred operaci odregistrovat a potom znovu registrovat. */
    void ChangeVertex(MCVertex * old_uzel, MCVertex * new_uzel)
    { int old_index = IsVertex(old_uzel); assert(old_index != -1); assert(new_uzel != NULL); u[old_index] = new_uzel; };

    /** Test daneho vrcholu jestli je obsazen tri s vracenim jeho indexu v ramci tri. */
    int IsVertex( MCVertex * test_uzel )                   { assert(test_uzel != NULL); return ( (test_uzel == u[0]) ? 0 : ((test_uzel == u[1]) ? 1 : ((test_uzel == u[2]) ? 2 : -1)) ); };
    /** Test daneho vrcholu jestli je obsazen tri s vracenim logickeho stavu true/false obsazeni. */
    bool IsVertexBool( MCVertex * test_uzel )              { assert(test_uzel != NULL); return ( ((test_uzel == u[0]) || (test_uzel == u[1]) || (test_uzel == u[2])) ? true : false); };

    /** Ziskani zbyvajiciho uzlu tri vuci danym dvema uzlu. */
    MCVertex * GetRestVertex(MCVertex * _u0, MCVertex * _u1);
    /** Ziskani zbyvajicich dvou uzlu tri vuci danemu uzlu, zachovava poradi vracenych vrcholu podle orientace tri. */
    void GetRestEdge(MCVertex * _uzel, MCVertex ** _u0, MCVertex ** _u1);

    /** Ziskani delky a ukazatele na vrcholy maximalni hrany tri. */
    double GetMaxEdge( MCVertex ** _u0, MCVertex ** _u1 );
    /** Ziskani delky a ukazatele na entitu maximalni hrany tri.
     *  Pokud neexistuji odpovidajici hrany, vraci misto ukazatele hrany NULL. */
    double GetMaxEdge( MCEdge ** max_hrana );
    /** Ziskani indexu maximalni hrany tri, podle jeho orientace. */
    int GetMaxEdge();

    /** Ziskani delky maximalni hrany tri. */
    double GetMaxEdgeLength();
    /** Ziskani delky minimalni hrany tri. */
    double GetMinEdgeLength();

    /** Ziskani seznamu ukazatelu na hrany tri.
     *  Pokud neexistuji odpovidajici hrany, vraci misto ukazatele hrany NULL. */
    void GetEdges( MCEdge * hrany[3] );
    /** Ziskani ukazatele na hranu tri podle daneho indexu s ohledem na jeho orientaci.
     *  Pokud neexistuji odpovidajici hrany, vraci misto ukazatele hrany NULL. */
    MCEdge * GetEdge( int hrana );
    /** Ziskani ukazatelu na vrcholy uzlu hrany tri podle daneho indexu s ohledem na jeho orientaci. */
    void GetEdge(int index, MCVertex ** _u0, MCVertex ** _u1);

    /** Ziskani obvodu tri. */
    double GetCircuit();
    /** Ziskani plochy tri. */
    double GetArea();
    /** Ziskani souradnic stredu tri. */
    void GetCenter(MCPoint3D & _center);
    /** Ziskani souradnic normaly tri.
     *  Ziskany vektor je normalizovan, ma delku 1. */
    MCVector3D GetNormal();
    /** Ziskani souradnic normaly tri, nastaveni souradnic na dany ukazatel.
     *  Ziskany vektor je normalizovan, ma delku 1. */
    void GetNormal(MCVector3D & nomala);
    /** Ziskani parametru obecne rovnice roviny tri. */
    void GetPlane( double & A, double & B, double & C, double & D );
    /** Ziskani hodnoty kvality tri. */
    double GetQuality()                                    { double   plocha = GetArea(); if (plocha == 0) return 1000000; return ( ( SQRT_3 * GetMaxEdgeLength() * GetCircuit() ) / ( 12 * plocha) ); };

    /** Ziskani seznamu ukazatelu na sousedni tri. */
    void GetNeighbours( MCTri * _pole[3] );
    /** Ziskani ukazatele na sousedni tri podle daneho indexu s ohledem na orietaci tri. */
    MCTri * GetNeighbour( int _index );

    /** Prevraceni orientace tri. */
    void InverseOrientation();

    /** Test orientace tri vuci danemu tri.
     *  Vracena hodnota 1 - tri jsou vuci sobe privracene.
     *  Vracena hodnota 0 - tri jsou vuci sobe kolme.
     *  Vracena hodnota -1 - tri jsou vuci sobe odvracene. */
    int CompareTriOrientation(MCTri * test_tri);

    /** Registrace aktualniho tri pro jeho vrcholy. */
    void Registration()                                    { s[0] = u[0]->GetRegisteredTri(); s[1] = u[1]->GetRegisteredTri(); s[2] = u[2]->GetRegisteredTri(); u[0]->SetRegisteredTri(this); u[1]->SetRegisteredTri(this); u[2]->SetRegisteredTri(this); };
    /** Uvolneni registrovanych vazeb tri u jeho vrcholu. */
    void DeRegistration();

    /** Test totoznosti daneho tri s aktualnim. Definuje ciste vistualni funkci MCEntity. */
    bool TestIdentity( MCTri * test_tri )
    { assert(test_tri != NULL); return( ((test_tri->IsVertexBool(u[0])) && (test_tri->IsVertexBool(u[1])) && (test_tri->IsVertexBool(u[2]))) ? true : false); };

    /** Ziskani hash kodu tri. Definuje ciste vistualni funkci MCEntity. */
    /** @todo Doplnit implementaci generovani hash kodu. */
    unsigned int GetHashCode()                             { return 0; };

    /** Prevedeni parametru tri na text ve forme string. */
    virtual std::string ToString()
    { std::ostringstream text_buff; text_buff << "Tri " << this << " : " << u[0] << " / " << u[1] << " / " << u[2]; return text_buff.str(); };

    /** Ziskani tri, ktery je registrovan pro dany vrchol. */
    MCTri * GetVertexTri( MCVertex * _uzel )               { int index = IsVertex(_uzel); return((index != -1) ? s[index] : NULL); };
    /** Nastaveni daneho tri, ktery je registrovan danemu vrcholu. */
    void SetVertexTri(MCVertex * _uzel, MCTri * _tri)      { int index = IsVertex(_uzel); assert(index != -1); s[index] = _tri; };

private:

protected:

};

////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////
// hlavicka tridy MCTetra

/**
  * Objektove zapouzdreni obsluhy 3D tetrahedru objemove site.
  * Je odvozeno od MCEntity.
  * Tetrahedr je definovan ctyrmi vrcholy, jejich ukazateli.
  * Alokace jako maly objekt pres Loki::SmallObject.
  */

class MCTetra: public MCEntity<MCTetra>, public mds::base::CSmallObject<>
{

    ////////////////////////////////////////////////////////////
    // class attributes

protected:

    MCVertex            * u[4];                 /**< Pole ukazatelu na uzly tvorici tetrahedr. */
    MCTetra             * s[4];                 /**< Pole ukazatelu na tetrahedry zdilejici dany uzel. */
    MCTetra             * n[4];                 /**< Array of direct pointers on tetrahedrons neighbour's, by side. */
    unsigned int        value_flag;             /**< Value flag for signing the tetrahedron. */

    ////////////////////////////////////////////////////////////
    // funkce tridy

public:

    /** Konstruktor tridy MCTetra, bez parametru, provadi nulovani hodnot. */
    MCTetra()   
        { for (int i = 0; i < 4; i++) { u[i] = NULL; s[i] = NULL; n[i] = NULL; } value_flag = 0; };

    /** Konstruktor tridy MCTetra, podle ukazatle daneho tetra nastavuje vrcholy tetra. */
    MCTetra(MCTetra * _tet)                                
        { assert(_tet != NULL); _tet->GetVerticeS(u); 
          for (int i = 0; i < 4; i++) { s[i] = NULL; n[i] = NULL; } value_flag = 0; };

    /** Konstruktor tridy MCTetra, podle danych ukazatelu na vrcholy nastavuje vrcholy tetra. */
    MCTetra(MCVertex * _u0, MCVertex * _u1, MCVertex * _u2, MCVertex * _u3)
        { SetVerticeS(_u0, _u1, _u2, _u3);
          for (int i = 0; i < 4; i++) { s[i] = NULL; n[i] = NULL; } value_flag = 0; };

    /** Konstruktor tridy MCTetra, podle pole danych ukazatelu na vrcholy nastavuje vrcholy tetra. */
    MCTetra(MCVertex * _u[4])                              
        { for (int i = 0; i < 4; i++) { assert(_u[i] != NULL); u[i] = _u[i]; s[i] = NULL; n[i] = NULL; } value_flag = 0; };

    /** Destruktor tridy MCTetra. */
    virtual ~MCTetra()    {};

    /** Get value of value flag.
      * @return actual value of value flag. */
    unsigned int GetValueFlag()                                           { return value_flag; };
    /** Set new value of value flag.
      * @param _value - new value of value flag. */
    void SetValueFlag(unsigned int _value)                                { value_flag = _value; };

    /** Writes the Tetra data. */
    void SerializeEntity(mds::mod::CChannelSerializer& Writer);
    /** Reads the Tetra data. */
    void DeserializeEntity(mds::mod::CChannelSerializer& Reader, std::vector<MCVertex *> & index_array);

    /** Nastaveni ukazatelu na vrchol tetra podle daneho indexu a daneho ukazatele. */
    void SetVertex( int index, MCVertex * new_uzel )       { assert((index >= 0) && (index < 4) && (new_uzel != NULL)); u[index] = new_uzel; };
    /** Nastaveni ukazatelu na vrcholy tetra podle danych ukazatelu na vrcholy. */
    void SetVerticeS( MCVertex * _u0, MCVertex * _u1, MCVertex * _u2, MCVertex * _u3 )
    { assert((_u0 != NULL) && (_u1 != NULL) && (_u2 != NULL) && (_u3 != NULL)); u[0] = _u0; u[1] = _u1; u[2] = _u2; u[3] = _u3; };
    /** Nastaveni ukazatelu na vrcholy tetra podle hodnot v danem poli ukazatelu. */
    void SetVerticeS( MCVertex * _u[4] )
    { assert((_u[0] != NULL) && (_u[1] != NULL) && (_u[2] != NULL) && (_u[3] != NULL)); u[0] = _u[0]; u[1] = _u[1]; u[2] = _u[2]; u[3] = _u[3]; };

    /** Ziskani ukazatele na vrchol tetra podle daneho indexu. */
    MCVertex * GetVertex( int index )                      { assert((index >= 0) && (index < 4)); return u[index]; };
    /** Ziskani ukazatelu na vrcholy tetra, vraci pres tri dane ukazatele na ukazatele vrcholu. */
    void GetVerticeS( MCVertex ** _u0, MCVertex ** _u1, MCVertex ** _u2, MCVertex ** _u3 )
    { assert((_u0 != NULL) && (_u1 != NULL) && (_u2 != NULL) && (_u3 != NULL)); *_u0 = u[0]; *_u1 = u[1]; *_u2 = u[2]; *_u3 = u[3]; };
    /** Ziskani ukazatelu na vrcholy tetra, vraci pres pole ukazatelu na ukazatele vrcholu. */
    void GetVerticeS( MCVertex * _u[4] )                   { _u[0] = u[0]; _u[1] = u[1]; _u[2] = u[2]; _u[3] = u[3]; };

    /** Ziskani zbyvajiciho uzlu tetra vuci danym uzlum. */
    MCVertex * GetRestVertex(MCVertex * _u0, MCVertex * _u1, MCVertex * _u2);
    /** Ziskani ukazatele na vrchol uzlu na proti zadane stene tetra */
    MCVertex * GetRestVertex( int index );

    /** Vymena daneho uzlu za novy uzel dane hodnoty. */
    void ChangeVertex(MCVertex * old_uzel, MCVertex * new_uzel)
    { int old_index = IsVertex(old_uzel); assert(old_index != -1); assert(new_uzel != NULL); u[old_index] = new_uzel; };

    /** Test daneho vrcholu jestli je obsazen tetra s vracenim jeho indexu v ramci tetra. */
    int IsVertex( MCVertex * test_uzel )
    { assert(test_uzel != NULL); return ( (test_uzel == u[0]) ? 0 : ((test_uzel == u[1]) ? 1 : ((test_uzel == u[2]) ? 2: ((test_uzel == u[3]) ? 3 : -1))) ); };
    /** Test daneho vrcholu jestli je obsazen tetra s vracenim logickeho stavu true/false obsazeni. */
    bool IsVertexBool( MCVertex * test_uzel )
    { assert(test_uzel != NULL); return ( ((test_uzel == u[0]) || (test_uzel == u[1]) || (test_uzel == u[2]) || (test_uzel == u[3])) ? true : false); };

    /** Test zda je dany tri obsazen v tetra */
    bool IsTriBool( MCVertex * _u0, MCVertex * _u1, MCVertex * _u2 )    { return (IsVertexBool(_u0) && IsVertexBool(_u1) && IsVertexBool(_u2)); };
    /** Test zda je dany tri obsazen v tetra */
    bool IsTriBool( MCVertex * _u[3] ) { return IsTriBool(_u[0], _u[1], _u[2]); }

    /** Ziskani delky maximalni hrany tetra. */
    double GetMaxEdgeLength();
    /** Ziskani delky minimalni hrany tetra. */
    double GetMinEdgeLength();
    /** Ziskani maximalni hrany tetra. */
    void GetMaxEdge( MCVertex ** _u0, MCVertex ** _u1 );
    /** Ziskani minimalni hrany tetra. */
    void GetMinEdge( MCVertex ** _u0, MCVertex ** _u1 );

    /** Ziskani ukazatelu na vrcholy uzlu steny tetra podle daneho indexu s ohledem na orientaci tetra. */
    void GetTri( int index, MCVertex ** _u0, MCVertex ** _u1, MCVertex ** _u2 );
    /** Ziskani ukazatelu na vrcholy uzlu steny tetra podle daneho indexu s ohledem na orientaci tetra. */
    void GetTri( int index, MCVertex * _u[3] ) { GetTri(index, _u, _u + 1, _u + 2); }

    /** Ziskani ukazatelu na vrcholy uzlu steny tetra protilehle k danemu vrcholu. */
    void GetRestTri( int index, MCVertex ** _u0, MCVertex ** _u1, MCVertex ** _u2 );
    /** Ziskani ukazatelu na vrcholy uzlu steny tetra protilehle k danemu vrcholu. */
    void GetRestTri( int index, MCVertex * _u[3] ) { GetRestTri(index, _u, _u + 1, _u + 2); }

    /** Get tetrahedron boundary area. */
    double GetArea();
    /** Get tetrahedron volume. */
    double GetVolume();
    /** Ziskani souradnic stredu tetra. */
    void GetCenter(MCPoint3D & _center);

    /** Ziskani souradnic normaly tri tvoriciho stenu tetra.
     *  Ziskany vektor je normalizovan, ma delku 1. */
    MCVector3D GetNormal( int index );
    /** Ziskani souradnic normaly tri tvoriciho stenu tetra.
     *  Ziskany vektor neni normalizovan, nemusi mit delku 1. */
    MCVector3D GetNormal2( int index );
    /** Ziskani souradnic normaly tri tvoriciho stenu tetra.
     *  Ziskany vektor je normalizovan, ma delku 1. */
    void GetNormal( int index, MCVector3D & normal );
    /** Ziskani souradnic normaly tri tvoriciho stenu tetra.
     *  Ziskany vektor neni normalizovan! */
    void GetNormal2( int index, MCVector3D & normal );

    /** Get the tetra neighbours. 
        Finding by the tetra registrations in his vertices. */
    void GetNeighboursByVertices( MCTetra * _pole[4] );
    /** Get the tetra neighbours.
        Directly by pointers on them. */
    void GetNeighbours( MCTetra * _n[4] )    { _n[0] = n[0]; _n[1] = n[1]; _n[2] = n[2]; _n[3] = n[3]; };
    /** Get the tetra neighbours.
        Directly by pointers on them.
        @return const pointer on neighbours pointers array MCTetra * [4].*/
    const MCTetra ** GetNeighbours()         { return ((const MCTetra **) n); };
    /** Get the tetra neighbour selected by index.
        Directly by pointer on him. */
    MCTetra * GetNeighbour( int _index )     { assert((_index >= 0) && (_index < 4)); return n[_index]; };
    /** Get the tetra neighbour selected by index.
        Finding by the tetra registrations in his vertices. */
    MCTetra * GetNeighbourByVertices( int _index );

    /** Prevraceni orientace tri v zakladne tetra. */
    void InverseOrientation();

    /** Registrace aktualniho tetra pro jeho vrcholy. */
    void Registration()   { for (int i = 0; i < 4; i++)   { s[i] = u[i]->GetRegisteredTetra(); u[i]->SetRegisteredTetra(this); }    NeighboursRegistration(); };

    /** Uvolneni registrovanych vazeb tetra u jeho vrcholu. */
    void DeRegistration();

    /** Test totoznosti daneho tetra s aktualnim. Definuje ciste vistualni funkci MCEntity. */
    bool TestIdentity( MCTetra * test_tetra )
    { assert(test_tetra != NULL); return( ((test_tetra->IsVertexBool(u[0])) && (test_tetra->IsVertexBool(u[1])) && (test_tetra->IsVertexBool(u[2])) && (test_tetra->IsVertexBool(u[3]))) ? true : false); };

    /** Ziskani hash kodu tetra. Definuje ciste vistualni funkci MCEntity. */
    /** @todo Doplnit implementaci generovani hash kodu. */
    unsigned int GetHashCode()                             { return 0; };

    /** Prevedeni parametru tetra na text ve forme string. */
    virtual std::string ToString()
    { std::ostringstream text_buff; text_buff << "Tetra " << this << " : " << u[0] << " / " << u[1] << " / " << u[2] << " / " << u[3]; return text_buff.str(); };

    /** Ziskani Tetra, ktery je registrovan pro dany vrchol. */
    MCTetra * GetVertexTetra( MCVertex * _uzel )           { int index = IsVertex(_uzel); return((index != -1) ? s[index] : NULL); };
    /** Nastaveni daneho Tetra, ktery je registrovan danemu vrcholu. */
    void SetVertexTetra(MCVertex * _uzel, MCTetra * _tet)  { int index = IsVertex(_uzel); assert(index != -1); s[index] = _tet; };

private:

protected:

    /** Tetrahedron neighbours registration. 
        Actualize his neighbours and actualize also neighbours of his new find neighbours. */
    void NeighboursRegistration();
    /** Tetrahedron neighbours de-registration. Actualize neighbour of his actual neighbours. 
        The function expect, that the tetra is de-registered from his vertices by function DeRegistration() first. */
    void NeighboursDeRegistration();

    /** Tetra neighbours actualization. Only for the tetra. */
    void NeighboursActualization()     { GetNeighboursByVertices(n); };

};
}

#endif

////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////

