//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)    \n
 * Copyright (c) 2003-2005 by Michal Spanel     \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz  \n
 * File:    mdsSemaphore.h                      \n
 * Section: libSystem                           \n
 * Date:    2004/12/18                          \n
 *
 * $Id: mdsSemaphore.h 64 2006-08-11 08:45:24Z spanel $
 *
 * Description:
 * - Named semaphore.
 */

#ifndef MDS_SEMAPHORE_H
#define MDS_SEMAPHORE_H

#include <MDSTk/Base/mdsSetup.h>

// Windows version
#ifdef _WIN32
#    include <windows.h>
#    include <limits.h>
#endif // _WIN32

// Linux version
#ifdef _LINUX
#    include <unistd.h>
#    include <sys/types.h>
#    include <sys/time.h>
#    include <sys/ipc.h>
#    include <sys/sem.h>
#    include <sys/stat.h>
#    include <errno.h>
#endif //_LINUX

#include <MDSTk/Base/mdsSharedPtr.h>

#include "mdsSystem.h"

// STL
#include <string>


namespace mds
{
namespace sys
{

//==============================================================================
/*!
 * Named semaphore.
 */
class CSemaphore : public mds::CObject
{
public:
    //! Standard method getClassName().
    MDS_CLASS_NAME(CSemaphore);

    //! Smart pointer type.
    MDS_SHAREDPTR(CSemaphore);

public:
    //! Semaphore constructor.
    CSemaphore(int iValue = 1,
               const std::string *psName = NULL,
               bool *pbAlreadyExists = NULL
               );

    //! Virtual destructor.
    virtual ~CSemaphore();

    //! Locks the semaphore.
    //! - Decreases the semaphore value by one.
    void lock();

    //! Releases the semaphore.
    //! - Increases the value.
    void unlock(int iValue = 1);

protected:
#ifdef _WIN32
    //! Semaphore handle.
    HANDLE m_hSemaphore;
#endif // _WIN32

#ifdef _LINUX
    //! Semaphore used for mutual exclusion among processes.
    int m_iSemaphore;

    //! Key of the mutex.
    key_t m_SemaphoreKey;

    //! If set to true, the semaphore will be deleted by destructor.
    bool m_bDestroy;
#endif // _LINUX

private:
    //! Private copy constructor.
    CSemaphore(const CSemaphore&);

    //! Private assignment operator.
    void operator=(const CSemaphore&);
};


//==============================================================================
/*!
 * Smart pointer to a semaphore.
 */
typedef CSemaphore::tSmartPtr   CSemaphorePtr;


} // namespace sys
} // namespace mds

#endif // MDS_SEMAPHORE_H

