//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)    \n
 * Copyright (c) 2003-2005 by Michal Spanel     \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz  \n
 * File:    mdsNamedPipe.h                      \n
 * Section: libSystem                           \n
 * Date:    2004/04/06                          \n
 *
 * $Id: mdsNamedPipe.h 116 2006-10-20 08:51:07Z spanel $
 *
 * Description:
 * - IPC using named pipes.
 */

#ifndef MDS_NAMEDPIPE_H
#define MDS_NAMEDPIPE_H

#include <MDSTk/Base/mdsSetup.h>

// Windows version
#ifdef _WIN32
#    include <windows.h>
#endif

// Linux version
#ifdef _LINUX
#    include <unistd.h>
#    include <fcntl.h>
#    include <limits.h>
#    include <errno.h>
#    include <sys/types.h>
#    include <sys/stat.h>
#endif

#include <MDSTk/Base/mdsSharedPtr.h>

#include "mdsSystem.h"

// STL
#include <string>


namespace mds
{
namespace sys
{

//==============================================================================
/*
 * Global constants.
 */

//! Type of the named pipe.
enum EPipeType
{
    P_IN            = 1,        //! Input pipe
    P_OUT           = 2         //! Output pipe
};

//! Named pipe state.
enum EPipeState
{
    P_DISCONNECTED  = 1,        //! Disconnected pipe.
    P_CONNECTED     = 2,        //! Successfully connected pipe.
    P_BROKEN        = 4 + 1,    //! Broken pipe.
    P_END           = 8 + 1     //! End of pipe reached.
};

//! Wait function resolution.
const unsigned P_RESOLUTION     = 100;


//==============================================================================
/*
 * Global constants.
 * - Windows version.
 */

#ifdef _WIN32

//! Default prefix of any pipe name.
const std::string P_NAME_PREFIX     = "\\\\.\\pipe\\mds::CNamedPipe::";

//! Default buffer size.
const int P_BUFFER_SIZE             = 4096;

#endif // _WIN32


//==============================================================================
/*
 * Global constants.
 * - Linux version.
 */

#ifdef _LINUX

//! Default prefix of any pipe name.
const std::string P_NAME_PREFIX     = "/tmp/mds::CNamedPipe::";

//! Default buffer size.
const int P_BUFFER_SIZE             = (int)PIPE_BUF;

#endif // _LINUX


//==============================================================================
/*!
 * Named pipe.
 */
class CNamedPipe : public mds::CObject
{
public:
    //! Standard method getClassName().
    MDS_CLASS_NAME(CNamedPipe);

    //! Smart pointer type.
    MDS_SHAREDPTR(CNamedPipe);

public:
    //! Constructor.
    CNamedPipe(const std::string& sName, bool bServerSide, EPipeType eType);

    //! Virtual destuctor.
    virtual ~CNamedPipe();

    //! Returns type of the named pipe.
    EPipeType type() { return m_eType; }

    //! Returns actual state of the pipe.
    EPipeState state() { return m_eState; }


    //! Connects to a specified named pipe.
    //! - Returns false on failure or if a given timeout elapses.
    bool connect(unsigned uTimeout);

    //! Returns true if the pipe is successfully connected.
    bool isConnected();

    //! Disconnects the named pipe.
    void disconnect();


    //! Waits for data.
    //! - Until the data received event is signalled.
    //! - Returns false if a timeout elapses or the channel is not input.
    bool wait(unsigned uTimeout);

    //! Reads 'iLength' bytes from the named pipe.
    //! - In case of error the number of successfully read bytes is returned.
    int read(char *pData, int iLength);

    //! Writes data to the named pipe.
    bool write(const char *pData, int iLength);

protected:
#ifdef _WIN32
    //! Windows named pipe handle.
    HANDLE m_hNamedPipe;

    //! Windows event used to wait for a connection.
    HANDLE m_hEvent;
#endif // _WIN32

#ifdef _LINUX
    //! File descriptor of a linux named pipe.
    int m_PipeFd/*, m_DummyPipeFd*/;
#endif // _LINUX

    //! Name of the pipe.
    std::string m_sName;

    //! Type of the named pipe.
    EPipeType m_eType;

    //! State of the pipe.
    EPipeState m_eState;

    //! Type of the connection.
    bool m_bServerSide;

private:
    //! Private copy constructor.
    CNamedPipe(const CNamedPipe&);

    //! Private assignment operator.
    void operator=(const CNamedPipe&);
};


//==============================================================================
/*!
 * Smart pointer to a named pipe.
 */
typedef CNamedPipe::tSmartPtr   CNamedPipePtr;


} // namespace sys
} // namespace mds

#endif // MDS_NAMEDPIPE_H

