//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)    \n
 * Copyright (c) 2003-2005 by Michal Spanel     \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz  \n
 * File:    mdsFileBrowser.h                    \n
 * Section: libSystem                           \n
 * Date:    2003/10/24                          \n
 *
 * $Id: mdsFileBrowser.h 64 2006-08-11 08:45:24Z spanel $
 *
 * Description:
 * - Class for walking through the file system.
 */

#ifndef MDS_FILEBROWSER_H
#define MDS_FILEBROWSER_H

#include <MDSTk/Base/mdsSetup.h>

// Windows version
#ifdef _WIN32
#    include <windows.h>
#endif // _WIN32

// Linux version
#ifdef _LINUX
#    include <stdlib.h>
#    include <unistd.h>
#    include <pthread.h>
#    include <limits.h>
#    include <sys/types.h>
#    include <sys/stat.h>
#    include <dirent.h>
#    include <errno.h>
#    include <regex.h>
#endif // _LINUX

#include <MDSTk/Base/mdsSharedPtr.h>

#include "mdsSystem.h"

// STL
#include <string>


namespace mds
{
namespace sys
{

//==============================================================================
/*!
 * Class for walking through the file system.
 */
class CFileBrowser : public mds::CObject
{
public:
    //! Standard method getClassName().
    MDS_CLASS_NAME(CFileBrowser);

    //! Smart pointer type.
    MDS_SHAREDPTR(CFileBrowser);

    //!    File/Directory attributes.
    struct SFileAttr
    {
        //! Filename.
        std::string m_sName;

        //! True if the structure describes a directory.
        bool m_bDirectory;
    };

public:
    //! Constructor.
    CFileBrowser(const std::string& sDirectory = "");

    //! Virtual destructor.
    virtual ~CFileBrowser();

    //! Sets the current working directory.
    bool setDirectory(const std::string& sDirectory);

    //! Returns current working directory.
    std::string getDirectory();

    //! Look up the first item in a current working directory.
    //! - Special charakters '?' and '*' can be used for mathing.
    bool findFirst(const std::string& sName, SFileAttr& Attributes);

    //! Finds next item.
    bool findNext(SFileAttr& Attributes);

private:
#ifdef _WIN32
    //! Windows handle.
    HANDLE m_hLastFound;
#endif // _WIN32

#ifdef _LINUX
    //! Directory handle.
    DIR *m_pDir;

    //! Directory entry.
    struct dirent *m_pEntry;

    //! Compiled regular expression used by regex functions.
    regex_t *m_pRegex;
#endif // _LINUX

private:
    //! Private copy constructor.
    CFileBrowser(const CFileBrowser&);

    //! Private assignment operator.
    void operator=(const CFileBrowser&);
};


//==============================================================================
/*!
 * Smart pointer to the file browser.
 */
typedef CFileBrowser::tSmartPtr     CFileBrowserPtr;


} // namespace sys
} // namespace mds

#endif // MDS_FILEBROWSER_H

