//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)    \n
 * Copyright (c) 2003-2005 by Michal Spanel     \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz  \n
 * File:    mdsCriticalSection.h                \n
 * Section: libSystem                           \n
 * Date:    2003/10/22                          \n
 *
 * $Id: mdsCriticalSection.h 116 2006-10-20 08:51:07Z spanel $
 *
 * Description:
 * - Critical section.
 */

#ifndef MDS_CRITICALSECTION_H
#define MDS_CRITICALSECTION_H

#include <MDSTk/Base/mdsSetup.h>

// Windows version
#ifdef _WIN32
#    include <windows.h>
#endif // _WIN32

// Linux version
#ifdef _LINUX
#    include <unistd.h>
#    include <pthread.h>
#endif // _LINUX

#include <MDSTk/Base/mdsSharedPtr.h>

#include "mdsSystem.h"


namespace mds
{
namespace sys
{

//==============================================================================
/*!
 * Class that encapsulates system critical section.
 */
class CCriticalSection : public mds::CObject
{
public:
    //! Standard method getClassName().
    MDS_CLASS_NAME(CCriticalSection);

    //! Smart pointer type.
    MDS_SHAREDPTR(CCriticalSection);

public:
    //! Default constructor.
    CCriticalSection();

    //! Destructor.
    virtual ~CCriticalSection();

    //! Enters critical section.
    //! - If there is a process in the section, execution is suspended.
    void enter();

    //! Tries to lock critical section. If it's already locked, returns immediately.
    //! - Returns true when section obtained. False otherwise.
    bool tryEnter();

    //! Leaves critical section.
    void leave();

private:
#ifdef _WIN32
    //! Windows critical section.
    CRITICAL_SECTION m_CritSec;
#endif // _WIN32

#ifdef _LINUX
    //! Linux critical section implemented via mutexes and conditions.
    pthread_mutex_t m_Mutex;
    pthread_mutexattr_t m_Attr;
#endif // _LINUX

private:
    //! Private copy constructor.
    CCriticalSection(const CCriticalSection&);

    //! Private assignment operator.
    void operator=(const CCriticalSection&);
};


//==============================================================================
/*!
 * Smart pointer to the critical section.
 */
typedef CCriticalSection::tSmartPtr     CCriticalSectionPtr;


} // namespace sys
} // namespace mds

#endif // MDS_CRITICALSECTION_H

