//==============================================================================`
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)    \n
 * Copyright (c) 2003-2007 by Michal Spanel     \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz  \n
 * File:    mdsSerializer.h                     \n
 * Section: libModule                           \n
 * Date:    2007/06/18                          \n
 *
 * $Id:$
 *
 * Description:
 * - Serialization of objects (data entities) over channels.
 * - Hides representation detailes from serializable objects.
 */

#ifndef MDS_CHANNELSERIALIZER_H
#define MDS_CHANNELSERIALIZER_H

#include <MDSTk/Base/mdsSetup.h>
#include <MDSTk/Base/mdsSharedPtr.h>

#include "mdsSerializable.h"
#include "mdsChannel.h"


namespace mds
{
namespace mod
{

//==============================================================================
/*!
 * Base class which encapsulates serialization of data entities over channels.
 * - Hides detailes of internal representation from serializable objects.
 */
class CChannelSerializer : public mds::CObject
{
public:
    //! Basic serializer flags.
    enum ESerializerFlags
    {
        ERROR_FLAG  = 1,    //! This error flag means that serialization has failed.
        RAW_DATA    = 2     //! Additional flag which suppresses compression and all data headers.
    };

    //! Standard method getClassName().
    MDS_CLASS_NAME(CChannelSerializer);

    //! Smart pointer type.
    MDS_SHAREDPTR(CChannelSerializer);

public:
    //! Default constructor.
    CChannelSerializer(CChannel *pChannel, unsigned int uiFlags = 0)
        : m_spChannel(pChannel)
        , m_uiFlags(uiFlags)
    {
        MDS_ASSERT(pChannel);
    }

    //! Virtual destructor.
    virtual ~CChannelSerializer() {}


    //! Returns pointer to the input/output channel.
    CChannel* getChannelPtr() { return m_spChannel; }

    //! Returns current flags.
    unsigned int getFlags() { return m_uiFlags; }
    
    //! Returns true if a given flag is set.
    bool testFlag(unsigned int uiFlag) { return (m_uiFlags & uiFlag) == uiFlag; }
   
    //! Sets a given flag.
    void setFlag(unsigned int uiFlag) { m_uiFlags |= uiFlag; }
    
    //! Removes a given flag.
    void clearFlag(unsigned int uiFlag) { m_uiFlags &= ~uiFlag; } 

    //! Returns true in case of some error.
    bool isError() { return (m_uiFlags & ERROR_FLAG) != 0; }

    //! Sets error flag.
    void setError() { m_uiFlags |= ERROR_FLAG; }


    //! Serializes a given data entity.
    //! - Returns false on failure.
    //! - Derived classes have to implement this method!
    virtual bool writeRoot(CSerializable& Object) = 0;

    //! Creates a new logical group of class members in the serialized data.
    //! - Serializer properties such as compression method may be changed.
    //! - Must be called before any writeX() method.
    //! - Sets the ERROR_FLAG on failure.
    virtual void beginWrite(const char *pcName, EChannelCompression Compression, tSize BlockSize) = 0;

    //! End of the logical group.
    virtual void endWrite() = 0;

    //! Serializes a given primitive value.
    //! - Sets the ERROR_FLAG on failure.
    //! - Derived classes have to implement this method!
    virtual void writeBool(bool bValue) = 0;
    virtual void writeInt(int iValue) = 0;
    virtual void writeUInt(unsigned int uiValue) = 0;
    virtual void writeFloat(float fValue) = 0;
    virtual void writeDouble(double dValue) = 0;
    virtual void writeSize(tSize Value) = 0;
    virtual void writeBytes(const void *pData, tSize Length) = 0;


    //! Deserializes a given data entity.
    //! - Returns false on failure.
    //! - Derived classes have to implement this method!
    virtual bool readRoot(CSerializable& Object) = 0;

    //! Reads information on logical group of class members from the serialized data.
    //! - Must be called before any readX() method.
    //! - Sets the ERROR_FLAG on failure.
    virtual void beginRead(const char *pcName) = 0;

    //! Reads end of the logical group.
    virtual void endRead() = 0;

    //! Deserializes a given primitive value.
    //! - Sets the ERROR_FLAG on failure.
    //! - Derived classes have to implement this method!
    virtual void readBool(bool& bValue) = 0;
    virtual void readInt(int& iValue) = 0;
    virtual void readUInt(unsigned int& uiValue) = 0;
    virtual void readFloat(float& fValue) = 0;
    virtual void readDouble(double& dValue) = 0;
    virtual void readSize(tSize& Value) = 0;
    virtual void readBytes(void *pData, tSize Length) = 0;

protected:
    //! Output/input channel.
    CChannelPtr m_spChannel;
    
    //! Helpful flags.
    unsigned int m_uiFlags;
};


//==============================================================================
/*!
 * Smart pointer to channel serializer.
 */
typedef CChannelSerializer::tSmartPtr   CChannelSerializerPtr;


//==============================================================================
/*
 * Global functions.
 */

//! Writes data entity into a given channel.
//! - Creates serializer with respect to the channel type.
//! - Uses method serialize() which should be overloaded by child classes.
//! - Returns false on failure.
bool write(CSerializable& Entity, CChannel& Channel, unsigned int uiFlags = 0);

//! Reads the data entity from a specified input channel.
//! - Creates serializer with respect to the channel type.
//! - Returns false on failure.
bool read(CSerializable& Entity, CChannel& Channel, unsigned int uiFlags = 0);


} // namespace mod
} // namespace mds

#endif // MDS_CHANNELSERIALIZER_H

