//==============================================================================`
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)    \n
 * Copyright (c) 2003-2007 by Michal Spanel     \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz  \n
 * File:    mdsSerializable.h                   \n
 * Section: libModule                           \n
 * Date:    2003/11/17                          \n
 *
 * $Id: mdsSerializable.h 383 2007-06-21 12:17:06Z spanel $
 *
 * Description:
 * - Introduces a concept of data entities and simple object serialization.
 */

#ifndef MDS_SERIALIZABLE_H
#define MDS_SERIALIZABLE_H

#include <MDSTk/Base/mdsSetup.h>

#include "mdsCompressor.h"

// STL
#include <cstring>
#include <string>


namespace mds
{
namespace mod
{

//==============================================================================
/*
 * Forward declarations.
 */

class CChannelSerializer;


//==============================================================================
/*
 * Several useful macros.
 */

//! Macro used to add standard method getEntityName() to a data entity.
#define MDS_ENTITY_NAME(Name) \
    const char *getEntityName() const \
    { \
        return #Name; \
    }

//! Macro used to add standard method getEntityName() to a data entity template.
//! Name containes information about concrete template parameters.
//! - Parameter 'Type' is a string literal or C/STL string.
#define MDS_ENTITY_TEMPLATE_NAME(Name, Type) \
    const char *getEntityName() const \
    { \
        static const std::string ssName = std::string(#Name) + '<' + std::string(Type) + '>'; \
        return ssName.c_str(); \
    }

//! Macro used to set compression method of a data entity.
#define MDS_ENTITY_COMPRESSION(Compression) \
    mds::mod::EChannelCompression getEntityCompression() const \
    { \
        return Compression; \
    }

//! Macro used to change block size of a data entity.
#define MDS_ENTITY_BLOCK_SIZE(Size) \
    int getEntityBlockSize() const \
    { \
        return Size; \
    }


//==============================================================================
/*!
 * Base class for all serializable objects further called data entities.
 * - Every data entity have to declare its name using the MDS_ENTITY_NAME() macro.
 * - Be aware of the maximal entity name length which is limited!
 */
class CSerializable
{
public:
    //! Maximal length of a data entity name in characters.
    enum { MAX_ENTITY_NAME_LENGTH = 32 };

    //! Current serialization interface version.
    enum { VERSION = 0x33 };

    //! Standard method getEntityName().
    MDS_ENTITY_NAME(Unknown);

    //! Standard method getEntityCompression().
    MDS_ENTITY_COMPRESSION(CC_RAW);

    //! Standard method getEntityBlockSize().
    MDS_ENTITY_BLOCK_SIZE(4096);

public:
    //! Default constructor.
    CSerializable() {}

    //! Virtual destructor.
    virtual ~CSerializable() {}

    //! Serializes the entity via a given serializer.
    //! - This method should be overloaded in each child class. Don't forget
    //!   to call serialize() method of the parent class at the begining.
    //! - Returns false on failure.
    virtual void serialize(CChannelSerializer& Writer) = 0;

    //! Deserializes the entity data via a given serializer.
    //! - This method should be overloaded in each child class. Don't forget
    //!   to call deserialize() method of the parent class at the begining.
    //! - Returns false on failure.
    virtual void deserialize(CChannelSerializer& Reader) = 0;
};


//==============================================================================
/*
 * Helper macros for implementation of data entity methods serialize()
 * and deserialize().
 */

//! Macro must precede any code which serializes group of data entity members.
//! - It can be called at the begining of the CSerializable::serialize() method.
#define MDS_DE_SERIALIZE_BEGIN      Writer.beginWrite(getEntityName(), getEntityCompression(), getEntityBlockSize())

//! Macro closes serialization code of the data entity group.
//! - Must be called at the end of the CSerializable::serialize() method.
#define MDS_DE_SERIALIZE_END        Writer.endWrite()

//! Macro must precede any code which deserializes group of data entity members.
//! - Must be called at the begining of the CSerializable::deserialize() method.
#define MDS_DE_DESERIALIZE_BEGIN    Reader.beginRead(getEntityName())

//! Macro closes deserialization code of the data entity group.
//! - Must be called at the end of the CSerializable::deserialize() method.
#define MDS_DE_DESERIALIZE_END      Reader.endRead()


} // namespace mod
} // namespace mds

#endif // MDS_SERIALIZABLE_H

