//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)    \n
 * Copyright (c) 2003-2007 by Michal Spanel     \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz  \n
 * File:    mdsSHMSerializer.hxx                \n
 * Section: libModule                           \n
 * Date:    2007/06/21                          \n
 *
 * $Id:$
 *
 * Description:
 * - Binary serialization of objects (data entities) over shared memory.
 */


//==============================================================================
/*
 * Implementation of the CSHMSerializer class.
 */

template <typename T>
inline void CSHMSerializer::writeData(mds::base::CSHMData<T>& Data)
{
    // Write size of the data block in elements T
    tSize Size = Data.getSize();
    if( !m_spActiveChannel->write((char *)&Size, (int)sizeof(tSize)) )
    {
        setError();
        return;
    }

    // Check if the pointer already addresses the shared memory
    if( Data.getSharedMemPtr() && Data.getSharedMemPtr()->getName() == m_spActiveChannel->getSharedMemPtr()->getName() )
    {
        // Compare distance from the begining
        size_t Dist1 = (char *)Data.getPtr() - (char *)Data.getSharedMemPtr()->getData();
        size_t Dist2 = (char *)m_spActiveChannel->getDataPtr() - (char *)m_spActiveChannel->getSharedMemPtr()->getData();
        if( Dist1 == Dist2 )
        {
            // Skip the data
            if( !m_spActiveChannel->skip(Data.getSize() * (int)sizeof(T)) )
            {
                setError();
                return;
            }
            
            // Logging
            MDS_LOG_NOTE("CSHMSerializer::writeData(): Data already stored in the SHM");
            return;
        }
    }

    // Write the data
    if( !m_spActiveChannel->write((char *)Data.getPtr(), Data.getSize() * (int)sizeof(T)) )
    {
        setError();
        return;
    }

    // Logging
    MDS_LOG_NOTE("CSHMSerializer::writeData(): Copying data to the shared memory");
}


template <typename T>
inline void CSHMSerializer::readData(mds::base::CSHMData<T>& Data)
{
    // Read size of the data block in elements T
    tSize Size;
    if( m_spActiveChannel->read((char *)&Size, (int)sizeof(tSize)) != (int)sizeof(tSize) )
    {
        setError();
        return;
    }

    // Initialize the data so that it will point to the shared memory 
    Data.create(m_spActiveChannel->getSharedMemPtr(),
                (T *)m_spActiveChannel->getDataPtr(),
                Size
                );

    // Skip the data
    if( !m_spActiveChannel->skip(Size * (int)sizeof(T)) )
    {
        setError();
        return;
    }

    // Logging
    MDS_LOG_NOTE("CSHMSerializer::readData(): Getting data from the shared memory");
}

