//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)    \n
 * Copyright (c) 2003-2007 by Michal Spanel     \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz  \n
 * File:    mdsSHMSerializer.h                  \n
 * Section: libModule                           \n
 * Date:    2007/06/21                          \n
 *
 * $Id:$
 *
 * Description:
 * - Binary serialization of objects (data entities) over shared memory.
 */

#ifndef MDS_SHMSERIALIZER_H
#define MDS_SHMSERIALIZER_H

#include <MDSTk/Base/mdsSHMData.h>
#include <MDSTk/Base/mdsGlobalLog.h>

#include "mdsSerializer.h"


namespace mds
{
namespace mod
{

//==============================================================================
/*!
 * Fast serialization of data entities and their members over shared memory.
 */
class CSHMSerializer : public CChannelSerializer
{
public:
    //! Additional serializer flag meaning serialization over shared memory.
    enum { SHM_FLAG = 4 };

    //! Standard method getClassName().
    MDS_CLASS_NAME(CSHMSerializer);

    //! Smart pointer type.
    //! - Declares type tSmartPtr.
    MDS_SHAREDPTR(CSHMSerializer);

public:
    //! Default constructor.
    CSHMSerializer(CSharedMemChannel *pChannel, unsigned int uiFlags = 0);
    
    //! Destructor.
    virtual ~CSHMSerializer();


    //! Serializes a given data entity.
    //! - Returns false on failure.
    virtual bool writeRoot(CSerializable& Object);

    //! Creates a new logical group of class members in the serialized data.
    //! - Serializer properties such as compression method may be changed.
    //! - Must be called before any writeX() method.
    //! - Sets the ERROR_FLAG on failure.
    virtual void beginWrite(const char *pcName, EChannelCompression Compression, tSize BlockSize);

    //! End of the logical group.
    virtual void endWrite();

    //! Serializes a given primitive value.
    //! - Sets the ERROR_FLAG on failure.
    virtual void writeBool(bool bValue);
    virtual void writeInt(int iValue);
    virtual void writeUInt(unsigned int uiValue);
    virtual void writeFloat(float fValue);
    virtual void writeDouble(double dValue);
    virtual void writeSize(tSize Value);
    virtual void writeBytes(const void *pData, tSize Length);
    
    //! Function does nothing if data are already placed in the shared memory.
    //! Otherwise, it copies data to the shared memory.
    template <typename T>
    inline void writeData(mds::base::CSHMData<T>& Data);


    //! Deserializes a given data entity.
    //! - Returns false on failure.
    virtual bool readRoot(CSerializable& Object);

    //! Reads information on logical group of class members from the serialized data.
    //! - Must be called before any readX() method.
    //! - Sets the ERROR_FLAG on failure.
    virtual void beginRead(const char *pcName);

    //! Reads end of the logical group.
    virtual void endRead();

    //! Deserializes a given primitive value.
    //! - Sets the ERROR_FLAG on failure.
    virtual void readBool(bool& bValue);
    virtual void readInt(int& iValue);
    virtual void readUInt(unsigned int& uiValue);
    virtual void readFloat(float& fValue);
    virtual void readDouble(double& dValue);
    virtual void readSize(tSize& Value);
    virtual void readBytes(void *pData, tSize Length);

    //! Initilizes internal pointer so that it will point directly
    //! to the shared memory. Does nothing if data are already placed
    //! in the same shared memory.
    template <typename T>
    inline void readData(mds::base::CSHMData<T>& Data);

protected:
    //! Smart pointer to actively used channel.
    CSharedMemChannelPtr m_spActiveChannel;
};


//==============================================================================
/*!
 * Smart pointer to SHM serializer.
 */
typedef CSHMSerializer::tSmartPtr   CCSHMSerializerPtr;


//==============================================================================
/*
 * Methods implementation.
 */

// Include file containing methods templates.
#include "mdsSHMSerializer.hxx"


} // namespace mod
} // namespace mds

#endif // MDS_SHMSERIALIZER_H

