//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)    \n
 * Copyright (c) 2003-2006 by Michal Spanel     \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz  \n
 * File:    mdsModule.h                         \n
 * Section: libModule                           \n
 * Date:    2003/11/08                          \n
 *
 * $Id: mdsModule.h 383 2007-06-21 12:17:06Z spanel $
 *
 * Description:
 * - Definition of a simple MDSTk module application.
 */

#ifndef MDS_MODULE_H
#define MDS_MODULE_H

#include <MDSTk/Base/mdsSetup.h>

#include "mdsConsole.h"
#include "mdsChannel.h"
#include "mdsSerializer.h"

// STL
#include <vector>


namespace mds
{
namespace mod
{

//==============================================================================
/*
 * Names of command line arguments.
 */

//! Namespace containing constants used by the CModule class.
namespace module
{
    
//! Command line arguments.
const std::string ARGUMENT_INPUT    = "i";
const std::string ARGUMENT_OUTPUT   = "o";
const std::string ARGUMENT_RAW      = "raw";

} // namespace module


//==============================================================================
/*!
 * Class encapsulating a single MDSTk module.
 * - Supports log files and logging.
 * - Provides command line arguments parsing.
 * - I/O channels.
 */
class CModule : public mds::mod::CConsole
{
public:
    //! Standard method getClassName().
    MDS_CLASS_NAME(CModule);

    //! Smart pointer type.
    //! - Declares type tSmartPtr.
    MDS_SHAREDPTR(CModule);

public:
    //! Default constructor.
    CModule(const std::string& sDescription);

    //! Virtual destructor.
    virtual ~CModule();


    //! Initializes module and parses a given command line.
    //! - Returns false on failure.
    bool init(int argc, char *argv[]);

    //! Writes a simple usage to the given stream.
    void printUsage();

    //! Writes a full help.
    void printFullUsage();


    //! Returns a number of input channels.
    int getNumOfInputs() const { return (int)m_InputChannels.size(); }

    //! Retreives an i-th input channel.
    //! - Returns NULL if a channel doesn't exist.
    CChannel *getInput(int i);

    //! Number of output channels.
    int getNumOfOutputs() const { return (int)m_OutputChannels.size(); }

    //! Retreives an i-th output channel.
    //! - Returns NULL if there is no i-th channel.
    CChannel *getOutput(int i);


    //! Writes a given data entity to the channel.
    //! - Returns true on success.
    bool writeOutput(CChannel *pChannel, CSerializable *pEntity);

    //! Reads a data entity from the input channel.
    //! - Returns true on success.
    bool readInput(CChannel *pChannel, CSerializable *pEntity);


    //! Virtual method called on module startup.
    //! - Expected to be overriden.
    //! - Has to return false on startup failure.
    virtual bool startup() = 0;

    //! Virtual method repeatedly called by the processing thread.
    //! - When you need to stop the execution just return false.
    virtual bool main() = 0;

    //! Called on module shutdown.
    virtual void shutdown() = 0;

    //! Called on writing a full usage statement.
    virtual void writeExtendedUsage(std::ostream& Stream) = 0;

protected:
    //! Vector of i/o channels.
    typedef std::vector<CChannelPtr> tChannels;

protected:
    //! Application input channels.
    tChannels m_InputChannels;

    //! Output channels.
    tChannels m_OutputChannels;

    //! Write raw data.
    bool m_bRawData;
};


//==============================================================================
/*!
 * Smart pointer to module.
 */
typedef CModule::tSmartPtr  CModulePtr;


} // namespace mod
} // namespace mds

#endif // MDS_MODULE_H

